// This may look like C code, but it is really -*- C++ -*-
#include "cell.h"

extern void saveSimulation( float time);
extern Cell* cell[];
extern Tissue* tissue[];

#ifdef _NEW_STREAMS
bool Cell::debug = FALSE;
int Cell::numberDefined = 0; 
float Cell::time = 0;
int Cell::width = 0;
int Cell::height = 0;
float Cell::intervalf = 1;
#endif

Representation* Cell::matrix = new Representation(); 
                                 //Initialize static member

Cell::Cell( Cell* tmpParent, bool createdByDivision)
{
  cellNumber = numberDefined++;
  exists = TRUE;
  dummy = FALSE;
  trace = FALSE;
  neighborListCurrent = FALSE;
  parent = tmpParent;
  lastDividingLine = Point(0,1);
  neighbor = (cellList *) NULL;
  if (cellNumber == 0) { // Environment cell
    dummy = TRUE;
    tissueType = 0;
    stateCounter  = tissue[ tissueType]->state;
    setCodePointers();
    delete matrix;
    matrix = new Representation( height, width);
    cout << "GridSize->(" << width << ',' << height << ")\n";
  }
  
  if (createdByDivision ) {
    tissueType = parent->tissueType;
    stateCounter = parent->stateCounter;
    code = parent->code;
    dummy = parent->dummy;
    lastDividingLine = parent->lastDividingLine;
  }
}

void Cell::divide(divideOptions dO)
{
#ifndef UNIT_CELLS
  int newCellNumber = numberDefined;
  cell[ newCellNumber] = new Cell( this);
  // set location,area
  lastDividingLine = matrix->divideCell( cellNumber, dO,
					 newCellNumber, lastDividingLine);
  area = matrix->getArea( cellNumber);
  cell[ newCellNumber]->area = matrix->getArea( newCellNumber); 
  cell[ newCellNumber]->perimeter = perimeter; //approx
  cell[ newCellNumber]->lastDividingLine = lastDividingLine;
  cell[ newCellNumber]->divideCopyVar( this);
  divideCopyVar( this); //This ordering is important since divideCopyVar
			//halves biochemical values
  neighborListCurrent = FALSE;
#endif not UNIT_CELLS
}

void
Cell::printInfo( ostream& s)
    // prints all the information about the cell
{
  s << " Details about cell " << cellNumber;
  s << "\n ---------------------\n";

  s << "Tissue type: " << tissue[tissueType]->name << '\n';
  s << "In state: " << stateCounter << '\n';
  s << "Location: "; 
  (matrix->getLocation( cellNumber) - Point(0,0)).print( s); s << '\n'; 
#ifndef UNIT_CELLS
  s << "Area = " << area << '\n';
  s << "Perimeter = " << perimeter << '\n';
#endif
}

ostream&
Cell::operator<<( ostream& s)
    // prints all the information about the cell
{
  printInfo( s);
  return s;
}

void 
Cell::saveState( ofstream& s)
{
  s << cellNumber << '\n';
  if (cellNumber == 0) { // save static members
    s << time << ' ' << height << ' ' << width << ' ' << numberDefined << ' '
      << debug << ' '  << intervalf << '\n';  
    matrix->saveState( s);
  }
  else {
    s << tissueType << ' ' << exists << ' ' << stateCounter << ' ' <<
      parent->cellNumber << ' ' <<  dummy  << ' ' <<
	trace  << '\n'; 
    lastDividingLine.saveState( s);
    saveVar(s);
  }    
}

void 
Cell::recoverState( ifstream& s)
{
  s >> cellNumber;
  if (cellNumber == 0) { // read static members
    setCodePointers();
    int idebug;
    s >> time >> height >> width >> numberDefined >> idebug >> intervalf; 
    if (idebug) debug = TRUE; else debug = FALSE;
    exists = trace = FALSE;
    parent = this;
    dummy = TRUE;
    tissueType = 0;
    stateCounter = tissue[tissueType]->state;
    matrix->recoverState( s);
  }
  else {
    int iparent, iexists, idummy, itrace;
    s >> tissueType >> iexists >> stateCounter >>
      iparent >> idummy  >>
	itrace;
    if (iexists) exists = TRUE; else exists = FALSE;
    if (idummy) dummy = TRUE; else dummy = FALSE;
    if (itrace) trace = TRUE; else trace = FALSE;
    parent = cell[ iparent];
    lastDividingLine.recoverState( s);
    recoverVar(s);
    neighbor = (cellList *) NULL;
    neighborListCurrent = FALSE;
    code = stateCode[ stateCounter];
    area = matrix->getArea( cellNumber);
  }    
}

void 
Cell::executeState() { 
// intelligent informing of the cells whose neighbors may have changed
#ifndef STABLE_NEIGHBORS
  neighborListCurrent = FALSE;
  if (neighbor != NULL) delete neighbor;
  neighbor = (cellList *)NULL;
#endif STABLE_NEIGHBORS
  (*code)();
#ifndef UNIT_CELLS
  if (!dummy) matrix->connect( cellNumber);
#endif
}

void
Cell::die()
{
  code = _CPLdoNothing;
}

#ifdef UNIT_CELLS
void Cell::reformNeighborList()
// This cells neighbors have changed reform its neighbor list, and inform its
// neighbor to throw old neighbor list, wont work if parallelized
{
  Cell* neighb;
  // reconstruct neighbor list, and inform new neighbors of change in
  // neighborhood 
  if (neighbor != NULL) delete neighbor;
  neighbor = matrix->neighborList( cellNumber, perimeter);
  neighborListCurrent = TRUE;
  while (!neighbor->emptyList()) {
    neighb = neighbor->nextItemInList()->cell;
    neighb->neighborListCurrent = FALSE;
  }
}
#endif UNIT_CELLS
