// Univariate Polynomials over modular integers.

#ifndef _CL_UNIVPOLY_MODINT_H
#define _CL_UNIVPOLY_MoDINT_H

#include "cl_ring.h"
#include "cl_univpoly.h"
#include "cl_modinteger.h"

// Normal univariate polynomials with stricter static typing:
// `cl_MI' instead of `cl_ring_element'.

class cl_heap_univpoly_modint_ring;

class cl_univpoly_modint_ring : public cl_univpoly_ring {
public:
	// Default constructor.
	cl_univpoly_modint_ring () : cl_univpoly_ring () {}
	// Copy constructor.
	cl_univpoly_modint_ring (const cl_univpoly_modint_ring&);
	// Assignment operator.
	cl_univpoly_modint_ring& operator= (const cl_univpoly_modint_ring&);
	// Automatic dereferencing.
	cl_heap_univpoly_modint_ring* operator-> () const
	{ return (cl_heap_univpoly_modint_ring*)heappointer; }
};
// Copy constructor and assignment operator.
CL_DEFINE_COPY_CONSTRUCTOR2(cl_univpoly_modint_ring,cl_univpoly_ring)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_univpoly_modint_ring,cl_univpoly_modint_ring)

class cl_UP_MI : public cl_UP {
public:
	const cl_univpoly_modint_ring& ring () const { return The(cl_univpoly_modint_ring)(_ring); }
	// Conversion.
	CL_DEFINE_CONVERTER(cl_ring_element)
	// Destructive modification.
	void set_coeff (uintL index, const cl_MI& y);
	void finalize();
	// Evaluation.
	cl_MI operator() (const cl_MI& y) const;
public:	// Ability to place an object at a given address.
	void* operator new (size_t size) { return cl_malloc_hook(size); }
	void* operator new (size_t size, cl_UP_MI* ptr) { (void)size; return ptr; }
	void operator delete (void* ptr) { cl_free_hook(ptr); }
};

class cl_heap_univpoly_modint_ring : public cl_heap_univpoly_ring {
	SUBCLASS_cl_heap_univpoly_ring()
	// High-level operations.
	void fprint (cl_ostream stream, const cl_UP_MI& x)
	{
		cl_heap_univpoly_ring::fprint(stream,x);
	}
	cl_boolean equal (const cl_UP_MI& x, const cl_UP_MI& y)
	{
		return cl_heap_univpoly_ring::equal(x,y);
	}
	cl_UP_MI zero ()
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::zero());
	}
	cl_boolean zerop (const cl_UP_MI& x)
	{
		return cl_heap_univpoly_ring::zerop(x);
	}
	cl_UP_MI plus (const cl_UP_MI& x, const cl_UP_MI& y)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::plus(x,y));
	}
	cl_UP_MI minus (const cl_UP_MI& x, const cl_UP_MI& y)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::minus(x,y));
	}
	cl_UP_MI uminus (const cl_UP_MI& x)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::uminus(x));
	}
	cl_UP_MI one ()
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::one());
	}
	cl_UP_MI canonhom (const cl_I& x)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::canonhom(x));
	}
	cl_UP_MI mul (const cl_UP_MI& x, const cl_UP_MI& y)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::mul(x,y));
	}
	cl_UP_MI square (const cl_UP_MI& x)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::square(x));
	}
	cl_UP_MI expt_pos (const cl_UP_MI& x, const cl_I& y)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::expt_pos(x,y));
	}
	sintL degree (const cl_UP_MI& x)
	{
		return cl_heap_univpoly_ring::degree(x);
	}
	cl_UP_MI monomial (const cl_MI& x, uintL e)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::monomial(x,e));
	}
	cl_MI coeff (const cl_UP_MI& x, uintL index)
	{
		return The2(cl_MI)(cl_heap_univpoly_ring::coeff(x,index));
	}
	cl_UP_MI create (sintL deg)
	{
		return The2(cl_UP_MI)(cl_heap_univpoly_ring::create(deg));
	}
	void set_coeff (cl_UP_MI& x, uintL index, const cl_MI& y)
	{
		cl_heap_univpoly_ring::set_coeff(x,index,y);
	}
	void finalize (cl_UP_MI& x)
	{
		cl_heap_univpoly_ring::finalize(x);
	}
	cl_MI eval (const cl_UP_MI& x, const cl_MI& y)
	{
		return The2(cl_MI)(cl_heap_univpoly_ring::eval(x,y));
	}
private:
	// No need for any constructors.
	cl_heap_univpoly_modint_ring ();
};

// Lookup of polynomial rings.
inline cl_univpoly_modint_ring cl_find_univpoly_ring (const cl_modint_ring& r)
{ return The(cl_univpoly_modint_ring) (cl_find_univpoly_ring((const cl_ring&)r)); }
inline cl_univpoly_modint_ring cl_find_univpoly_ring (const cl_modint_ring& r, const cl_symbol& varname)
{ return The(cl_univpoly_modint_ring) (cl_find_univpoly_ring((const cl_ring&)r,varname)); }

// Operations on polynomials.

// Add.
inline cl_UP_MI operator+ (const cl_UP_MI& x, const cl_UP_MI& y)
	{ return x.ring()->plus(x,y); }

// Negate.
inline cl_UP_MI operator- (const cl_UP_MI& x)
	{ return x.ring()->uminus(x); }

// Subtract.
inline cl_UP_MI operator- (const cl_UP_MI& x, const cl_UP_MI& y)
	{ return x.ring()->minus(x,y); }

// Multiply.
inline cl_UP_MI operator* (const cl_UP_MI& x, const cl_UP_MI& y)
	{ return x.ring()->mul(x,y); }

// Squaring.
inline cl_UP_MI square (const cl_UP_MI& x)
	{ return x.ring()->square(x); }

// Exponentiation x^y, where y > 0.
inline cl_UP_MI expt_pos (const cl_UP_MI& x, const cl_I& y)
	{ return x.ring()->expt_pos(x,y); }

// Scalar multiplication.
inline cl_UP_MI operator* (const cl_I& x, const cl_UP_MI& y)
	{ return y.ring()->mul(y.ring()->canonhom(x),y); }
inline cl_UP_MI operator* (const cl_UP_MI& x, const cl_I& y)
	{ return x.ring()->mul(x.ring()->canonhom(y),x); }

// Coefficient.
inline cl_MI coeff (const cl_UP_MI& x, uintL index)
	{ return x.ring()->coeff(x,index); }

// Destructive modification.
inline void set_coeff (cl_UP_MI& x, uintL index, const cl_MI& y)
	{ x.ring()->set_coeff(x,index,y); }
inline void finalize (cl_UP_MI& x)
	{ x.ring()->finalize(x); }
inline void cl_UP_MI::set_coeff (uintL index, const cl_MI& y)
	{ ring()->set_coeff(*this,index,y); }
inline void cl_UP_MI::finalize ()
	{ ring()->finalize(*this); }

// Evaluation. (No extension of the base ring allowed here for now.)
inline cl_MI cl_UP_MI::operator() (const cl_MI& y) const
{
	return ring()->eval(*this,y);
}

// Derivative.
inline cl_UP_MI deriv (const cl_UP_MI& x)
	{ return The2(cl_UP_MI)(deriv((const cl_UP&)x)); }

#endif /* _CL_UNIVPOLY_MODINT_H */
