// eval_rational_series().

// General includes.
#include "cl_sysdep.h"

// Specification.
#include "cl_LF_tran.h"


// Implementation.

#include "cl_lfloat.h"
#include "cl_integer.h"
#include "cl_abort.h"
#include "cl_LF.h"

// Subroutine.
// Evaluates S = sum(N1 <= n < N2, a(n)/b(n) * (p(N1)...p(n))/(q(N1)...q(n)))
// and returns P = p(N1)...p(N2-1), Q = q(N1)...q(N2-1), B = B(N1)...B(N2-1)
// and T = B*Q*S (all integers). On entry N1 < N2.
// P will not be computed if a NULL pointer is passed.

// Subroutines with q, but no qs:
//   eval_pqab_series_aux
//   eval_pqb_series_aux
//   eval_pqa_series_aux
//   eval_pq_series_aux
// Subroutines with q and qs:
//   eval_pqsab_series_aux
//   eval_pqsb_series_aux
//   eval_pqsa_series_aux
//   eval_pqs_series_aux
// Subroutines without q:
//   eval_pab_series_aux
//   eval_pb_series_aux
//   eval_pqa_series_aux
//   eval_p_series_aux
// Subroutines without p:
//   eval_qab_series_aux
//   eval_qb_series_aux
//   eval_qa_series_aux
//   eval_q_series_aux
// Subroutines without p and q:
//   eval_ab_series_aux
//   eval_b_series_aux
//   eval_a_series_aux
//   eval__series_aux

static void eval_pqab_series_aux (uintL N1, uintL N2,
                                  const cl_rational_series& args,
                                  cl_I* P, cl_I* Q, cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*Q = args.qv[N1];
		*B = args.bv[N1];
		*T = args.av[N1] * args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*Q = args.qv[N1] * args.qv[N1+1];
		*B = args.bv[N1] * args.bv[N1+1];
		*T = args.bv[N1+1] * args.qv[N1+1] * args.av[N1] * args.pv[N1]
		   + args.bv[N1] * args.av[N1+1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = b12 * q12 * args.av[N1] * args.pv[N1]
		   + args.bv[N1] * (args.bv[N1+2] * args.qv[N1+2] * args.av[N1+1] * p01
		                    + args.bv[N1+1] * args.av[N1+2] * p012);
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = b23 * (args.bv[N1+1] * q123 * args.av[N1] * args.pv[N1]
		            + args.bv[N1] * q23 * args.av[N1+1] * p01)
		   + b01 * (args.bv[N1+3] * args.qv[N1+3] * args.av[N1+2] * p012
		            + args.bv[N1+2] * args.av[N1+3] * p0123);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LQ, LB, LT;
		eval_pqab_series_aux(N1,Nm,args,&LP,&LQ,&LB,&LT);
		// Compute right part.
		var cl_I RP, RQ, RB, RT;
		eval_pqab_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RQ,&RB,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*Q = LQ*RQ;
		*B = LB*RB;
		// S = LS + LP/LQ * RS, so T = RB*RQ*LT + LB*LP*RT.
		*T = RB*RQ*LT + LB*LP*RT;
		break;
		}
	}
}
static void eval_pqb_series_aux (uintL N1, uintL N2,
                                 const cl_rational_series& args,
                                 cl_I* P, cl_I* Q, cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*Q = args.qv[N1];
		*B = args.bv[N1];
		*T = args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*Q = args.qv[N1] * args.qv[N1+1];
		*B = args.bv[N1] * args.bv[N1+1];
		*T = args.bv[N1+1] * args.qv[N1+1] * args.pv[N1]
		   + args.bv[N1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = b12 * q12 * args.pv[N1]
		   + args.bv[N1] * (args.bv[N1+2] * args.qv[N1+2] * p01
		                    + args.bv[N1+1] * p012);
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = b23 * (args.bv[N1+1] * q123 * args.pv[N1]
		            + args.bv[N1] * q23 * p01)
		   + b01 * (args.bv[N1+3] * args.qv[N1+3] * p012
		            + args.bv[N1+2] * p0123);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LQ, LB, LT;
		eval_pqb_series_aux(N1,Nm,args,&LP,&LQ,&LB,&LT);
		// Compute right part.
		var cl_I RP, RQ, RB, RT;
		eval_pqb_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RQ,&RB,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*Q = LQ*RQ;
		*B = LB*RB;
		// S = LS + LP/LQ * RS, so T = RB*RQ*LT + LB*LP*RT.
		*T = RB*RQ*LT + LB*LP*RT;
		break;
		}
	}
}
static void eval_pqa_series_aux (uintL N1, uintL N2,
                                 const cl_rational_series& args,
                                 cl_I* P, cl_I* Q, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*Q = args.qv[N1];
		*T = args.av[N1] * args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*Q = args.qv[N1] * args.qv[N1+1];
		*T = args.qv[N1+1] * args.av[N1] * args.pv[N1]
		   + args.av[N1+1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		*T = q12 * args.av[N1] * args.pv[N1]
		   + args.qv[N1+2] * args.av[N1+1] * p01
		   + args.av[N1+2] * p012;
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		*T = q123 * args.av[N1] * args.pv[N1]
		   + q23 * args.av[N1+1] * p01
		   + args.qv[N1+3] * args.av[N1+2] * p012
		   + args.av[N1+3] * p0123;
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LQ, LT;
		eval_pqa_series_aux(N1,Nm,args,&LP,&LQ,&LT);
		// Compute right part.
		var cl_I RP, RQ, RT;
		eval_pqa_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RQ,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*Q = LQ*RQ;
		// S = LS + LP/LQ * RS, so T = RQ*LT + LP*RT.
		*T = RQ*LT + LP*RT;
		break;
		}
	}
}
static void eval_pq_series_aux (uintL N1, uintL N2,
                                const cl_rational_series& args,
                                cl_I* P, cl_I* Q, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*Q = args.qv[N1];
		*T = args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*Q = args.qv[N1] * args.qv[N1+1];
		*T = args.qv[N1+1] * args.pv[N1]
		   + p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		*T = q12 * args.pv[N1]
		   + args.qv[N1+2] * p01
		   + p012;
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		*T = q123 * args.pv[N1]
		   + q23 * p01
		   + args.qv[N1+3] * p012
		   + p0123;
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LQ, LT;
		eval_pq_series_aux(N1,Nm,args,&LP,&LQ,&LT);
		// Compute right part.
		var cl_I RP, RQ, RT;
		eval_pq_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RQ,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*Q = LQ*RQ;
		// S = LS + LP/LQ * RS, so T = RQ*LT + LP*RT.
		*T = RQ*LT + LP*RT;
		break;
		}
	}
}

static void eval_pqsab_series_aux (uintL N1, uintL N2,
                                   const cl_rational_series& args,
                                   cl_I* P, cl_I* Q, uintL* QS, cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*Q = args.qv[N1];
		*QS = args.qsv[N1];
		*B = args.bv[N1];
		*T = args.av[N1] * args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*Q = args.qv[N1] * args.qv[N1+1];
		*QS = args.qsv[N1] + args.qsv[N1+1];
		*B = args.bv[N1] * args.bv[N1+1];
		*T = ((args.bv[N1+1] * args.qv[N1+1] * args.av[N1] * args.pv[N1]) << args.qsv[N1+1])
		   + args.bv[N1] * args.av[N1+1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		*QS = args.qsv[N1] + args.qsv[N1+1] + args.qsv[N1+2];
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = ((b12 * q12 * args.av[N1] * args.pv[N1]) << (args.qsv[N1+1] + args.qsv[N1+2]))
		   + args.bv[N1] * (((args.bv[N1+2] * args.qv[N1+2] * args.av[N1+1] * p01) << args.qsv[N1+2])
		                    + args.bv[N1+1] * args.av[N1+2] * p012);
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		*QS = args.qsv[N1] + args.qsv[N1+1] + args.qsv[N1+2] + args.qsv[N1+3];
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = ((b23 * (((args.bv[N1+1] * q123 * args.av[N1] * args.pv[N1]) << args.qsv[N1+1])
		              + args.bv[N1] * q23 * args.av[N1+1] * p01)) << (args.qsv[N1+2] + args.qsv[N1+3]))
		   + b01 * (((args.bv[N1+3] * args.qv[N1+3] * args.av[N1+2] * p012) << args.qsv[N1+3])
		            + args.bv[N1+2] * args.av[N1+3] * p0123);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LQ, LB, LT;
		var uintL LQS;
		eval_pqsab_series_aux(N1,Nm,args,&LP,&LQ,&LQS,&LB,&LT);
		// Compute right part.
		var cl_I RP, RQ, RB, RT;
		var uintL RQS;
		eval_pqsab_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RQ,&RQS,&RB,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*Q = LQ*RQ;
		*QS = LQS+RQS;
		*B = LB*RB;
		// S = LS + LP/LQ * RS, so T = RB*RQ*LT + LB*LP*RT.
		*T = ((RB*RQ*LT) << RQS) + LB*LP*RT;
		break;
		}
	}
}
static void eval_pqsb_series_aux (uintL N1, uintL N2,
                                  const cl_rational_series& args,
                                  cl_I* P, cl_I* Q, uintL* QS, cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*Q = args.qv[N1];
		*QS = args.qsv[N1];
		*B = args.bv[N1];
		*T = args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*Q = args.qv[N1] * args.qv[N1+1];
		*QS = args.qsv[N1] + args.qsv[N1+1];
		*B = args.bv[N1] * args.bv[N1+1];
		*T = ((args.bv[N1+1] * args.qv[N1+1] * args.pv[N1]) << args.qsv[N1+1])
		   + args.bv[N1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		*QS = args.qsv[N1] + args.qsv[N1+1] + args.qsv[N1+2];
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = ((b12 * q12 * args.pv[N1]) << (args.qsv[N1+1] + args.qsv[N1+2]))
		   + args.bv[N1] * (((args.bv[N1+2] * args.qv[N1+2] * p01) << args.qsv[N1+2])
		                    + args.bv[N1+1] * p012);
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		*QS = args.qsv[N1] + args.qsv[N1+1] + args.qsv[N1+2] + args.qsv[N1+3];
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = ((b23 * (((args.bv[N1+1] * q123 * args.pv[N1]) << args.qsv[N1+1])
		              + args.bv[N1] * q23 * p01)) << (args.qsv[N1+2] + args.qsv[N1+3]))
		   + b01 * (((args.bv[N1+3] * args.qv[N1+3] * p012) << args.qsv[N1+3])
		            + args.bv[N1+2] * p0123);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LQ, LB, LT;
		var uintL LQS;
		eval_pqsb_series_aux(N1,Nm,args,&LP,&LQ,&LQS,&LB,&LT);
		// Compute right part.
		var cl_I RP, RQ, RB, RT;
		var uintL RQS;
		eval_pqsb_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RQ,&RQS,&RB,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*Q = LQ*RQ;
		*QS = LQS+RQS;
		*B = LB*RB;
		// S = LS + LP/LQ * RS, so T = RB*RQ*LT + LB*LP*RT.
		*T = ((RB*RQ*LT) << RQS) + LB*LP*RT;
		break;
		}
	}
}
static void eval_pqsa_series_aux (uintL N1, uintL N2,
                                  const cl_rational_series& args,
                                  cl_I* P, cl_I* Q, uintL* QS, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*Q = args.qv[N1];
		*QS = args.qsv[N1];
		*T = args.av[N1] * args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*Q = args.qv[N1] * args.qv[N1+1];
		*QS = args.qsv[N1] + args.qsv[N1+1];
		*T = ((args.qv[N1+1] * args.av[N1] * args.pv[N1]) << args.qsv[N1+1])
		   + args.av[N1+1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		*QS = args.qsv[N1] + args.qsv[N1+1] + args.qsv[N1+2];
		*T = ((q12 * args.av[N1] * args.pv[N1]) << (args.qsv[N1+1] + args.qsv[N1+2]))
		   + ((args.qv[N1+2] * args.av[N1+1] * p01) << args.qsv[N1+2])
		   + args.av[N1+2] * p012;
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		*QS = args.qsv[N1] + args.qsv[N1+1] + args.qsv[N1+2] + args.qsv[N1+3];
		*T = ((((((q123 * args.av[N1] * args.pv[N1]) << args.qsv[N1+1])
		         + q23 * args.av[N1+1] * p01) << args.qsv[N1+2])
		       + args.qv[N1+3] * args.av[N1+2] * p012) << args.qsv[N1+3])
		   + args.av[N1+3] * p0123;
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LQ, LT;
		var uintL LQS;
		eval_pqsa_series_aux(N1,Nm,args,&LP,&LQ,&LQS,&LT);
		// Compute right part.
		var cl_I RP, RQ, RT;
		var uintL RQS;
		eval_pqsa_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RQ,&RQS,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*Q = LQ*RQ;
		*QS = LQS+RQS;
		// S = LS + LP/LQ * RS, so T = RQ*LT + LP*RT.
		*T = ((RQ*LT) << RQS) + LP*RT;
		break;
		}
	}
}
static void eval_pqs_series_aux (uintL N1, uintL N2,
                                 const cl_rational_series& args,
                                 cl_I* P, cl_I* Q, uintL* QS, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*Q = args.qv[N1];
		*QS = args.qsv[N1];
		*T = args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*Q = args.qv[N1] * args.qv[N1+1];
		*QS = args.qsv[N1] + args.qsv[N1+1];
		*T = ((args.qv[N1+1] * args.pv[N1]) << args.qsv[N1+1])
		   + p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		*QS = args.qsv[N1] + args.qsv[N1+1] + args.qsv[N1+2];
		*T = ((q12 * args.pv[N1]) << (args.qsv[N1+1] + args.qsv[N1+2]))
		   + ((args.qv[N1+2] * p01) << args.qsv[N1+2])
		   + p012;
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		*QS = args.qsv[N1] + args.qsv[N1+1] + args.qsv[N1+2] + args.qsv[N1+3];
		*T = ((((((q123 * args.pv[N1]) << args.qsv[N1+1])
		         + q23 * p01) << args.qsv[N1+2])
		       + args.qv[N1+3] * p012) << args.qsv[N1+3])
		   + p0123;
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LQ, LT;
		var uintL LQS;
		eval_pqs_series_aux(N1,Nm,args,&LP,&LQ,&LQS,&LT);
		// Compute right part.
		var cl_I RP, RQ, RT;
		var uintL RQS;
		eval_pqs_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RQ,&RQS,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*Q = LQ*RQ;
		*QS = LQS+RQS;
		// S = LS + LP/LQ * RS, so T = RQ*LT + LP*RT.
		*T = ((RQ*LT) << RQS) + LP*RT;
		break;
		}
	}
}

static void eval_pab_series_aux (uintL N1, uintL N2,
                                 const cl_rational_series& args,
                                 cl_I* P, cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*B = args.bv[N1];
		*T = args.av[N1] * args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*B = args.bv[N1] * args.bv[N1+1];
		*T = args.bv[N1+1] * args.av[N1] * args.pv[N1]
		   + args.bv[N1] * args.av[N1+1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = b12 * args.av[N1] * args.pv[N1]
		   + args.bv[N1] * (args.bv[N1+2] * args.av[N1+1] * p01
		                    + args.bv[N1+1] * args.av[N1+2] * p012);
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = b23 * (args.bv[N1+1] * args.av[N1] * args.pv[N1]
		            + args.bv[N1] * args.av[N1+1] * p01)
		   + b01 * (args.bv[N1+3] * args.av[N1+2] * p012
		            + args.bv[N1+2] * args.av[N1+3] * p0123);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LB, LT;
		eval_pab_series_aux(N1,Nm,args,&LP,&LB,&LT);
		// Compute right part.
		var cl_I RP, RB, RT;
		eval_pab_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RB,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*B = LB*RB;
		// S = LS + LP * RS, so T = RB*LT + LB*LP*RT.
		*T = RB*LT + LB*LP*RT;
		break;
		}
	}
}
static void eval_pb_series_aux (uintL N1, uintL N2,
                                const cl_rational_series& args,
                                cl_I* P, cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*B = args.bv[N1];
		*T = args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*B = args.bv[N1] * args.bv[N1+1];
		*T = args.bv[N1+1] * args.pv[N1]
		   + args.bv[N1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = b12 * args.pv[N1]
		   + args.bv[N1] * (args.bv[N1+2] * p01
		                    + args.bv[N1+1] * p012);
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = b23 * (args.bv[N1+1] * args.pv[N1]
		            + args.bv[N1] * p01)
		   + b01 * (args.bv[N1+3] * p012
		            + args.bv[N1+2] * p0123);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LB, LT;
		eval_pb_series_aux(N1,Nm,args,&LP,&LB,&LT);
		// Compute right part.
		var cl_I RP, RB, RT;
		eval_pb_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RB,&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		*B = LB*RB;
		// S = LS + LP * RS, so T = RB*LT + LB*LP*RT.
		*T = RB*LT + LB*LP*RT;
		break;
		}
	}
}
static void eval_pa_series_aux (uintL N1, uintL N2,
                                const cl_rational_series& args,
                                cl_I* P, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*T = args.av[N1] * args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*T = args.av[N1] * args.pv[N1]
		   + args.av[N1+1] * p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		*T = args.av[N1] * args.pv[N1]
		   + args.av[N1+1] * p01
		   + args.av[N1+2] * p012;
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		*T = args.av[N1] * args.pv[N1]
		   + args.av[N1+1] * p01
		   + args.av[N1+2] * p012
		   + args.av[N1+3] * p0123;
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LT;
		eval_pa_series_aux(N1,Nm,args,&LP,&LT);
		// Compute right part.
		var cl_I RP, RT;
		eval_pa_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		// S = LS + LP * RS, so T = LT + LP*RT.
		*T = LT + LP*RT;
		break;
		}
	}
}
static void eval_p_series_aux (uintL N1, uintL N2,
                               const cl_rational_series& args,
                               cl_I* P, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		if (P) { *P = args.pv[N1]; }
		*T = args.pv[N1];
		break;
	case 2: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		if (P) { *P = p01; }
		*T = args.pv[N1]
		   + p01;
		break;
		}
	case 3: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		if (P) { *P = p012; }
		*T = args.pv[N1]
		   + p01
		   + p012;
		break;
		}
	case 4: {
		var cl_I p01 = args.pv[N1] * args.pv[N1+1];
		var cl_I p012 = p01 * args.pv[N1+2];
		var cl_I p0123 = p012 * args.pv[N1+3];
		if (P) { *P = p0123; }
		*T = args.pv[N1]
		   + p01
		   + p012
		   + p0123;
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LP, LT;
		eval_p_series_aux(N1,Nm,args,&LP,&LT);
		// Compute right part.
		var cl_I RP, RT;
		eval_p_series_aux(Nm,N2,args,(P?&RP:(cl_I*)0),&RT);
		// Put together partial results.
		if (P) { *P = LP*RP; }
		// S = LS + LP * RS, so T = LT + LP*RT.
		*T = LT + LP*RT;
		break;
		}
	}
}

static void eval_qab_series_aux (uintL N1, uintL N2,
                                 const cl_rational_series& args,
                                 cl_I* Q, cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		*Q = args.qv[N1];
		*B = args.bv[N1];
		*T = args.av[N1];
		break;
	case 2: {
		*Q = args.qv[N1] * args.qv[N1+1];
		*B = args.bv[N1] * args.bv[N1+1];
		*T = args.bv[N1+1] * args.qv[N1+1] * args.av[N1]
		   + args.bv[N1] * args.av[N1+1];
		break;
		}
	case 3: {
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = b12 * q12 * args.av[N1]
		   + args.bv[N1] * (args.bv[N1+2] * args.qv[N1+2] * args.av[N1+1]
		                    + args.bv[N1+1] * args.av[N1+2]);
		break;
		}
	case 4: {
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = b23 * (args.bv[N1+1] * q123 * args.av[N1]
		            + args.bv[N1] * q23 * args.av[N1+1])
		   + b01 * (args.bv[N1+3] * args.qv[N1+3] * args.av[N1+2]
		            + args.bv[N1+2] * args.av[N1+3]);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LQ, LB, LT;
		eval_qab_series_aux(N1,Nm,args,&LQ,&LB,&LT);
		// Compute right part.
		var cl_I RQ, RB, RT;
		eval_qab_series_aux(Nm,N2,args,&RQ,&RB,&RT);
		// Put together partial results.
		*Q = LQ*RQ;
		*B = LB*RB;
		// S = LS + 1/LQ * RS, so T = RB*RQ*LT + LB*RT.
		*T = RB*RQ*LT + LB*RT;
		break;
		}
	}
}
static void eval_qb_series_aux (uintL N1, uintL N2,
                                const cl_rational_series& args,
                                cl_I* Q, cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		*Q = args.qv[N1];
		*B = args.bv[N1];
		*T = 1;
		break;
	case 2: {
		*Q = args.qv[N1] * args.qv[N1+1];
		*B = args.bv[N1] * args.bv[N1+1];
		*T = args.bv[N1+1] * args.qv[N1+1]
		   + args.bv[N1];
		break;
		}
	case 3: {
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = b12 * q12
		   + args.bv[N1] * (args.bv[N1+2] * args.qv[N1+2]
		                    + args.bv[N1+1]);
		break;
		}
	case 4: {
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = b23 * (args.bv[N1+1] * q123
		            + args.bv[N1] * q23)
		   + b01 * (args.bv[N1+3] * args.qv[N1+3]
		            + args.bv[N1+2]);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LQ, LB, LT;
		eval_qb_series_aux(N1,Nm,args,&LQ,&LB,&LT);
		// Compute right part.
		var cl_I RQ, RB, RT;
		eval_qb_series_aux(Nm,N2,args,&RQ,&RB,&RT);
		// Put together partial results.
		*Q = LQ*RQ;
		*B = LB*RB;
		// S = LS + 1/LQ * RS, so T = RB*RQ*LT + LB*RT.
		*T = RB*RQ*LT + LB*RT;
		break;
		}
	}
}
static void eval_qa_series_aux (uintL N1, uintL N2,
                                const cl_rational_series& args,
                                cl_I* Q, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		*Q = args.qv[N1];
		*T = args.av[N1];
		break;
	case 2: {
		*Q = args.qv[N1] * args.qv[N1+1];
		*T = args.qv[N1+1] * args.av[N1]
		   + args.av[N1+1];
		break;
		}
	case 3: {
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		*T = q12 * args.av[N1]
		   + args.qv[N1+2] * args.av[N1+1]
		   + args.av[N1+2];
		break;
		}
	case 4: {
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		*T = q123 * args.av[N1]
		   + q23 * args.av[N1+1]
		   + args.qv[N1+3] * args.av[N1+2]
		   + args.av[N1+3];
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LQ, LT;
		eval_qa_series_aux(N1,Nm,args,&LQ,&LT);
		// Compute right part.
		var cl_I RQ, RT;
		eval_qa_series_aux(Nm,N2,args,&RQ,&RT);
		// Put together partial results.
		*Q = LQ*RQ;
		// S = LS + 1/LQ * RS, so T = RQ*LT + RT.
		*T = RQ*LT + RT;
		break;
		}
	}
}
static void eval_q_series_aux (uintL N1, uintL N2,
                               const cl_rational_series& args,
                               cl_I* Q, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		*Q = args.qv[N1];
		*T = 1;
		break;
	case 2: {
		*Q = args.qv[N1] * args.qv[N1+1];
		*T = args.qv[N1+1]
		   + 1;
		break;
		}
	case 3: {
		var cl_I q12 = args.qv[N1+1] * args.qv[N1+2];
		*Q = args.qv[N1] * q12;
		*T = q12
		   + args.qv[N1+2]
		   + 1;
		break;
		}
	case 4: {
		var cl_I q23 = args.qv[N1+2] * args.qv[N1+3];
		var cl_I q123 = args.qv[N1+1] * q23;
		*Q = args.qv[N1] * q123;
		*T = q123
		   + q23
		   + args.qv[N1+3]
		   + 1;
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LQ, LT;
		eval_q_series_aux(N1,Nm,args,&LQ,&LT);
		// Compute right part.
		var cl_I RQ, RT;
		eval_q_series_aux(Nm,N2,args,&RQ,&RT);
		// Put together partial results.
		*Q = LQ*RQ;
		// S = LS + 1/LQ * RS, so T = RQ*LT + RT.
		*T = RQ*LT + RT;
		break;
		}
	}
}

static void eval_ab_series_aux (uintL N1, uintL N2,
                                const cl_rational_series& args,
                                cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		*B = args.bv[N1];
		*T = args.av[N1];
		break;
	case 2: {
		*B = args.bv[N1] * args.bv[N1+1];
		*T = args.bv[N1+1] * args.av[N1]
		   + args.bv[N1] * args.av[N1+1];
		break;
		}
	case 3: {
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = b12 * args.av[N1]
		   + args.bv[N1] * (args.bv[N1+2] * args.av[N1+1]
		                    + args.bv[N1+1] * args.av[N1+2]);
		break;
		}
	case 4: {
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = b23 * (args.bv[N1+1] * args.av[N1]
		            + args.bv[N1] * args.av[N1+1])
		   + b01 * (args.bv[N1+3] * args.av[N1+2]
		            + args.bv[N1+2] * args.av[N1+3]);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LB, LT;
		eval_ab_series_aux(N1,Nm,args,&LB,&LT);
		// Compute right part.
		var cl_I RB, RT;
		eval_ab_series_aux(Nm,N2,args,&RB,&RT);
		// Put together partial results.
		*B = LB*RB;
		// S = LS + RS, so T = RB*LT + LB*RT.
		*T = RB*LT + LB*RT;
		break;
		}
	}
}
static void eval_b_series_aux (uintL N1, uintL N2,
                               const cl_rational_series& args,
                               cl_I* B, cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		*B = args.bv[N1];
		*T = 1;
		break;
	case 2: {
		*B = args.bv[N1] * args.bv[N1+1];
		*T = args.bv[N1+1]
		   + args.bv[N1];
		break;
		}
	case 3: {
		var cl_I b12 = args.bv[N1+1] * args.bv[N1+2];
		*B = args.bv[N1] * b12;
		*T = b12
		   + args.bv[N1] * (args.bv[N1+2]
		                    + args.bv[N1+1]);
		break;
		}
	case 4: {
		var cl_I b01 = args.bv[N1] * args.bv[N1+1];
		var cl_I b23 = args.bv[N1+2] * args.bv[N1+3];
		*B = b01 * b23;
		*T = b23 * (args.bv[N1+1]
		            + args.bv[N1])
		   + b01 * (args.bv[N1+3]
		            + args.bv[N1+2]);
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LB, LT;
		eval_b_series_aux(N1,Nm,args,&LB,&LT);
		// Compute right part.
		var cl_I RB, RT;
		eval_b_series_aux(Nm,N2,args,&RB,&RT);
		// Put together partial results.
		*B = LB*RB;
		// S = LS + RS, so T = RB*LT + LB*RT.
		*T = RB*LT + LB*RT;
		break;
		}
	}
}
static void eval_a_series_aux (uintL N1, uintL N2,
                               const cl_rational_series& args,
                               cl_I* T)
{
	switch (N2 - N1) {
	case 0:
		cl_abort(); break;
	case 1:
		*T = args.av[N1];
		break;
	case 2: {
		*T = args.av[N1]
		   + args.av[N1+1];
		break;
		}
	case 3: {
		*T = args.av[N1]
		   + args.av[N1+1]
		   + args.av[N1+2];
		break;
		}
	case 4: {
		*T = args.av[N1]
		   + args.av[N1+1]
		   + args.av[N1+2]
		   + args.av[N1+3];
		break;
		}
	default: {
		var uintL Nm = (N1+N2)/2; // midpoint
		// Compute left part.
		var cl_I LT;
		eval_a_series_aux(N1,Nm,args,&LT);
		// Compute right part.
		var cl_I RT;
		eval_a_series_aux(Nm,N2,args,&RT);
		// Put together partial results.
		// S = LS + RS, so T = LT + RT.
		*T = LT + RT;
		break;
		}
	}
}
static inline void eval__series_aux (uintL N1, uintL N2,
                                     const cl_rational_series& args,
                                     cl_I* T)
{
	unused args;
	*T = N2-N1;
}

cl_LF eval_rational_series (uintL N, const cl_rational_series& args, uintC len)
{
	if (N==0)
		return cl_I_to_LF(0,len);
	var cl_I Q, B, T;
	if (args.pv) {
		if (args.qv) {
			if (!args.qsv) {
				if (args.bv) {
					if (args.av)
						eval_pqab_series_aux(0,N,args,NULL,&Q,&B,&T);
					else
						eval_pqb_series_aux(0,N,args,NULL,&Q,&B,&T);
					return cl_I_to_LF(T,len) / cl_I_to_LF(B*Q,len);
				} else {
					if (args.av)
						eval_pqa_series_aux(0,N,args,NULL,&Q,&T);
					else
						eval_pq_series_aux(0,N,args,NULL,&Q,&T);
					return cl_I_to_LF(T,len) / cl_I_to_LF(Q,len);
				}
			} else {
				// Precomputation of the shift counts:
				// Split qv[n] into qv[n]*2^qsv[n].
				{
					var cl_I* qp = args.qv;
					var uintL* qsp = args.qsv;
					for (var uintL n = 0; n < N; n++, qp++, qsp++) {
						// Pull out maximal power of 2 out of *qp = args.qv[n].
						var uintL qs = 0;
						if (!zerop(*qp)) {
							qs = ord2(*qp);
							if (qs > 0)
								*qp = *qp >> qs;
						}
						*qsp = qs;
					}
				}
				// Main computation.
				uintL QS;
				if (args.bv) {
					if (args.av)
						eval_pqsab_series_aux(0,N,args,NULL,&Q,&QS,&B,&T);
					else
						eval_pqsb_series_aux(0,N,args,NULL,&Q,&QS,&B,&T);
					return cl_I_to_LF(T,len) / scale_float(cl_I_to_LF(B*Q,len),QS);
				} else {
					if (args.av)
						eval_pqsa_series_aux(0,N,args,NULL,&Q,&QS,&T);
					else
						eval_pqs_series_aux(0,N,args,NULL,&Q,&QS,&T);
					return cl_I_to_LF(T,len) / scale_float(cl_I_to_LF(Q,len),QS);
				}
			}
		} else {
			if (args.bv) {
				if (args.av)
					eval_pab_series_aux(0,N,args,NULL,&B,&T);
				else
					eval_pb_series_aux(0,N,args,NULL,&B,&T);
				return cl_I_to_LF(T,len) / cl_I_to_LF(B,len);
			} else {
				if (args.av)
					eval_pa_series_aux(0,N,args,NULL,&T);
				else
					eval_p_series_aux(0,N,args,NULL,&T);
				return cl_I_to_LF(T,len);
			}
		}
	} else {
		if (args.qv) {
			if (args.bv) {
				if (args.av)
					eval_qab_series_aux(0,N,args,&Q,&B,&T);
				else
					eval_qb_series_aux(0,N,args,&Q,&B,&T);
				return cl_I_to_LF(T,len) / cl_I_to_LF(B*Q,len);
			} else {
				if (args.av)
					eval_qa_series_aux(0,N,args,&Q,&T);
				else
					eval_q_series_aux(0,N,args,&Q,&T);
				return cl_I_to_LF(T,len) / cl_I_to_LF(Q,len);
			}
		} else {
			if (args.bv) {
				if (args.av)
					eval_ab_series_aux(0,N,args,&B,&T);
				else
					eval_b_series_aux(0,N,args,&B,&T);
				return cl_I_to_LF(T,len) / cl_I_to_LF(B,len);
			} else {
				if (args.av)
					eval_a_series_aux(0,N,args,&T);
				else
					eval__series_aux(0,N,args,&T);
				return cl_I_to_LF(T,len);
			}
		}
	}
}
// Bit complexity (if p(n), q(n), a(n), b(n) have length O(log(n))):
// O(log(N)^2*M(N)).
