/*
C
C  _______________________________________________________________
C
C*   Licence
C    =======
C
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C
C  _______________________________________________________________
C
*/


#include "math.h"

#include "kask.h"
#include "kaskcmd.h"
#include "kasktri.h"
#include "kaskass.h"
#include "kasksol.h"

extern void Direct(), MeanValue(), ExtraPol(), Iterate(), Refine();

int StdF();
int StdCauchyF();
int StdDirichF(); 

#define NO_OF_F_VALS 6
#define NO_OF_CAU_VALS 2
#define NO_OF_DIR_VALS 1
#define NO_OF_SOL_VALS 3

static REAL	fVals[NO_OF_F_VALS],cauVals[NO_OF_CAU_VALS],
		dirVals[NO_OF_DIR_VALS],solVals[NO_OF_SOL_VALS]; 	

static PROBLEM initProblem =
  {
    "std",
	(PROC)StdF, 
	fVals,
	NO_OF_F_VALS,
	(PROC)StdCauchyF,
	cauVals,
	NO_OF_CAU_VALS,
	(PROC)StdDirichF,
	dirVals,
	NO_OF_DIR_VALS,
	nil,
	solVals,
	NO_OF_SOL_VALS,
	(PROC)NumAss,
	ZERO,
	true,
	nil
  };


PROBLEM *actProblem = & initProblem;



	/*		Example: The Model Problem		*/

static int StdF(x, y,class,fVals)
	REAL x, y;
	int class;
	REAL fVals[];
  {
	fVals[0]	=  1.0;		/*	pX	*/
	fVals[1]	=  1.0;		/*	pY	*/
	fVals[2]	=  0.0;		/*	q	*/
	fVals[3]	= -1.0;		/*	g	*/
    return true;
  }
 
static int StdCauchyF(x, y,class,cauVals)
	REAL x, y;
	int class;
	REAL cauVals[];
  {
	cauVals[0]	= 0.0;		/*	Sigma	*/
	cauVals[1]	= 0.0;		/*	Xi	*/
    return true;
  }
 

static int StdDirichF(x, y,class,dirVals)
	REAL x, y;
	int class;
	REAL dirVals[];
  {
	dirVals[0] = 0.0;
    return true ;
  }


/*	Example: Randy Bank	*/

static int BankF(x, y,class,fVals)
	REAL x, y;
	int class;
	REAL fVals[];
  {
	fVals[0]	=  1.0;		/*	pX	*/
	fVals[1]	=  1.0;		/*	pY	*/
	fVals[2]	=  0.0;		/*	q	*/
	fVals[3]	=  0.0;		/*	g	*/
    return true;
  }

static REAL arccos(x)
  REAL x;
  {
	REAL y=(fabs(x)<SMALL)?REALPI2:atan(sqrt(ONE-x*x)/x);
	if (x<=-SMALL) y = REALPI+y;
	return y;
  }
 
static int RandyBF(x, y,class,dirVals)
	REAL x, y;
	int class;
	REAL dirVals[];
  {  
   	REAL r = sqrt(x*x+y*y), phi;
	if (class==1)
	 	dirVals[0]=sqrt(sqrt(r));
	else
	   	{
		phi = (r>0)?arccos(x/r):ZERO;
		if (y<ZERO) phi = TWO*REALPI-phi;
		r =sqrt(sqrt(r));
		dirVals[0]= r*sin(QUARTER*phi);
		}

    return true ;
  }

static int BankSol(x, y, class, solVals)
	REAL x, y, solVals[];
	int class;
  {
   	REAL r = sqrt(x*x+y*y), phi;

	if ((class==2) || (class==1))
		{
	 	solVals[0]=sqrt(sqrt(r));
		}
	else
	   	{
		phi = (r>0)?arccos(x/r):ZERO;
		if (y<ZERO) phi = TWO*REALPI-phi;
		r =sqrt(sqrt(r));
		solVals[0]= r*sin(QUARTER*phi);
		}

    return true;    
  }

/*	Example: Discontinuous Dirichlet Boundary Conditions	*/

static int RandUnF(x, y,class,fVals)
	REAL x, y;
	int class;
	REAL fVals[];
  {
	fVals[0]	=  1.0;		/*	pX	*/
	fVals[1]	=  1.0;		/*	pY	*/
	fVals[2]	=  0.0;		/*	q	*/
	fVals[3]	=  0.0;		/*	g	*/
    return true;
  }
 
static int RandUnDirichF(x, y,class,dirVals)
	REAL x, y;
	int class;
	REAL dirVals[];
  {
	dirVals[0] = 0.0;
	if ((x<HALF)&&(y<HALF))   dirVals[0] = ONE;
	if ((x==ZERO)&&(y==HALF)) dirVals[0] = HALF;
	if ((y==ZERO)&&(x==HALF)) dirVals[0] = HALF;
    return true ;
  }

/*	Example: Discontinuous Coefficients	*/

static int RegpF(x, y,class,fVals)
	REAL x, y;
	int class;
	REAL fVals[];
  {
					/*	pX	*/
	if( class == 1) fVals[0]=1.0;
		else	fVals[0]=1.0e6;

	fVals[1]	=  fVals[0];	/*	pY	*/
	fVals[2]	=  0.0;		/*	q	*/
	fVals[3]	= -1.0;		/*	g	*/
    return true;
  }
 


/*		Example: mixed boundary values			*/

static int KreisF(x, y,class,fVals)
	REAL x, y;
	int class;
	REAL fVals[];
  {
	fVals[0] =  1.0;		/*	pX	*/
	fVals[1] =  1.0;		/*	pY	*/
	fVals[2] =  0.0;		/*	q	*/
	fVals[3] = -4.0;		/*	g	*/

    return true;
  }
 
  
static int KreisCauchyF(x, y,class,cauVals)
	REAL x, y;
	int class;
	REAL cauVals[];
  {
	REAL r;
	r = sqrt(x*x + y*y);
	cauVals[0] = 0.0;		/*	Sigma	*/
	cauVals[1] = 2.0*r;		/*	Xi	*/
     return true;   
  }


static int KreisDirichF(x, y,class,dirVals)
	REAL x, y;
	int class;
	REAL dirVals[];
  {
	dirVals[0] = 1.0;
    return true;
  }

static int KreisSol(x, y, class, solVals)
	REAL x, y, solVals[];
	int class;
  {
    solVals[0] = x*x+y*y;    
    solVals[1] = 2.0*x;    
    solVals[2] = 2.0*y;    
    return true;    
  }

/********************************************************/
/*	Problem-Command: Select a special example	*/
/********************************************************/

int StdProblem(cmd)
  COMMAND *cmd;
  {
	int index = 0;
	char *tp;

	if (ParsCheck(cmd,0,1)) return false;

	if ((cmd->noOfPars)==1)
	  {
	    tp = (cmd->pars)[1];
	    index = CheckName(&tp, cmd->names, nameClass);
	  }
	if (index==-1)
	  {
	    sprintf(globBuf,"Parameter %s of %s not allowed\n", (cmd->pars)[1], (cmd->pars)[0]);
		ZIBStdOut(globBuf);
	    return false;
	  }

/*		 Presets for Solve		*/
	if (!CheckPreSets()) return false;
	actSolve->Direct = Direct;
	actSolve->Estimate = MeanValue;
	actSolve->Refine = Refine;
	actSolve->Iterate = Iterate;
	
/*	select a special problem		*/

	actProblem->fVals		= fVals;
	actProblem->noFVals 	= NO_OF_F_VALS;
	actProblem->cauVals		= cauVals;
	actProblem->noCauVals	= NO_OF_CAU_VALS;
	actProblem->dirVals		= dirVals;
	actProblem->noDirVals	= NO_OF_DIR_VALS;
	actProblem->Sol			= nil;
	actProblem->solVals	 	= solVals;
	actProblem->noSolVals	= NO_OF_SOL_VALS;
	actProblem->NumAss	 	= NumAss;
	actProblem->class		= ZERO;
	actProblem->symP 		= SYMMETRIC;

	switch (index)
	  {
		case 0:				/*	Std		*/
		  actProblem->F			= (PROC)StdF;
		  actProblem->CauchyF	= (PROC)StdCauchyF;
		  actProblem->DirichF	= (PROC)StdDirichF;	
		  break;

		case 1:				/*	Banks		*/
	  	  actProblem->F			= (PROC)BankF;
		  actProblem->CauchyF	= (PROC)StdCauchyF;
		  actProblem->DirichF	= (PROC)RandyBF;
		  actProblem->Sol		= (PROC)BankSol;	
		  break;

		case 2:				/*	bounddisc	*/
	  	  actProblem->F			= (PROC)RandUnF;
		  actProblem->CauchyF	= (PROC)StdCauchyF;	
		  actProblem->DirichF	= (PROC)RandUnDirichF;
		  break;

		case 3:				/*	region		*/
	  	  actProblem->F			= (PROC)RegpF;
		  actProblem->CauchyF	= (PROC)StdCauchyF;
		  actProblem->DirichF	= (PROC)StdDirichF;
		  break;
	
		case 4:				/*	circle		*/
	  	  actProblem->F			= (PROC)KreisF;
		  actProblem->CauchyF	= (PROC)KreisCauchyF;	
		  actProblem->DirichF	= (PROC)KreisDirichF;
		  actProblem->Sol		= (PROC)KreisSol;
		  break;

		  }

	SetBound(R_SOL);
	InitNumAss(N_STD);

	if ((cmd->noOfPars)==1) actProblem->name = (cmd->names)[index];
	return true;
  }

int InfPrb(cmd)
  COMMAND *cmd;
  {
	sprintf(globBuf,"Problem: current problem is %s \n",actProblem->name);
	ZIBStdOut(globBuf);
	return true;
  }
