;;; JACAL: Symbolic Mathematics System.        -*-scheme-*-
;;; Copyright 1992, 1993 Aubrey Jaffer.
;;; See the file "COPYING" for terms applying to this program.

;(require 'record)
;(define grammar-rtd
;  (make-record-type "grammar"
;		    '(name reader lex-tab read-tab writer write-tab)))
;(define make-grammar (record-constructor grammar-rtd))
;(define grammar-name (record-accessor grammar-rtd 'name))
;(define grammar-reader (record-accessor grammar-rtd 'reader))
;(define grammar-lex-tab (record-accessor grammar-rtd 'lex-tab))
;(define grammar-read-tab (record-accessor grammar-rtd 'read-tab))
;(define grammar-writer (record-accessor grammar-rtd 'writer))
;(define grammar-write-tab (record-accessor grammar-rtd 'write-tab))

(define (make-grammar name reader lex-tab read-tab writer write-tab)
  (cons (cons name reader)
	(cons (cons lex-tab read-tab) (cons writer write-tab))))
(define grammar-name caar)
(define grammar-reader cdar)
(define grammar-lex-tab caadr)
(define grammar-read-tab cdadr)
(define grammar-writer caddr)
(define grammar-write-tab cdddr)

(require 'alist)
(define *grammars* '())
(define grammar-associator (alist-associator eq?))
(define (defgrammar name grm)
  (set! *grammars* (grammar-associator *grammars* name grm)))
(define grammar-remover (alist-remover eq?))
(define (rem-grammar name grm)
  (set! *grammars* (grammar-remover *grammars* name grm)))
(define grammar-inquirer (alist-inquirer eq?))
(define (get-grammar name) (grammar-inquirer *grammars* name))
(define (list-of-grammars)
  (define grammars '())
  (alist-for-each (lambda (k v) (set! grammars (cons k grammars))) *grammars*)
  grammars)


(defgrammar 'scheme
  (make-grammar 'scheme
		(lambda (grm) (read))
		#f
		#f
		(lambda (sexp grm) (write sexp))
		#f))

(defgrammar 'null
  (make-grammar 'null
		(lambda (grm) (math:error 'cannot-read-null-grammar))
		#f
		#f
		(lambda (sexp grm) #t)
		#f))

;;; Establish autoload for PRETTY-PRINT.
(defgrammar 'SchemePretty
  (let ((pploaded #f))
    (make-grammar 'SchemePretty
		  (lambda (grm) (read))
		  #f
		  #f
		  (lambda (sexp grm)
		    (or pploaded (begin (require 'pretty-print)
					(set! pploaded #t)))
		    (pretty-print sexp))
		  #f)))

(define (read-sexp grm)
  (funcall (grammar-reader grm) grm))
(define (write-sexp sexp grm)
  (funcall (grammar-writer grm) sexp grm))
(define (math:write e grm)
  (if (not (eq? 'null (grammar-name grm)))
      (write-sexp (math->sexp e horner) grm)))

(define (write-diag obj) (write obj (current-error-port)))
(define (display-diag obj) (display obj (current-error-port)))
(define (newline-diag)
  (let ((cep (current-error-port)))
    (newline cep) (force-output cep)))

;;;; careful write for displaying internal stuff
(define (math:print obj)
  (cond ((pair? obj)
	 (display-diag #\[)
	 (math:print (car obj))
	 (cond ((null? (cdr obj)))
	       ((list? (cdr obj))
		(for-each (lambda (x) (display-diag #\ ) (math:print x))
			  (cdr obj)))
	       (else (display-diag " . ") (math:print (cdr obj))))
	 (display-diag #\]))
	((poly:var? obj) (display-diag (var:sexp obj)))
	(else (write-diag obj)))
  obj)
(define (tran:translate sym)
  (let ((as (assq sym tran:translations)))
  (if as (cdr as) sym)))
(define (tran:display sym)
  (display (tran:translate sym)))
(define (math:warn . args)
  (display-diag ";;;")
  (let ((ans '()))
    (for-each (lambda (obj)
		(display-diag #\space)
		(if (symbol? obj)
		    (let ((symt (tran:translate obj)))
		      (display-diag symt)
		      (if (symbol? symt) (display-diag #\:)))
		    (set! ans (math:print obj))))
	      args)
    (newline-diag)
    ans))
(define (math:error . args)
  (newline-diag)
  (apply math:warn args)
  (if math:debug (error "") (math:exit #f)))
(define eval-error math:error)

(define (test ans fun . args)
  (let ((res (apply fun args)))
    (if (equal? ans res) #t (math:warn 'trouble-with fun))))

;;; outputs list of strings with as much per line as possible.
(define (block-write-strings l)
  (let* ((column 5)
	 (width (- (get-page-width) column))
	 (ps (make-string column #\  )))
    (set! column width)
    (for-each (lambda (ap)
		(set! column (+ (string-length ap) column))
		(cond ((and (positive? width) (>= column width))
		       (newline)
		       (display ps)
		       (set! column (string-length ap)))
		      (else
		       (display " ")
		       (set! column (+ column 1))))
		(display ap))
	      l)
    (newline)))

(define (get-page-height)
  (case page-height
    ((#f) 0)
    ((#t) (output-port-height (current-output-port)))
    (else page-height)))

(define (get-page-width)
  (case page-width
    ((#f) 0)
    ((#t) (output-port-width (current-output-port)))
    (else page-width)))

(define (paginate-file file)
  (call-with-input-file
      file
    (lambda (infile)
      (call-with-current-continuation
       (lambda (escape)
	 (let ((h (get-page-height))
	       (l 0))
	   (do ((c (read-char infile) (read-char infile)))
	       ((eof-object? c) novalue)
	     (display c)
	     (cond ((not (char=? #\newline c)))
		   ((zero? h))
		   ((< l h) (set! l (+ 1 l)))
		   ((do-more) (set! l 0))
		   (else (escape #f))))))))))

(define (do-more)
  (define helped #f)
  (tran:display 'more)
  (force-output)
  (let loop ((r (read-char)))
    (cond ((char=? #\  r) #t)
	  ((eof-object? r) #t)
	  ((char-whitespace? r) (loop (read-char)))
	  ((char-ci=? #\q r) #f)
	  (helped (loop (read-char)))
	  (else (tran:display 'q-to-quit-space-for-more:-)
		(force-output)
		(set! helped #t)
		(loop (read-char))))))
