/******************************************************************************
	anf_elt_unit_is_mth_power.c
******************************************************************************/

#include "kant.h"

integer_small
anf_elt_unit_is_mth_power WITH_4_ARGS(
	order,		ord,
	anf_elt,	alpha,
	integer_small,	p,
	anf_elt*,	beta
)

/******************************************************************************
 
Description:	Looks for a beta in ord satisfying alpha = beta^m
                modulo torsion units.

Calling sequence:
 
	bv =   anf_elt_unit_is_mth_power(ord,alpha,m,&beta);
	
	bv		integer_small	: -1 = unsufficient precision
					   0 = beta does not exist
					   1 = beta exists

	order		ord		: see above
	anf_elt		alpha		: see above (unit in ord !!)
	m		integer_small	: see above
	anf_elt		beta		: if bv gets 1 beta is returned here	

	Careful! This routine does not care for any denominators (i.e.
	the denominator of alpha must be 1).
 
History:
 
	93-02-26 KW	new return values
			precision check
	93-02-08 KW	minor changes
	92-12-04 KW	special for units
	92-09-04 KW	written (anf_elt_is_mth_power)
 
******************************************************************************/
{
	block_declarations;

	t_handle	C,R;
	integer_small	bv,i,j,n,r1,r2,r12;
	lat_elt		lelt;
	lattice		lat,llllat;
	lat_enum_env	env;
	matrix		mat,mrb,trans,inv_trans;
	t_comp		ctemp;
	t_real		tempu,tempv,tempw,tempx,tempz;
	vector		lambda;

	n = order_abs_degree(ord);
	R = order_reals(ord);

	r1 = order_r1(ord);
	r2 = order_r2(ord);
	r12 = r1+r2;
	mrb = order_basis_real(ord);
	alpha = (anf_elt_is_con(alpha)) ? anf_elt_incref(alpha) : anf_elt_con(ord,alpha); 

/*
**	Let w1,...,wn be a Z-basis of ord.
**	All x = \sum_{i=1}^n x_i*w_i in ord satisfying x^m = alpha satisfy
**	T2L(x) = n where
**	
**	(a) T2L = \sum_{i=1}^n \lambda_j^{-2} |\sum_{i=1}^n x_i*w_i^{(j)}|^2
**
**	(b) \lambda_j = |alpha^(j)|^{1/m} (1<=j<=n).
**
**	Let's enumerate.........
*/	
	C = comp_create(real_dec_prec(R));

/*
**	Set lambda
*/
	tempz = ring_one(R);
	lambda = vec_new(n);
	for (i=1;i<=r12;i++)
	{
		ctemp = anf_elt_ith_con(ord,alpha,i);
		tempu = comp_norm(C,ctemp);
		tempv = real_sqrt(R,tempu);
		tempw = real_root(R,tempv,p);
		vec_entry(lambda,i) = real_inverse(R,tempw);
		tempx = tempz;
		tempz = (i <= r1) ? real_mult(R,tempz,tempv) : real_mult(R,tempz,tempu);
		comp_elt_delete(C,&ctemp);
		real_delete(&tempu);
		real_delete(&tempv);
		real_delete(&tempw);
		real_delete(&tempx);
	}
	for (i=r1+1;i<=r12;i++)
	{
		vec_entry(lambda,i+r2) = real_incref(vec_entry(lambda,i));
	}

/*
**	Check accurancy
*/
	tempu = ring_one(R);
	tempv = real_subtract(R,tempz,tempu);
	tempx = conv_double_to_real(R,0.1);
	bv = real_zero_eps(R,tempv,tempx);
	real_delete(&tempu);
	real_delete(&tempv);
	real_delete(&tempx);
	real_delete(&tempz);
	if (!bv)
	{
		if (anf_print_level > 0) printf("precision problem in anf_elt_is_mth_power (#1).\n");
		anf_elt_delete(ord,&alpha);
		vec_delete(R,&lambda);
		ring_delete(&C);
		return(-1);
	}

/*
**	Set modul basis
*/
	mat = mat_new(n,n);
	for (i=1;i<=n;i++)	
	{		
		for (j=1;j<=n;j++)
		{
			mat_elt(mat,i,j) = real_mult(R,vec_entry(lambda,i),mat_elt(mrb,i,j));
		}
	}

	lat = lat_create();
	lat_rank(lat) = n;
	lat_basis(lat) = mat;
	lat_basis_ring(lat) = ring_incref(R);
	llllat = lat_lll_reduce(lat,&trans,&inv_trans);
	lat_chol_ring(llllat) = ring_incref(R);
	env = lat_enum_create(llllat);
	lat_enum_request_set_next(env);
	lat_enum_precision_set_long(env);
	lat_enum_strategy_set_up(env);
/*
**	We look for elements with T2L(x) <= n+1/100
*/
	tempu = conv_int_to_real(R,n);
	tempv = conv_double_to_real(R,0.01);
	lat_enum_ubound(env) = real_add(R,tempu,tempv);
	real_delete(&tempu);
	real_delete(&tempv);
	bv = 0;
	if (lat_enum(llllat,env))
	{
		lelt = lat_elt_move(llllat,lat_enum_act_coefs(env),trans);
		*beta = lat_elt_to_anf_elt(lat,lelt,ord);
		bv = 1;
	}
/*
**	Did we really find an unit ? (precision)
*/
	if (bv && !anf_elt_is_unit(ord,*beta))
	{
		if (anf_print_level > 0) printf("precision problem in anf_elt_is_mth_power (#2).\n");
		anf_elt_delete(ord,beta);
		bv = -1;
	}

	mat_delref(structure_z,&trans);
	mat_delref(structure_z,&inv_trans);
	lat_enum_delete(lat,&env);
	lat_delete(&lat);
	lat_delete(&llllat);
	vec_delete(R,&lambda);
	anf_elt_delete(ord,&alpha);
	ring_delete(&C);

	return(bv);
}
