/*******************************************************************************
  order_p_radical.c
********************************************************************************/

         
#include <stdio.h>             
#include "kant.h"
#include "anf.h"
#include "mat.h"



anf_ideal
order_p_radical WITH_2_ARGS (order,        act_ord,
                                 integer_big,  prime)
/*******************************************************************************

Description:                               

                                                                     
   This function calculates the p-radical for a given order and a      
   given prime - number. The p-radical will be represented by a Z-basis.
   The calculation is done by calculating the kernel of a map in Z/pZ.
   The map is defined by :
                                        power                    
     phi : R/(pR) ---> NR(pR)   x |--> x                         
                                                                 
                         (power-1)                         power  
   where we assume : prime            < rel_degree <= prime        
  
   For more  information see :                                         
     Pohst/Zassenhaus : "Algorithmic algebraic number theory"  page 115
                                                                     

Calling sequence:                                      
                                                                     
                  order          act_ord;                            
                  integer_big    prime;                              
                  anf_ideal      ideal;                              
      

      ideal = order_p_radical (act_ord,prime);                     
     
                                                                     
   The return value will be an ideal A satisfying :                    
                                                                     
                  A = NR (pR)    ; p = prime   R = act_ord           
                                                                     
 The parameters are :                                                    
                                                                     
   1) act_ord, prime : relative to these parameters the p-radical    
                       will be computed. You have to make shure, that
                       the order act_ord is given by a transformation
                       matrix relative to the equation order.        
                       The number prime is just an integer_big.      
                                                                     
                                                                     
History:  
                                                               
       JS 92-02-10         : renaming into order_p_radical
       MD 91-11-23         : first version                
                                                    
********************************************************************************/
{                              
     block_declarations;         
     order            Z;
                     

     anf_ideal        p_radical;

     matrix           map_matrix,kernel;
     matrix           radical;   
     matrix           mat1;

     integer_small    rel_degree;
     integer_small    i,k,l;
     integer_big      power;
     anf_elt          alpha,beta; 
     t_logical          ok;
                   

/* The ideal NR (pR) is given (essantially) by the kernel            		*/
/* of the map :                                                      		*/
/*                                        potenz                     		*/
/*     phi : R/(pR) ---> NR(pR)   x |--> x                           		*/
/*                                                                   		*/
/* where we assume :                                                            */
/*                                                                   		*/
/*                    (potenz-1)                      potenz    		*/
/*  (1)          prime           < rel_degree <= prime            		*/
/*                                                                   		*/


   Z          = order_coef_order (act_ord);
   rel_degree = order_rel_degree (act_ord);        

/* the value of power will satisfy condition 1                                  */
   power     = integer_prime_power (prime,rel_degree);                  

/* Will want to determine how phi is working on the basis elements of           */
/* the order act_ord (=R)                                                       */
   map_matrix =   mat_ring_create_id (Z,rel_degree);  

/* Now we are going to calculate the images  of the basis - elements            */
                                         
   for (i=1;i<=rel_degree;i++)                                                                     
   {                                                          
/* We are taking one basis - element and calculate it's image under phi.        */
      alpha =  mat_order_col_to_anf_elt (act_ord,map_matrix,i); 
      beta  =  anf_elt_power_mod (act_ord,prime,alpha,power);  

/* We are now replacing the basis - element by his image                        */
      anf_elt_to_mat_order_col (act_ord,beta,map_matrix,i);     

      anf_elt_delete (act_ord,&alpha); 
      anf_elt_delete (act_ord,&beta);
   }                     
    integer_delref (power);

    if (anf_print_level > 3) 
    {    
      puts ("Matrix der Abb. :");
      mat_anf_write (Z,map_matrix);
      mat_z_maple_out (Z,map_matrix);
    }
                                                       

/* Now calculate the kernel of phi.                                             */
/* The map phi is represented by the images of phi on a certain basis.          */
/* The kernel is calculated over Z/pZ.                                          */
    mat1 = mat_zm_kernel (prime,map_matrix);           

    mat_delref (Z,&map_matrix);
    mat_zm_to_z (&mat1);
                                                   
/* The function mat_zm_kernel returns a matrix with the dimension :             */
/* (degree x dimension) where dimension is the dimension of the kernel.         */
/* THIS IS NOT A SQUARE MATRIX.                                                 */
/* Since we need a square matirx for our later computations we embedd this non  */
/* square matrix into a sqaure matrix by adding some zero - columns.            */
    kernel = mat_ring_make_square_mat (Z,mat1); 

    mat_delref (Z,&mat1);

    if (anf_print_level > 3) 
    {    
      puts ("kernel");
      mat_anf_write (Z,kernel);                
    }

/* As you can see the elements "prime*a"  (a in R) are also in the kernel of    */
/* phi. So we reduce the generating - system of phi's kernel hermite i order    */
/* to get a basis.                                                              */
/* A generating system for phi's kernel is :                                    */
/*                                                                              */
/*                  {col(mat1,i)|1<=i<=n} U {prime*a(i)|1<=i<=n}                */
/*                                                                              */
/* with :                    mat1  : the calcuated kernel.                      */
/*                           a(i)  : the i-th. basis element of R               */

    radical = mat_ring_col_hnf_mod (Z,kernel,prime);        
    mat_delref (Z,&kernel);
  
    if (anf_print_level > 3) 
    {    
      puts ("mod. HNF");                 
      mat_anf_write (Z,radical);                    
    }
    
/* We now have all necessary information. Create the ideal.                     */
    p_radical = anf_ideal_alloc ();
                                                                                  
/* Set some flags                                                               */
    anf_ideal_tran (p_radical)     = radical;                                   
    anf_ideal_tran_den (p_radical) =  1;
    anf_ideal_tran_hnf (p_radical) = -1;
                                                
    return p_radical;

}
