#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_to_frac	WITH_2_ARGS(
	mp_float,	x,
	mp_float,	y
)
/*
Sets y to the fractional part of mp x and returns y.
i.e. y = x - integer part of x (truncated towards 0).
The current rounding type is irrelevant.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_expt_type	expt;


    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+mp_to_frac {\n");
    DEBUG_1("x = ", xp);

    mp_check_2("mp_to_frac", xp, yp);

    mp_copy_ptr(xp, yp);

    if (!mp_is_zero(yp))
    {
	expt = mp_expt(xp);

	/*
	If the exponent is <= 0 (i.e. int_abs(x) < 1), then the result is x.
	*/

	if (expt > 0)
	{
	    /*
	    Check for zero fractional part.
	    */

	    if (expt >= mp_t(xp))
		mp_set_sign(yp, 0);

	    else
	    {
		/*
		Here 0 < expt < t.  Clear integer part and normalize.
		*/

		mp_set_digits_zero(mp_digit_ptr(yp, 0), expt);

		mp_nzr(mp_sign(yp), expt, yp, mp_digit_ptr(yp, 0), 0);
	    }
	}
    }

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return y;
}



mp_int
mp_to_expt_frac		WITH_3_ARGS(
	mp_float,	x,
	mp_base_type,	base,
	mp_float,	y
)
/*
Given mp x, sets mp y and returns integer n such that x = base^n * y and
1 <= int_abs(y) < base.  It is assumed that x is not so large or small that
n overflows.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_base_type	b;
    mp_int		n, j;
    mp_sign_type	y_sign;


    /*
    Should base be allowed to be > 16?
    */

    if (base < 2 || base > 16)
	mp_error("mp_to_expt_frac: illegal base");


    if (mp_is_zero(xp))
    {
	mp_set_sign(yp, 0);
	return 0;
    }


    mp_copy_ptr(xp, yp);

    n = 0;
    b = mp_b(xp);

    mp_change_up();


#define	fix()		if (mp_has_changed()) yp = mp_ptr(y);


    /*
    Loop up to 100 times (usually one is sufficient).
    */

    for (j = 0; j < 100; j++)
    {
	/*
	Estimate log(int_abs(y)) to given base.
	*/

	mp_int	guess = mp_change_base(base, b, mp_expt(yp) - 1) +
			mp_change_base(base, mp_digit(yp, 0), 1);


	/*
	The following avoids the possibility of overflow below.
	*/

	if (j == 0 && int_abs(guess) > MAX_EXPT / 4)
	    guess >>= 1;

	
	/*
	Leave loop if guess is small.
	*/

	if (int_abs(guess) <= 3)
	    break;

	
	/*
	Divide by base^guess, taking care for directed roundings.
	*/

	mp_scale(y, base, -guess);

	fix();
	if (mp_is_zero(yp))
	    j = 100;

	n += guess;
    }

    if (j == 100)
	mp_bug("mp_to_expt_frac: error: maybe exponent too large");


    while (mp_expt(yp) > 0)
    {
	n++;
	mp_div_int_eq(y, base);
	fix();
    }

    do
    {
	n--;
	mp_mul_int_eq(y, base);
	fix();
    } while (mp_expt(yp) <= 0);


    /*
    Check for the possibility that the rounding up was to base.
    */

    y_sign = mp_sign(yp);
    mp_set_sign(yp, 1);

    if (mp_cmp_int(y, base) >= 0)
    {
	/*
	It was, so set y to 1 and add one to exponent.
	*/

	mp_int_to_mp(1, y);
	n++;
    }

    /*
    Restore sign of y.
    */

    fix();
    mp_set_sign(yp, y_sign);

    mp_change_down();

    return n;
}
