#include "defs.h"
#include "poly_internals.h"
#include "ipoly_faclst.h"
#include "anf.h"

/*
 
This file contains:
 
order_prime_factorize         (generic)
order_prime_factorize_power
order_prime_factorize_trans
 
*/
 
handle
order_prime_factorize_poly WITH_2_ARGS(
			             	order,		ord,
					integer_big,	p
)
/*******************************************************************************
    
Description:
 
	Factorizes the principal ideal of p (which is a prime) into a power 
	product of prime ideals (generic routine).
 
History:
 
        91-10-01 JS	first version
 
*******************************************************************************/
{
	trbl_declarations;         
	
 	handle		order_prime_factorize_power();
 	handle		order_prime_factorize_trans();
 
	handle		id_facs;

 
	if (!integer_is_single(p))
	error_internal("Berlekamp etc. not yet implemented for long p.");
 
	order_must_be_over_z(ord);
 
 
	if (order_basis_is_power(ord))
	{
		id_facs = order_prime_factorize_power(ord, p);
	}
	else if (order_basis_is_rel(ord))
	{
		id_facs = order_prime_factorize_trans(ord, p);
	}
	else
	{	
		error_internal("order_prime_factorize: Unknown situation.");
	}
 	
	if (anf_print_level > 2)
	{
		printf("Factorization of %d: \n", p);
		anf_ideal_faclst_write(ord, id_facs);
	}
	
	return id_facs;
}
                                      
 
  
 
 

handle
order_prime_factorize_power WITH_2_ARGS(
	order,		ord,
	integer_big,	p
)
/*******************************************************************************
 
Description:
 
	Factorizes the principal ideal of p (which is a prime) into a
	power product of prime ideals.
 
	The order must be equation order over Z.
 
	At the moment the Header file for polynomial faclists is also used for the
	faclist of ideals. This should be abolished in future.
  

History:
 
        92-05-12 JS	... there was an anf_elt_incref too many!
        92-03-19 JS	minimum, degree 
        91-10-01 JS	first version
 
*******************************************************************************/
{
	trbl_declarations;         
 
	integer_small	facs, i, power, deg;
	poly_poly	mpoly, pol;
	handle		poly_facs, id_facs, polh;
	anf_ideal	id, idsq;
	anf_elt		alpha, temp;

              
        deg = order_rel_degree(ord);
 
/*
    factorization of the minimal polynomial over Zp 
*/                            
	mpoly = modpoly_hom(p, order_poly(ord));
	poly_facs = modupoly_fact(p, mpoly);
 
	facs = ipoly_faclst_len(poly_facs);
 
/*
    Our ideal factor list
*/
	id_facs = ipoly_faclst_alloc(facs);
	ipoly_faclst_len_put(id_facs, facs);
 
/*
    constructing two-element presentations for the prime ideals 
*/
 
/*
    first the trivial case (prime is inert)
*/
	if (facs==1 && ipoly_faclst_power(poly_facs, 0) == 1)
	{
		id 				= anf_ideal_alloc();
		anf_ideal_gen1(id)  		= integer_incref(p);
		anf_ideal_gen2(id)  		= integer_incref(p);
		anf_ideal_gen_g(id)		= integer_incref(p);
                anf_ideal_min(id)   		= integer_incref(p);
                anf_ideal_degree(id)		= deg;
		ipoly_faclst_factor(id_facs, 0) = id;
	    	ipoly_faclst_power(id_facs, 0) 	= 1;
	}
 
	else
 
	{
		for (i=0; i<facs; ++i)
 
		{
			power = ipoly_faclst_power(poly_facs, i);
		    	ipoly_faclst_power(id_facs, i) = power;
	 
		    	id = anf_ideal_alloc();
	 
		    	anf_ideal_gen1(id) = integer_incref(p);
/*
    Now we construct a normal presentation. If the ramification index
    (= the exponent) is larger than 1 we already have what we want.
    Unfortunately this only happens to few primes (those which divide
    the order discriminant).
    For the others we have to check whether alpha lies in the square of
    the ideal. If not, we are done, in the other case we have to add p
    to alpha.
*/              
                        pol   		    = ipoly_faclst_factor(poly_facs, i);
			polh                = poly_poly_to_handle(pol);
		    	anf_ideal_gen_g(id) = integer_incref(p);
		    	alpha 	            = poly_to_anf_elt(ord, pol);
		    	anf_ideal_gen2(id)  = alpha;
 
	    		if (power == 1)
		    	{	
				idsq = anf_ideal_2_mult(ord, id, id);
				if (anf_elt_in_ideal(ord, alpha, idsq))
				{
					anf_ideal_gen2(id) = 
					anf_add(ord, alpha, p);
					anf_elt_delete(ord, &alpha);
				}
				anf_ideal_delete(ord, &idsq);
			} 
 
                        anf_ideal_min(id) 	= integer_incref(p);
                        anf_ideal_degree(id) 	= poly_expt(polh, poly_nterms(polh)-1);
	 
			ipoly_faclst_factor(id_facs, i) = id;
		}     
	}
 
	ipoly_delref(mpoly);
	ipoly_faclst_delete(&poly_facs);	
	
	return id_facs;
}
 
 
 
 
 
handle
order_prime_factorize_trans WITH_2_ARGS(
	order,		ord,
	integer_big,	p
)
/*******************************************************************************
      
Factorizes the principal ideal of p (which is a prime) into a
power product of prime ideals.
 
The order must be given via a transformation matrix referring to a suborder.
 
At the moment the Header file for polynomial faclists is also used for the
faclist of ideals. This should be abolished in future.
 
 
History:
 
        92-07-28 MD     calling rec. order_prime_factorize_poly instead of
                        order_prime_factorize
        91-10-01 JS	first version
 
*******************************************************************************/
{
	trbl_declarations;         
 
	order		subord;
	integer_small	facs, i;
	handle		id_facs;
	anf_ideal	id;
	anf_elt		temp;
	integer_big     q, r;

 
	subord = order_suborder(ord);
 
/*
    We can't deal with index divisors yet
*/
                             
	integer_quot_rem(order_index(ord), p, &q, &r);
	if (!r) error_internal("Do not know how to factorize index divisor.");
	integer_delref(q);
	integer_delref(r);
 
/*
   Now we factorize p in the suborder and move the ideals up.
   I think they still will be in normal presentation.
*/
	id_facs = order_prime_factorize_poly (subord, p);
 
	facs = ipoly_faclst_len(id_facs);
 
	for (i=0; i<facs; ++i)
	{
		id 			= ipoly_faclst_factor(id_facs, i);
		temp 			= anf_ideal_gen2(id);
		anf_ideal_gen2(id) 	= anf_elt_move(subord, temp, ord);
 
		anf_elt_delete(subord, &temp);
	}
	
	return id_facs;
}


