/******************************************************************************
  anf_ideal_norm.c
******************************************************************************/
 
#include "kant.h"
#include "mat.h"
#include "anf.h"

void
anf_ideal_norm WITH_4_ARGS(
	order,		ord,
	anf_ideal,	id,
	integer_big *,	norm,
	integer_big *,	den

)
/*******************************************************************************
 
 
Description:
 
	Computes the norm of the ideal given. If the ideal is given only in
	2-element presentation it is first converted to Z-basis presentation.
	The norm is then computed as the determinant of the transformation
	matrix.
 
	The coefficient order must be Z.
 
   
Calling sequence:
 
	anf_ideal_norm(ord, id, &norm, &den);
                                              
      	order  	        ord      = t_handle of order 
      	anf_ideal       id       = t_handle of ideal
	integer_big	norm	 = nominator of norm
	integer_big	den	 = denominator of norm
 
 
History:
 
	92-05-13 JS	... in this case also degree must be known!
	92-05-05 JS	case id is prime
	91-09-01 JS	first version
 
*******************************************************************************/
{       
	block_declarations;
 
	integer_small	deg;
        order		ordcoef;
	integer_big	temp, prod;
	matrix		mat;
	integer_small	i;
 
	order_must_be_over_z(ord);
 
        ordcoef = order_coef_order(ord);
	deg 	= order_rel_degree(ord);
  
/*
    easy case: ideal is prime ideal
*/
	if (anf_ideal_min_known(id) && anf_ideal_degree(id) > 0)
        {
                *den  = 1;
                *norm = integer_power(anf_ideal_min(id), anf_ideal_degree(id));
                return;
        }
         
/*
    General case. We need a Z-basis presentation.
*/
	if(!anf_ideal_is_z(id)) anf_ideal_2_z(ord, id);
 
/*
    Now we have a transformation matrix the determinant of which we 
    have to evaluate. If the transition matrix is given in Hermite
    normal form we simply have to multiply the diagonal entries.
*/
        
	mat = anf_ideal_tran(id);
 
	if(anf_ideal_tran_hnf(id) != 0)
	{
		temp = mat_ring_det(ordcoef, mat);
		*norm = integer_abs(temp);
		integer_delref(temp);
	}
	else
	{
		prod = 1;
		for (i=1; i<=deg; ++i)
		{
			temp = prod;
			prod = integer_mult(temp, mat_elt(mat, i, i));
			integer_delref(temp);
		}
		*norm = prod;
	}
 
	*den = integer_power(anf_ideal_tran_den(id), deg);
 
	return;
}
