/******************************************************************************
  mat_real_echelon.c
******************************************************************************/
#include "kant.h"

t_void
mat_real_echelon WITH_6_ARGS(
	t_handle,	rrng,
	matrix,		mat,
	matrix*,	echo,
	matrix*,	ltrans,
	t_logical,	cflag,
	matrix*,	rtrans
)

/******************************************************************************
 
Description:	Compute the echelon form of a real matrix mat and the 
		transformation matrixes from mat to its echelon form

		Note:
		(1) if mat is a mxn-matrix its rank must be min(m,n).
		(2) echelon form means: e(i,j) = 0 for i != j and e(i,i) > 0.
		(3) echelon = ltrans*mat*rtrans
		(4) rtrans is only used to swap columns ==> entries are 0 or 1.
		   
Calling sequence:
 
	mat_real_echelon(rrng,mat,&echo,&ltrans,cflag,&rtrans)

	t_handle	rrng	: the real ring of coefficients
	matrix		mat	: mxn-matrix to be echelonized
	matrix		echo	: the echelon form 
	matrix		ltrans	: left transformation matrix (mxm)
	t_logical	cflag	: TRUE		column swaps allowed
				  FALSE	        no column swaps allowed
	matrix		rtrans	: right transformation matrix (nxn)
				  (undefined, if cflag = FALSE)

History:
 
	92-06-25 KW	written
 
******************************************************************************/
{
	block_declarations;

	integer_small	i,ii,iii,j,jj,jjj,ki,kj;
	integer_small	col,row;
	t_real		tempu,tempv,tempw,tempx;

	row = mat_row(mat);
	col = mat_col(mat);

	*echo	= mat_ring_copy(rrng,mat);
	*ltrans = mat_ring_create_id(rrng,row);
	if (cflag) *rtrans = mat_ring_create_id(rrng,col);

	tempx = ring_minus1(rrng);
	ki = integer_min(col,row);
	for(i=1;i<=ki;i++)
	{
/*
**		Determine pivot element
*/
		tempu = ring_zero(rrng);
		for(ii=i;ii<=ki;ii++)
		{
			kj = (cflag) ? col : i;
			for (jj=i;jj<=kj;jj++)
			{
				tempv = real_abs(rrng,mat_elt(*echo,ii,jj));
				if (real_compare(rrng,tempu,tempv) == -1)
				{
					iii = ii;
					jjj = jj;
					real_delete(&tempu);
					tempu = real_incref(tempv);
				}
				real_delete(&tempv);
			}
		}
		real_delete(&tempu);

/*
**		A(i,i) <-- pivot element (= A(iii,jjj))
*/
		if (cflag)
		{
			mat_ring_col_swap(rrng,*echo,i,jjj,1,row);
			mat_ring_col_swap(rrng,*rtrans,i,jjj,1,col);
		}
		mat_ring_row_swap(rrng,*echo,i,iii,i,col);
		mat_ring_row_swap(rrng,*ltrans,i,iii,1,row);

/*
**		A(i,i) > 0
*/
		if (real_sign(rrng,mat_elt(*echo,i,i)) == -1)
		{
			mat_ring_row_mult(rrng,*echo,i,tempx,1,col);
			mat_ring_row_mult(rrng,*ltrans,i,tempx,1,row);
		}

/*
**		A(j,i) = 0  for 1<=j<=k,i!=j
*/
		tempu = real_inverse(rrng,mat_elt(*echo,i,i));
		for (j=1;j<=row;j++)
		{
			if (j == i) continue;
			tempv = real_mult(rrng,mat_elt(*echo,j,i),tempu);
			tempw = real_negate(rrng,tempv);
			mat_ring_row_add(rrng,*echo,i,j,tempw,i,col);
			mat_ring_row_add(rrng,*ltrans,i,j,tempw,1,row);
			real_delete(&tempv);
			real_delete(&tempw);
		}
		real_delete(&tempu);
	}
	real_delete(&tempx);

	return;
}
