#include "defs.h"
#include "integer.e"
#include "dyn_arr.h"

/*
 * Table for the number of partitions for integers up to 100.  The table comes
 * from G.E. Andrews, The Theory of Partitions (Addison-Wesley, 1976).
 */

#define KNOWN	100

static t_int partitions[101] = {
	1,	   1,	      2,	 3,	    5,	       7,	 11,
       15,	  22,	     30,	42,	   56,	      77,	101,
      135,	 176,	    231,       297,	  385,	     490,	627,
      792,	1002,	   1255,      1575,	 1958,	    2436,      3010,
     3718,	4565,	   5604,      6842,	 8349,	   10143,     12310,
    14883,     17977,	  21637,     26015,	31185,	   37338,     44583,
    53174,     63261,	  75175,     89134,    105558,	  124754,    147273,
   173525,    204226,	 239943,    281589,    329931,	  386155,    451276,
   526823,    614154,	 715220,    831820,    966467,	 1121505,   1300156,
  1505499,   1741630,	2012558,   2323520,   2679689,	 3087735,   3554345,
  4087968,   4697205,	5392783,   6185689,   7089500,	 8118264,   9289091,
 10619863,  12132164,  13848650,  15796476,  18004327,	20506255,  23338469,
 26543660,  30167357,  34262962,  38887673,  44108109,	49995925,  56634173,
 64112359,  72533807,  82010177,  92669720, 104651419, 118114304, 133230930,
150198136, 169229875, 190569292  };

Private t_int	calculate_partitions();

t_int
integer_num_partitions(n)
t_int	n;
{
	block_declarations;

	register t_int	i;
	t_handle		temp;
	register t_int	result;

	/*
	 * Given a single precision integer n, return the number of partitions
	 * of n.
	 * The algorithm comes from G.E. Andrews, The Theory of Partitions
	 * (Addison-Wesley, 1976).
	 *
	 * John Brownie, October 1989.
	 */

	if (!integer_is_single(n))
	{
		error_internal("Infinite precision argument in integer_num_partitions");
		return 0;
	}
	if (n < 0)
		return 0;
	if (n <= KNOWN)
		return partitions[n];
	if (n == KNOWN + 1)
		return calculate_partitions(n, 0);
	temp = dyn_arr_alloc(n - KNOWN - 1);
	for (i = KNOWN + 1; i < n; i++)
	    dyn_arr_element(temp, i - KNOWN - 1)=calculate_partitions(i, temp);
	result = calculate_partitions(n, temp);
	for (i = n - KNOWN - 2; i >= 0; i--)
		integer_delref(dyn_arr_element(temp, i));
	block_decref_delete(temp);
	return result;
}

Private t_int
calculate_partitions(n, parts)
t_int	n;
t_handle	parts;
{
	block_declarations;

	register t_int	m;
	register t_int	value;
	register t_int	m1;
	register t_int	m2;
	register t_int	val1;
	register t_int	oldval;
	register Logical	neg;

	/*
	 * Calculate the number of partitions of n using the previously
	 * calculated values in the stack parts, as well as the known values
	 * in the partitions array.
	 */

	value = 0;
	m = 1;
	for (;;)
	{
		m1 = n - m * (3 * m - 1) / 2;
		m2 = m1 - m;
		neg = m % 2 == 0;
		if (m1 > KNOWN)
			val1 = dyn_arr_element(parts, m1 - KNOWN - 1);
		else if (m1 >= 0)
			val1 = partitions[m1];
		else
			return value;
		oldval = value;
		if (neg)
			value = integer_subtract(value, val1);
		else
			value = integer_add(value, val1);
		integer_delref(oldval);
		if (m2 > KNOWN)
			val1 = dyn_arr_element(parts, m2 - KNOWN - 1);
		else if (m2 >= 0)
			val1 = partitions[m2];
		else
			return value;
		oldval = value;
		if (neg)
			value = integer_subtract(value, val1);
		else
			value = integer_add(value, val1);
		integer_delref(oldval);
		m++;
	}
}
