
#include "defs.h"
#include "integer.e"
#include "dmat.h"
#include "mat.h"
#include <math.h>
#include "conv.e"
#include "debug.h"
#include "dyn_arr.h"
#include "globals.e"

#define DEBUG_MLLL_INT_D_REDUCE 46
#undef DEBUG_FLAG
#define DEBUG_FLAG(n) v3_df[(n)]

t_void
mlll_int_d_reduce WITH_7_ARGS(
    t_handle,	zrng,
    matrix,	b_orig,
    double,	delta,
    Logical *,  dep,
    matrix *,	bh_orig,
    t_handle *,	marr,
    matrix *,   trans
)
/*
** The integer matrix b is LLL-reduced into *bh, using a varaint of the
** LLLFP algorithm from ``Lattice Basis Reduction: Improved Practical
** Algorithms and Solving Subset Sum Problems.'' by Schnorr and Euchner.
 
	92-05-06 JS    zero, dep and trans and much more...
*/
{
    block_declarations;
    double        *dmptr1;
    double        *dmptr2;
    double        *dmptr3;
    integer_big    *imptr1;
    integer_big    *imptr2;
    integer_small    k;
    integer_small    l;
    integer_small    m;
    integer_small    n;
    matrix        b;
    Logical        bh_pack;
    matrix        bh;
    t_dmat        c;
    t_dmat        bhprime;
    t_dmat        bhprimesn;
    double        twotomtau;
    double        twotohtau;
    double        d1;
    double        d2;
    double        d3;
    integer_small    i1;
    integer_small    i2;
    integer_small    i3;
    integer_small    i;
    double        muprime;
    integer_big    mu;
    t_dmat        mumat;
    double        s;
    Logical        flag_c;
    Logical        flag_r;
    Logical        flag_nonzero;

#ifdef DEBUG
    if ( delta <= 0.25 || delta >= 1.00 )
    {
        error_internal( "delta not in (1/4, 1) in lll_int_d_reduce()" );
    }
#endif

    k = mat_col( b_orig ) - 1;
    n = mat_row( b_orig );

    *dep = FALSE;
 
    if ( k <= 0 )
    {
        error_internal( "Need more than zero vectors in MLLL" );
    }

    mat_create_unpkd( zrng, b_orig, b, n, k+1 );

    bh_pack = mat_result_pkd( zrng, *bh_orig );
    mat_alloc_result_unpkd( bh_pack, *bh_orig, bh, n, k+1 );
    mat_ring_copy_sub( zrng, b, &bh );

    mat_free_unpkd( b_orig, b );

    *trans = mat_new( k+1, k+1 );
    mat_ring_create_id_sub( zrng, *trans );

    bhprime = dmat_new( n, k+1 );
    c = dmat_new( 1, k+1 );
    bhprimesn = dmat_new( 1, k+1 );
    mumat = dmat_new( k+1, k+1 );

    twotohtau = pow( (double) 2, (double) 32 / 2 );
    twotomtau = pow( (double) 2, (double) 32 );

    dmptr1 = dmat_eltptr( bhprime );
    imptr1 = mat_elt0_ptr( bh );

    for ( i=1; i<=(k+1)*n; ++i )
    {
        dmptr1[i] = conv_int_to_double( imptr1[i] );
    }

    m = 2;

    while ( TRUE )
    {
        if ( m == 2 )
        {
            dmptr1 = dmat_eltptr( bhprime );

            d1 = dmptr1[1] * dmptr1[1];
            for ( i=k+2; i<=(k+1)*n; i+=(k+1) )
            {
                d1 += ( dmptr1[i] * dmptr1[i] );
            }
            dmat_entry( bhprimesn, 1 ) = dmat_entry( c, 1 ) = d1;
        }

        dmptr1 = dmat_eltptr( bhprime );

        d1 = dmptr1[m] * dmptr1[m];
        for ( i=(k+1)+m; i<=(k+1)*n; i+=(k+1) )
        {
            d1 += ( dmptr1[i] * dmptr1[i] );
        }
        dmat_entry( bhprimesn, m ) = dmat_entry( c, m ) = d1;

        for ( l=1; l<m; ++l )
        {
            dmptr1 = dmat_eltptr( bhprime ) + m;
            dmptr2 = dmat_eltptr( bhprime ) + l;

            d1 = dmptr1[0] * dmptr2[0];
            for ( i=k+1; i<(k+1)*n; i+=(k+1) )
            {
                d1 += ( dmptr1[i] * dmptr2[i] );
            }

            d2 = d1 * d1;

            if ( d2 < twotomtau * dmat_entry( bhprimesn, m ) * dmat_entry( bhprimesn, l ))
            {
                i1 = integer_mult( mat_entry( bh, l ), mat_entry( bh, m ));
                for ( i=k+1; i<(k+1)*n; i+=(k+1) )
                {
                    i2 = integer_mult( mat_entry( bh, l+i ), mat_entry( bh, m+i ));
                    i3 = integer_add( i1, i2 );
                    integer_delref( i2 );
                    integer_delref( i1 );
                    i1 = i3;
                }
    
                s = conv_int_to_double( i1 );
                integer_delref( i1 );
            }
            else
            {
                s = d1;
            }

            dmptr1 = dmat_eltptr( mumat ) + (m-1)*(k+1);
            dmptr2 = dmat_eltptr( mumat ) + (l-1)*(k+1);
            dmptr3 = dmat_eltptr( c );

            for ( i=1; i<l; i++ )
            {
                s -= ( dmptr2[ i ] * dmptr1[ i ] * dmptr3[ i ] );
            }
            d1 = dmptr1[ l ] = s / dmptr3[ l ];

            dmptr3[ m ] -= ( d1 * s );
        }

#ifdef DEBUG
        if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
        {
            cay_print( "Recalculated for m = %d\n", m );
            cay_print( "bh =\n" );
            print_matrix( structure_z, bh, 0 );
            cay_print( "mu =\n" );
            print_dmat( mumat );
            cay_print( "bhprime =\n" );
            print_dmat( bhprime );
            cay_print( "bhprimesn =\n" );
            print_dmat( bhprimesn );
            cay_print( "c =\n" );
            print_dmat( c );
            cay_print( "\n" );
        }
#endif

/*
** STEP 3
*/

        flag_c = FALSE;
        flag_r = FALSE;

        for ( l=m-1; l>=1; --l )
        {
            muprime = dmat_entry( mumat, (m-1)*(k+1) + l );
#ifdef DEBUG
            if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
            {
                cay_print( "muprime = %f\n", muprime );
            }
#endif
            if ( muprime > 0.5 || muprime < -0.5 )
            {
                flag_r = TRUE;

                if ( muprime > twotohtau || muprime < -twotohtau )
                {
#ifdef DEBUG
                    if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
                    {
                        cay_print( "muprime = %f, twotohtau = %f, set F_c true\n", muprime, twotohtau );
                    }
#endif

                    flag_c = TRUE;
                }

                mu = conv_double_to_int_round( muprime );
                muprime = conv_int_to_double( mu );
#ifdef DEBUG
                if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
                {
                    cay_print( "subtracting %d by col %d from col %d\n", mu, l, m );
                }
#endif

                dmptr1 = dmat_eltptr( mumat ) + (l-1)*(k+1);
                dmptr2 = dmat_eltptr( mumat ) + (m-1)*(k+1);

                for ( i=1; i<l; ++i )
                {
                    dmptr2[ i ] -= ( muprime * dmptr1[ i ] );
                }

                dmat_entry( mumat, (m-1)*(k+1) + l ) -= muprime;

                flag_nonzero = FALSE;
                for ( i=0; i<(k+1)*n; i+=(k+1) )
                {
                    i1 = mat_entry( bh, i+m );
                    i2 = integer_mult( mat_entry( bh, i+l ), mu );
                    flag_nonzero = ( mat_entry( bh, i+m ) = integer_subtract( i1, i2 )) || flag_nonzero;
                    integer_delref( i1 );
                    integer_delref( i2 );
                }

                for ( i=0; i<(k+1)*(k+1); i+=(k+1) )
                {
                    i1 = mat_entry( *trans, i+m );
                    i2 = integer_mult( mat_entry( *trans, i+l ), mu );
                    mat_entry( *trans, i+m ) = integer_subtract( i1, i2 );
   /* MJ deleted due to a mail from Wieb                 integer_delref( i1 );  */
                    integer_delref( i2 );
                }

                integer_delref( mu );
#ifdef DEBUG
                if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
                {
                    cay_print( "After column addition\n" );
                    cay_print( "bh =\n" );
                    print_matrix( structure_z, bh, 0 );
                    cay_print( "mu =\n" );
                    print_dmat( mumat );
                    cay_print( "bhprime =\n" );
                    print_dmat( bhprime );
                    cay_print( "bhprimesn =\n" );
                    print_dmat( bhprimesn );
                    cay_print( "c =\n" );
                    print_dmat( c );
                    cay_print( "\n" );
                }
#endif

                if ( ! flag_nonzero )
                {
		    /*
		    ** Zero vector found.
		    */
                       
                    *dep = TRUE;
                       
/*		    for ( i=m; i<=k; ++i )
		    {
			for ( p=1; p<=n; ++p )
			{
			    mat_elt( bh, p, i ) = mat_elt( bh, p, i+1 );
			}
		    }

		    for ( p=1; p<=n; ++p )
		    {
			mat_elt( bh, p, k+1 ) = 0;
		    }
*/
		    if ((*marr) == 0 )
		    {
			*marr = dyn_arr_alloc( k+1 );
		    }

		    for ( i=1; i<=k+1; ++i )
		    {
			dyn_arr_element( *marr, i-1 ) = integer_incref( mat_elt( *trans, i, m ));
		    }
		    dyn_arr_curr_length( *marr ) = k+1;
 
/* JS 11.05.92 */
                           
                    for (i=m; i<=k; ++i)
                    {
			mat_ring_col_swap(zrng, bh, i, i+1, 1, n);
			mat_ring_col_swap(zrng, *trans, i, i+1, 1, k+1);
                    }

                    /*
                    ** Clean up.
                    */

                    mat_create_result( zrng, bh_pack, *bh_orig, bh );

                    dmat_delete( &c );
                    dmat_delete( &bhprime );
                    dmat_delete( &bhprimesn );
                    dmat_delete( &mumat );

                    return;
                }
            }
        }

        if ( flag_r )
        {
#ifdef DEBUG
            if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
            {
                cay_print( "F_r true, recalculating\n" );
            }
#endif

            imptr1 = mat_elt0_ptr( bh );
            dmptr1 = dmat_eltptr( bhprime );
            d1 = 0.0;
            for ( i=m; i<=(k+1)*n; i+=(k+1) )
            {
                d2 = dmptr1[ i ] = conv_int_to_double( imptr1[ i ] );
                d1 += d2 * d2;
            }
            dmat_entry( bhprimesn, m ) = d1;
        }

        if ( flag_c )
        {
#ifdef DEBUG
            if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
            {
                cay_print( "F_c true\n" );
            }
#endif

            if ( m > 2 )
            {
#ifdef DEBUG
                if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
                {
                    cay_print( "decreasing m from %d to %d\n", m, m-1 );
                }
#endif
                m--;
            }
            else
            {
#ifdef DEBUG
                if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
                {
                    cay_print( "decreasing m impossible, keep at 2\n" );
                }
#endif
            }

            continue;
        }

/*
** STEP 4 */

        d1 = dmat_entry( mumat, (m-1)*(k+2) );
        d2 = dmat_entry( c, m-1 );
        d3 = dmat_entry( c, m );

        if ( d3 < (delta - d1 * d1) * d2 )
        {
#ifdef DEBUG
            if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
            {
                cay_print( "swapping %d and %d\n", m, m-1 );
            }
#endif

            imptr1 = mat_elt0_ptr( bh ) + m;
            imptr2 = mat_elt0_ptr( *trans ) + m;
            dmptr1 = dmat_eltptr( bhprime ) + m;

            for ( i=0; i<(k+1)*n; i+=(k+1) )
            {
                i1 = imptr1[ i-1 ];
                imptr1[ i-1 ] = imptr1[ i ];
                imptr1[ i ] = i1;

                d1 = dmptr1[ i-1 ];
                dmptr1[ i-1 ] = dmptr1[ i ];
                dmptr1[ i ] = d1;
            }

            for ( i=0; i<(k+1)*(k+1); i+=(k+1) )
            {
                i1 = imptr2[ i-1 ];
                imptr2[ i-1 ] = imptr2[ i ];
                imptr2[ i ] = i1;
            }

            dmptr1 = dmat_eltptr( bhprimesn );
            d1 = dmptr1[ m ];
            dmptr1[ m ] = dmptr1[ m-1 ];
            dmptr1[ m-1 ] = d1;

#ifdef DEBUG
            if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
            {
                cay_print( "After late swap of %d and %d\n", m, m-1 );
                cay_print( "bh =\n" );
                print_matrix( structure_z, bh, 0 );
                cay_print( "mu =\n" );
                print_dmat( mumat );
                cay_print( "bhprime =\n" );
                print_dmat( bhprime );
                cay_print( "bhprimesn =\n" );
                print_dmat( bhprimesn );
                cay_print( "c =\n" );
                print_dmat( c );
                cay_print( "\n" );
            }
#endif

            if ( m > 2 )
            {
#ifdef DEBUG
                if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
                {
                    cay_print( "decreasing m from %d to %d\n", m, m-1 );
                }
#endif

                m--;
            }
#ifdef DEBUG
            else
            {
                if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
                {
                    cay_print( "decreasing m impossible, keep m at 2\n" );
                }
            }
#endif
        }
        else
        {
#ifdef DEBUG
            if ( DEBUG_FLAG( DEBUG_MLLL_INT_D_REDUCE ))
            {
                cay_print( "increasing m from %d to %d\n", m, m+1 );
            }
#endif

            m++;

            if ( m > k+1 )
            {
                /*
                ** No dependency found.
                */

                if ( *marr == 0 )
                {
                    *marr = dyn_arr_alloc( k+1 );
                }

                for ( i=1; i<=k+1; ++i )
                {
                    dyn_arr_element( *marr, i-1 ) = 0;
                }
                dyn_arr_curr_length( *marr ) = k+1;

                /*
                ** Clean up.
                */

                mat_create_result( zrng, bh_pack, *bh_orig, bh );

                dmat_delete( &c );
                dmat_delete( &bhprime );
                dmat_delete( &bhprimesn );
                dmat_delete( &mumat );

                return;
            }
        }
    }
}
