#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_besj		WITH_3_ARGS(
	mp_float,	x,
	mp_int,		nu,
	mp_float,	y
)
/*
Returns y = J(nu, x), the first-kind Bessel function of order nu, for
small integer nu, and mp x and y.  Results for negative arguments are
defined by: J(-nu, x) = J(nu, -x) = (-)^nu * J(nu, x).  The method used
is Hankel's asymptotic expansion if int_abs(x) is large, the power series if
int_abs(x) is small, and the backward recurrence method otherwise.
Error could be induced by O(b^(1-t)) perturbations in x and y.
The time taken is O(t * M(t)) for fixed x and nu and increases as x and
nu increase unless x is large enough for the asymptotic series to be used.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, new_t;
    mp_sign_type	x_sign;
    mp_acc_float	temp1;
    mp_expt_type	x_expt;
    mp_int		abs_nu;


    DEBUG_BEGIN(DEBUG_BESSEL);
    DEBUG_PRINTF_1("+besj {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("nu = %d\n", nu);

    mp_check_2("mp_besj", xp, yp);

    x_sign = mp_sign(xp);

    if (x_sign == 0)
    {
	if (nu)
	    mp_set_sign(yp, 0);

	else
	    mp_int_to_mp(1, y);

	DEBUG_1("-} y = ", yp);
	DEBUG_END();
	return y;
    }


    b = mp_b(xp);
    t = mp_t(xp);

    if ((x_expt = mp_expt(xp)) >= t)
	mp_error("mp_besj: int_abs(x) too large");

    round = MP_TRUNC;
    new_t = t + 1 + mp_extra_guard_digits(1, b);

    mp_change_up();
    mp_acc_float_alloc(b, new_t, temp1);

    mp_move(x, temp1);

    abs_nu = int_abs(nu);

    /*
    Try to use Hankel's asymptotic series.
    */

    if (!mp_hank(temp1, abs_nu, temp1))
    {
	/*
	The asymptotic series is not good enough.
	*/

	mp_length	base_t;
	mp_int		k, max_k;
	mp_acc_float	temp2, temp3;
	mp_ptr_type	temp1_ptr, temp2_ptr, temp3_ptr;



	if (x_expt > 0)
	{
	    /*
	    Estimate the number of digits required to compensate for
	    cancellation.  First reduce t to the equivalent of 6 decimal places
	    (we could use single-precision real arithmetic here if trusted).
	    */

	    mp_length		t6 = mp_change_base(b, 10, 5) + 1, it, extra_t;


	    mp_acc_float_alloc_2(b, new_t, temp2, temp3);
	    temp1_ptr = mp_acc_float_ptr(temp1);


#define fix_pointers()		if (mp_has_changed())	\
				    temp1_ptr = mp_acc_float_ptr(temp1);

	    if (t6 < 2)
		t6 = 2;

	    else if (t6 > t)
		t6 = t;

	    mp_t(temp1_ptr) = mp_t(mp_acc_float_ptr(temp2)) =
		mp_t(mp_acc_float_ptr(temp3)) = t6;

	    mp_move(x, temp1);
	    fix_pointers();
	    mp_set_sign(temp1_ptr, 1);

	    mp_div_int(temp1, 2, temp2);
	    mp_log(temp2, temp2);
	    mp_mul_q_eq(temp2, 2 * abs_nu + 1, 2);
	    mp_add_eq(temp1, temp2);

	    mp_log_int(b, temp2);
	    mp_div_eq(temp1, temp2);

	    it = mp_to_int(temp1);


	    extra_t = new_t;

	    new_t = t;
	    if (it > -1)
		new_t += it + 1;

	    if (new_t > 2 * extra_t)
	    {
		/*
		If we need more digits than space allows for power series,
		then use the recurrence method instead and return.
		*/

		fix_pointers();
		mp_t(temp1_ptr) = extra_t;

		mp_move(x, temp1);

		fix_pointers();
		mp_set_sign(temp1_ptr, 1);

		mp_bes2(temp1, abs_nu, temp1);


		/*
		Correct sign if nu is odd.
		*/

		if (abs_nu % 2)
		{
		    fix_pointers();
		    mp_sign(temp1_ptr) *= x_sign;
		}


		/*
		The simplest and easiest way....
		*/

		mp_move(temp1, y);
		mp_acc_float_delete(temp2);
		mp_acc_float_delete(temp3);
		goto finished;

#undef fix_pointers

#if 0

		round = save_round;
		mp_acc_float_delete(temp1);
		mp_change_down();

		DEBUG_1("-} y = ", yp);
		DEBUG_END();
		return y;
#endif
	    }
	    mp_acc_float_delete(temp2);
	    mp_acc_float_delete(temp3);
	}
	else
	    new_t = t;


	/*
	Prepare for power series.
	*/

	mp_acc_float_delete(temp1);
	mp_acc_float_alloc_3(b, new_t, temp1, temp2, temp3);

	temp1_ptr = mp_acc_float_ptr(temp1);
	temp2_ptr = mp_acc_float_ptr(temp2);
	temp3_ptr = mp_acc_float_ptr(temp3);

	mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) = t;

#define fix_pointers()		if (mp_has_changed())			    \
				{					    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				    temp3_ptr = mp_acc_float_ptr(temp3);    \
				}


	mp_div_int(x, 2, temp3);
	mp_abs_int_power(temp3, abs_nu, temp3);
	mp_gam_q(abs_nu + 1, 1, temp1);
	mp_div_eq(temp3, temp1);
	mp_mul(x, x, temp2);
	mp_div_int_eq(temp2, -4);


	/*
	Clear trailing digits of temp2 and temp3.
	*/

	fix_pointers();

	mp_set_digits_zero(mp_digit_ptr(temp1_ptr, t), new_t - t);
	mp_set_digits_zero(mp_digit_ptr(temp2_ptr, t), new_t - t);
	mp_set_digits_zero(mp_digit_ptr(temp3_ptr, t), new_t - t);


	mp_t(temp1_ptr) = mp_t(temp3_ptr) = new_t;
	mp_copy_ptr(temp3_ptr, temp1_ptr);

	base_t = new_t + 2 - mp_expt(temp1_ptr);

	k = 0;
	max_k = MAX_INT / 2 - abs_nu;

	do
	{
	    mp_length	mul_t = base_t + mp_expt(temp3_ptr);

	    if (mul_t < 2)
		break;

	    if (mul_t > new_t)
		mul_t = new_t;

	    mp_t(temp2_ptr) = mp_t(temp3_ptr) = mul_t;

	    mp_mul_eq(temp3, temp2);

	    if (++k > max_k)
		mp_error("mp_besj: int_abs(nu) too large");
	    
	    mp_mul_double_q(temp3, 1, 1, k, k + abs_nu);

	    /*
	    Restore t of temp3 for addition.
	    */

	    fix_pointers();
	    mp_t(temp3_ptr) = new_t;

	    mp_add_eq(temp1, temp3);

	    fix_pointers();
	
	} while (!mp_is_zero(temp3_ptr) &&
		    mp_expt(temp3_ptr) >= mp_expt(temp1_ptr) - t);

	mp_acc_float_delete(temp2);
	mp_acc_float_delete(temp3);
    }


    mp_move(temp1, y);


    /*
    Correct sign if nu is odd.
    */

    if (nu < 0 && abs_nu % 2)
    {
	yp = mp_ptr(y);
	mp_set_sign(yp, -mp_sign(yp));
    }

finished:

    round = save_round;

    mp_acc_float_delete(temp1);
    mp_change_down();

    DEBUG_1("-} y = ", mp_ptr(y));
    DEBUG_END();

    return y;

#undef fix_pointers
}



void
mp_priv_bes2	WITH_3_ARGS(
	mp_float,	x,
	mp_int,		nu,
	mp_float,	y
)
/*
Uses the backward recurrence method to evaluate y = J(nu, x) as in
mp_besj().  Assumes that nu > 0 (not too large) and x > 0.
*/
{
    /*
    For normalization the following identity is used:

	J(0, x) + 2 * J(2, x) + 2 * J(4, x) + ... = 1.
    */

    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_base_type	b;
    mp_length		t, t6;
    mp_acc_float	temp1, temp2, temp3, temp4, temp5;
    mp_ptr_type		temp1_ptr, temp2_ptr, temp3_ptr;
    mp_int		nu1, i;


    DEBUG_BEGIN(DEBUG_BESSEL);
    DEBUG_PRINTF_1("+bes2 {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("nu = %d\n", nu);

    mp_check_2("mp_bes2", xp, yp);


    if (nu < 0 || !mp_is_pos(xp))
	mp_error("mp_bes2: illegal arguments");

    b = mp_b(xp);
    t = mp_t(xp);

    round = MP_TRUNC;

    mp_acc_float_alloc_3(b, t, temp1, temp2, temp3);
    mp_change_up();

    mp_move(x, temp1);


    /*
    Use about 6 decimal places to compute starting point.
    */

    t6 = mp_change_base(b, 10, 5) + 1;

    if (t6 < 3)
	t6 = 3;

    else if (t6 > t)
	t6 = t;


    temp1_ptr = mp_acc_float_ptr(temp1);
    temp2_ptr = mp_acc_float_ptr(temp2);
    temp3_ptr = mp_acc_float_ptr(temp3);


#define fix_pointers()		if (mp_has_changed())			    \
				{					    \
				    temp1_ptr = mp_acc_float_ptr(temp1);    \
				    temp2_ptr = mp_acc_float_ptr(temp2);    \
				    temp3_ptr = mp_acc_float_ptr(temp3);    \
				}

    /*
    Change t of x also for use with temp2, etc..
    */
    
    mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) = mp_t(mp_ptr(x)) = t6;


    mp_int_to_mp(int_max(1, nu), temp1);
    mp_mul_int(temp1, 2, temp2);

    mp_div_eq(temp2, x);

    mp_log(temp2, temp2);
    mp_add_int_eq(temp2, -1);

    mp_int_to_mp(1, temp3);
    mp_max(temp2, temp3, temp2);
    mp_mul_eq(temp2, temp1);

    mp_log_int(b, temp1);
    mp_mul_q_eq(temp1, t, 2);
    mp_add_eq(temp1, temp2);
    mp_div_eq(temp1, x);


    /*
    Note that 125/92 < e/2.
    */

    mp_mul_q_eq(temp1, 92, 125);
    mp_int_to_mp(2, temp3);
    mp_max(temp1, temp3, temp1);

    mp_copy(temp1, temp2);


    /*
    Do two newton iterations.
    */

    for (i = 0; i < 2; i++)
    {
	mp_add(temp1, temp2, temp3);
	mp_log(temp2, temp2);
	mp_add_int_eq(temp2, 1);
	mp_div(temp3, temp2, temp2);
    }

    mp_mul(temp2, x, temp1);


    /*
    Note that 34/25 > e/2.
    */

    mp_mul_q_eq(temp1, 34, 25);

    if (mp_cmp_int(temp1, MAX_INT - 2) > 0)
	mp_error("mp_bes2: illegal arguments");

    nu1 = mp_to_int(temp1) + 2;

    DEBUG_PRINTF_2("nu1 = %d\n", nu1);


    /*
    Restore t for everything.
    */

    fix_pointers();
    mp_t(temp1_ptr) = mp_t(temp2_ptr) = mp_t(temp3_ptr) =
	    mp_t(mp_ptr(x)) = t;

    mp_set_digits_zero(mp_digit_ptr(temp1_ptr, t6), t - t6);
    mp_set_digits_zero(mp_digit_ptr(temp2_ptr, t6), t - t6);
    mp_set_digits_zero(mp_digit_ptr(temp3_ptr, t6), t - t6);

    mp_acc_float_alloc_2(b, t, temp4, temp5);

    mp_int_to_mp((nu1 + 1) % 2, temp5);
    mp_rec(x, temp1);

    mp_mul_int_eq(temp1, 2);

    fix_pointers();
    mp_set_sign(temp3_ptr, 0);

    mp_int_to_mp(1, temp4);


    /*
    Backward recurrence loop.
    */

    do
    {
	mp_acc_float	rot_temp;


	mp_mul(temp3, temp1, temp4);
	mp_mul_int_eq(temp4, nu1);
	mp_sub_eq(temp4, temp2);


	/*
	Rotate the values of (temp2, temp3, temp4) left.  It's faster to
	change pointers than move the values around.
	*/

	rot_temp = temp2;
	temp2 = temp3;
	temp3 = temp4;
	temp4 = rot_temp;


	if (--nu1 % 2 == 0)
	{
	    /*
	    nu1 is even so update normalizing sum.
	    */

	    if (nu1 == 0)
		mp_mul_int_eq(temp5, 2);

	    mp_add_eq(temp5, temp3);
	}


	/*
	If nu1 == nu save unnormalized  result.
	*/

	if (nu1 == nu)
	    mp_copy(temp3, y);
    
    } while (nu1 > 0);


    /*
    Normalize result and return.
    */

    mp_div_eq(y, temp5);

    round = save_round;

    mp_acc_float_delete(temp5);
    mp_acc_float_delete(temp4);
    mp_acc_float_delete(temp3);
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);
    mp_change_down();

    DEBUG_1("-} y = ", yp);
    DEBUG_END();
}


mp_bool
mp_priv_hank	WITH_3_ARGS(
	mp_float,	x,
	mp_int,	nu,
	mp_float,	y
)
/*
Tries to compute y = J(nu, x) using Hankel's asymptotic series.  nu is a
non-negative integer not too large.  The return value is whether the
computation is successful.  The time taken is O(t^3).
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_round_type	save_round = round;
    mp_sign_type	x_sign;
    mp_base_type	b;
    mp_length		t;
    mp_acc_float	cos_temp, temp2, temp3, temp4, temp5;
    mp_ptr_type		temp5_ptr;
    mp_expt_type	last_expt;
    mp_int		k, max_k;


    DEBUG_BEGIN(DEBUG_BESSEL);
    DEBUG_PRINTF_1("+hank {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("nu = %d\n", nu);


    mp_check_2("mp_hank", xp, yp);


    if (nu < 0)
    {
	DEBUG_PRINTF_1("-} fail\n");
	DEBUG_END();
	
	return FALSE;
    }

    b = mp_b(xp);
    t = mp_t(xp);


    /*
    Work with int_abs(x).
    */

    x_sign = mp_sign(xp);
    mp_set_sign(xp, int_abs(x_sign));


    /*
    Check if int_abs(x) is clearly too small for asymptotic series.
    */

    if (mp_cmp_int(x, mp_times_log2_b(t, b) / 3) <= 0)
    {
	/*
	Restore sign of x and return failure.
	*/

	mp_set_sign(mp_ptr(x), x_sign);

	DEBUG_PRINTF_1("-} fail\n");
	DEBUG_END();
	return FALSE;
    }


    /*
    Use truncating internally and allocate more temporary space.
    */

    round = MP_TRUNC;

    mp_acc_float_alloc_5(b, t, cos_temp, temp2, temp3, temp4, temp5);
    mp_change_up();


    mp_int_power(x, -2, temp2);
    mp_div_int_eq(temp2, -64);
    mp_int_to_mp(1, temp3);
    mp_set_sign(mp_acc_float_ptr(temp4), 0);
    mp_copy(temp3, temp5);


    /*
    Sum two asymptotic series.
    */

    last_expt = 1;

    k = 2;
    max_k = MAX_INT / 2 - nu;

    temp5_ptr = mp_acc_float_ptr(temp5);

    do
    {
	if (k >= max_k || mp_expt(temp5_ptr) > last_expt)
	{
	    /*
	    k is too large or the terms are increasing.
	    Restore sign of x and return failure.
	    */

	    mp_set_sign(mp_ptr(x), x_sign);

	    mp_acc_float_delete(cos_temp);
	    mp_change_down();

	    round = save_round;

	    DEBUG_1("-} y = ", yp);
	    DEBUG_END();
	    return FALSE;
	}

	last_expt = mp_expt(temp5_ptr);

	mp_mul_double_q(temp5, 2 * (nu + k) - 3, 2 * (nu - k) + 3, k - 1, 1);
	mp_add_eq(temp4, temp5);

	mp_mul_double_q(temp5, 2 * (nu + k) - 1, 2 * (nu - k) + 1, k, 1);
	mp_mul_eq(temp5, temp2);
	mp_add_eq(temp3, temp5);

	k += 2;

	if (mp_has_changed())
	    temp5_ptr = mp_acc_float_ptr(temp5);

    } while (!mp_is_zero(temp5_ptr) && mp_expt(temp5_ptr) > -t);


    mp_div_eq(temp4, x);
    mp_div_int_eq(temp4, 8);


    /*
    Compute pi/4 (slightly more accurate than calling mp_pi() and dividing by 4.
    */

    mp_atn1(5, temp5);
    mp_mul_int_eq(temp5, 4);
    mp_atn1(239, temp2);
    mp_sub(temp5, temp2, temp2);


    /*
    Avoid too much cancellation in subtracting a multiple of pi.
    */

    mp_mul_int(temp2, (2 * nu + 1) % 8, temp5);
    mp_sub(x, temp5, temp5);

    mp_cis(temp5, temp5, cos_temp, TRUE, TRUE);

    mp_mul_eq(temp3, temp5);
    mp_mul_eq(temp4, cos_temp);
    mp_sub_eq(temp3, temp4);

    mp_mul_eq(temp2, x);
    mp_mul_int_eq(temp2, 2);
    mp_root(temp2, -2, temp2);
    mp_mul_eq(temp2, temp3);


    /*
    Correct sign of result.
    */

    if (nu % 2)
	mp_sign(mp_acc_float_ptr(temp2)) *= x_sign;


    /*
    Restore sign of x and return success.
    */

    mp_set_sign(mp_ptr(x), x_sign);

    mp_copy(temp2, y);

    mp_acc_float_delete(temp5);
    mp_acc_float_delete(temp4);
    mp_acc_float_delete(temp3);
    mp_acc_float_delete(temp2);
    mp_acc_float_delete(cos_temp);
    mp_change_down();

    round = save_round;

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return TRUE;
}
