#include "defs.h"
#include "mp.e"
#include "mp.h"


mp_float
mp_priv_abs_int_power	WITH_3_ARGS(
	mp_float,	x,
	mp_int,		n,
	mp_float,	y
)
/*
Sets y = x ^ int_abs(n) for mp x and y, and returns y.  0^0 is defined to be 1.
The routine uses no guard digits with truncated rather than rounded arithmetic.
The rounding is not the best possible, but with directed rounding true upper
and lower bounds are obtained.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);


    mp_check_2("mp_priv_abs_power", xp, yp);


    DEBUG_BEGIN(DEBUG_POWER);
    DEBUG_PRINTF_1("+abs_power {\n");
    DEBUG_1("x = ", xp);
    DEBUG_PRINTF_2("n = %d\n\n", n);

    n = int_abs(n);

    if (!n)
	mp_int_to_mp(1, y);

    else if (n == 1)
	mp_copy_ptr(xp, yp);
    
    else if (mp_is_zero(xp))
	mp_set_sign(yp, 0);

    else
    {
	mp_acc_float		result, multiplier;
	mp_sign_type		sign = mp_sign(xp);
	mp_round_type		round_save = round;
	mp_bool			first;


	if (n % 2 == 0)
	    sign = int_abs(sign);

	/*
	Use chopped rather than rounded arithmetic to save time, except with
	directed rounding.
	*/

	round = (round == MP_TRUNC || round == MP_RND)?
		    MP_TRUNC: mp_fix_directed(sign, round);


	/*
	Move int_abs(x) to temporary storage (result will be set in the loop).
	*/

	mp_change_up();

	mp_acc_float_alloc_2(mp_b(xp), mp_t(xp), multiplier, result);
	mp_move(x, multiplier);

	mp_set_sign(mp_acc_float_ptr(multiplier), 1);

	first = TRUE;

	while (n > 0)
	{
	    /*
	    If least significant bit of n is on, multiply in the appropriate
	    power of x.  For the first time, result = 1, so we can just
	    copy multiplier into result.
	    */

	    if (n & 1)
		if (first)
		{
		    mp_copy(multiplier, result);
		    first = FALSE;
		}

		else
		    mp_mul_eq(result, multiplier);


	    /*
	    Shift n right, and square multiplier (don't bother to square
	    if this is the last iteration).
	    */

	    if (n >>= 1)
		mp_mul_eq(multiplier, multiplier);
	}

	mp_move(result, y);

	if (mp_has_changed())
	    yp = mp_ptr(y);

	mp_set_sign(yp, sign);

	mp_acc_float_delete(result);
	mp_acc_float_delete(multiplier);
	mp_change_down();

	round = round_save;
    }

    DEBUG_1("-} y = ", mp_ptr(y));
    DEBUG_END();

    return y;
}


mp_float
mp_int_power	WITH_3_ARGS(
	mp_float,	x,
	mp_int,		n,
	mp_float,	y
)
/*
Sets y = x ^ n for mp x and y, and int n, and returns y.  0^0 is defined to
be 1.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y);
    mp_acc_float	temp;
    mp_base_type	b;
    mp_round_type	save_round;


    DEBUG_BEGIN(DEBUG_POWER);
    DEBUG_PRINTF_1("+int_power {\n");
    DEBUG_1("x = ", xp);
    DEBUG_1("y = ", yp);
    DEBUG_PRINTF_2("n = %d\n\n", n);

    mp_check_2("int_power", xp, yp);


    /*
    Allocate greater precision temporary variable and copy x into it.
    */


    b = mp_b(xp);
    mp_acc_float_alloc(b, mp_t(xp) + mp_extra_guard_digits(n, b), temp);

    mp_move(x, temp);

    if (n < 0)
    {
	mp_round_type	save_round = round;

	if (mp_is_zero(mp_ptr(x)))
	    mp_error("mp_int_power: x = 0 and n < 0");

	if ((-n) % 2 == 0 && (round == MP_RND_UP || round == MP_RND_DOWN))
	    round = mp_fix_directed(mp_sign(xp), round);

	mp_rec(temp, temp);

	round = save_round;
    }

    mp_abs_int_power(temp, n, temp);
    mp_move(temp, y);

    mp_acc_float_delete(temp);

    DEBUG_1("-} y = ", yp);
    DEBUG_END();

    return y;
}


mp_float
mp_power	WITH_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
)
/*
Returns z = x^y, where x is positive (x == 0 allowed if y > 0).
This is slower than mp_int_power() and mp_q_power(), so use them
if possible.  Accumulator operations are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x), yp = mp_ptr(y), zp = mp_ptr(z);
    mp_round_type	save_round;
    mp_base_type	b;
    mp_length		new_t;
    mp_acc_float	temp1, temp2;

    mp_check_3("mp_power", xp, yp, zp);

    DEBUG_BEGIN(DEBUG_OTHER);
    DEBUG_PRINTF_1("+power {\n");
    DEBUG_1("x = ", xp);
    DEBUG_1("y = ", yp);

    if (mp_is_neg(xp))
	mp_error("mp_power: x is negative");

    if (mp_is_zero(xp))
    {
	if (!mp_is_pos(yp))
	    mp_error("mp_power: x is zero and y not positive");

	/*
	0^y = 0.
	*/

	mp_set_sign(zp, 0);

	DEBUG_PRINTF_1("-} y = 0\n");
	DEBUG_END();

	return z;
    }

    if (mp_is_zero(yp) || mp_cmp_int(x, 1) == 0)
    {
	/*
	x^0 == 1 and 1^y == 1.
	*/

	mp_int_to_mp(1, z);

	DEBUG_PRINTF_1("-} y = 1\n");
	DEBUG_END();

	return z;
    }

    /*
    Here we have the usual case: x > 0 && y != 0.  Allocate temporary floats
    with sufficient guard digits and use mp_log() and mp_exp() to compute the
    power.
    */

    b = mp_b(xp);

    new_t = mp_t(xp) + 1 + mp_guard_digits(100 * mp_times_log2_b(1, b) +
		mp_guard_digits(mp_expt(xp) + int_max(0, mp_expt(yp)), b), b);

    mp_acc_float_alloc_2(b, new_t, temp1, temp2);
    mp_move(x, temp1);

    /*
    Take care for directed rounding.
    */

    save_round = round;
    round = mp_fix_directed(mp_sign(yp), round);

    /*
    Set temp1 = log(x), then y * log(x).
    */

    mp_log(temp1, temp1);

    mp_move(y, temp2);
    mp_mul_eq(temp1, temp2);

    /*
    Restore rounding type for mp_exp().
    */

    round = save_round;

    /*
    exp(y * log(x)) will give the final result - if x^y is too large,
    mp_exp() will print the error message.
    */

    mp_exp(temp1, temp1);
    mp_move(temp1, z);

    mp_acc_float_delete(temp2);
    mp_acc_float_delete(temp1);

    DEBUG_1("-} z = ", zp);
    DEBUG_END();

    return z;
}



mp_float
mp_q_power	WITH_5_ARGS(
	mp_int,	i,
	mp_int,	j,
	mp_int,	k,
	mp_int,	l,
	mp_float,	x
)
/*
Returns x = (i/j)^(k/l).  The time taken is O(t^2).  Accumulator operations
are performed.
*/
{
    mp_ptr_type		xp = mp_ptr(x);
    mp_round_type	save_round;
    mp_base_type	b;
    mp_length		new_t;
    mp_acc_float	temp;


    DEBUG_BEGIN(DEBUG_POWER);
    DEBUG_PRINTF_1("+q_power {\n");
    DEBUG_PRINTF_5("i = %d, j = %d, k = %d, l = %d\n", i, j, k, l);

    if (l == 0)
	mp_error("mp_q_power: l == 0");

    if (k == 0)
    {
	/*
	(i/j)^0 == 1 except if j == 0.
	*/

	if (j == 0)
	    mp_error("mp_q_power: k == 0 and j == 0");

	mp_int_to_mp(1, x);

	DEBUG_1("-} x = ", xp);
	DEBUG_END();

	return x;
    }

    /*
    For efficiency make k positive and l negative (see comments in mp_root()
    and mp_abs_int_power()).
    */

    if (k < 0)
    {
	/*
	Force k to be positive.
	*/

	k = -k;
	l = -l;
    }
    
    if (l > 0)
    {
	/*
	Force l to be negative by inverting the fraction i/j.
	*/

	mp_int	swap_temp;

	mp_swap_vars(i, j, swap_temp);
	l = -l;
    }


    /*
    Now k > 0 and l < 0.
    */

    if (i == 0)
	mp_error("mp_q_power: i == 0 and k/l < 0 or j == 0 and k/l > 0");

    if (j == 0)
    {
	/*
	(i/0)^(negative) == 0 if i is non-zero.
	*/

	mp_set_sign(xp, 0);

	DEBUG_1("-} x = ", xp);
	DEBUG_END();
	return x;
    }

    /*
    To save time in mp_root() and mp_abs_int_power(), remove any common
    factor of k and l.
    */

    mp_int_gcd(&k, &l);


    /*
    Check for directed rounding since the root has the opposite sense
    in rounding to the final result.
    */

    save_round = round;

    if (k % 2 == 0)
	round = mp_fix_directed(-i * int_sign(j), round);


    /*
    Allocate a temporary float with sufficient guard digits.
    */

    b = mp_b(xp);
    new_t = mp_t(xp) + 1 + mp_extra_guard_digits(k, b);

    mp_acc_float_alloc(b, new_t, temp);


    if (l == -1)

	/*
	Special case: avoid root phase by just swapping i and j.
	*/

	mp_q_to_mp(j, i, temp);

    else
    {
	mp_q_to_mp(i, j, temp);
	mp_root(temp, l, temp);
    }


    /*
    Restore rounding since the operations are now in the same sense of the
    desired directly rounded result.
    */

    round = save_round;


    /*
    Since k is positive, we may call mp_abs_int_power() instead of
    mp_int_power().
    */

    mp_abs_int_power(temp, k, temp);


    /*
    Move result into x and restore stack, etc.
    */

    mp_move(temp, x);
    mp_acc_float_delete(temp);

    DEBUG_1("-} x = ", xp);
    DEBUG_END();

    return x;
}
