#ifndef INCLUDED_MP_PRIV_INTERNALS
#define INCLUDED_MP_PRIV_INTERNALS

#include "mp.e"
#define NO_DEBUG

#ifndef TRUE
#define TRUE	1
#define FALSE	0
#endif

#define MP_USE_RECALL

/******************************************************************************
		     THE FOLLOWING DEFINITIONS ARE NON-PORTABLE
******************************************************************************/

#define byte_cpy	memcpy
#define char_to_int	byte_to_int



/******************************************************************************
				PRIVATE TYPES
******************************************************************************/

typedef struct
{
    mp_base_type	mp_bse;
    mp_length		mp_len;		/* equivalent to t */
    mp_sign_type	mp_sgn;
    mp_expt_type	mp_xpt;
    mp_digit_type	mp_dgts[VARIABLE_LENGTH];
} mp_body_type;


/*
The internal structure of the public mp_float:
*/

typedef struct
{
    t_block_header		block_header;
    mp_body_type	mp_bdy;
} mp_float_hdr;



/*
The internal structure of the mp_float_array:
*/

typedef struct
{
    t_block_header		block_header;
    mp_int		mp_arr_dim;
} mp_float_array_hdr;

#define MP_ARRAY_HDR_LEN	MEM_BYTES_TO_WORDS(sizeof(mp_float_array_hdr))



typedef mp_length	mp_acc_index;
typedef mp_digit_type	*mp_digit_ptr_type;
typedef mp_body_type	*mp_ptr_type;
typedef mp_float	mp_acc_float;
typedef mp_int		*mp_int_ptr;
typedef mp_length	mp_digit_index;


/******************************************************************************
			GLOBAL VARIABLES
******************************************************************************/


/*
The pre-processor string GLOBAL is used to define variables in one and only
one source file and to declare them in all the others.
*/

#ifdef MP_DEFINITIONS
#define GLOBAL
#else
#define GLOBAL extern
#endif


GLOBAL	mp_round_type	mp_priv_round;	   /* rounding type */
GLOBAL	mp_expt_type	mp_priv_min_expt;  /* maximum exponent encountered */
GLOBAL	mp_expt_type	mp_priv_max_expt;  /* minimum exponent encountered */
GLOBAL	mp_int		mp_priv_num_unfl;  /* number of overflows encountered */
GLOBAL	mp_acc_index	mp_priv_acc_sp;	   /* accumulator stack pointer */
GLOBAL	void_func	mp_priv_callback;  /* user callback for errors */

GLOBAL	mp_acc_index	max_sp;		   /* maximum sp statistic */

#ifndef NO_DEBUG
GLOBAL	mp_int		mp_priv_indent;	   /* debugging identation level */
#endif



/*
Make the name of some functions/variables easier to reference.
*/

#define mp_error	mp_priv_error
#define mp_bug		mp_priv_bug
#define mp_reset	mp_priv_reset
#define round		mp_priv_round
#define min_expt	mp_priv_min_expt
#define max_expt	mp_priv_max_expt
#define num_unfl	mp_priv_num_unfl
#define mp_acc_sp	mp_priv_acc_sp
#define mp_callback	mp_priv_callback




/******************************************************************************
			CONVERSION MACROS
******************************************************************************/

/*
The following macros convert n objects of one type to the equivalent
number of another type.
*/

#define byte_to_int(n)		(((n) + sizeof(mp_int) - 1) / sizeof(mp_int))
#define digit_to_int(n)		byte_to_int((n) * sizeof(mp_digit_type))
#define int_to_digit(n)		((n) * sizeof(t_word) / sizeof(mp_digit_type))


/*
The following define the length of a body (excluding the digits) and 
the length of a user float header.
*/

#define MP_BODY_LEN		byte_to_int(sizeof(mp_body_type) - \
				    sizeof(mp_digit_type))

#define MP_FLOAT_HDR_LEN	byte_to_int(sizeof(mp_float_hdr) - \
					sizeof(mp_digit_type))



/*
The following return the number of mp_ints required to represent an
mp_body or mp_float with t digits.
*/

#define mp_body_size(t)		(MP_BODY_LEN + digit_to_int(t))
#define mp_size(t)		(MP_FLOAT_HDR_LEN + digit_to_int(t))



/******************************************************************************
		MACROS USED BY THE MP ROUTINES
******************************************************************************/

/*
The following, given an mp_float, returns a pointer to the body of that float.
*/

#define mp_ptr(f)	(&((mp_float_hdr *)mem_access(f))->mp_bdy)


/*
The following macros access various fields of a float body.
*/

#define mp_t(z)			((z)->mp_len)
#define mp_b(z)			((z)->mp_bse)

#define mp_sign(z)		((z)->mp_sgn)
#define mp_is_zero(z)		(mp_sign(z) == 0)
#define mp_is_pos(z)		(mp_sign(z) > 0)
#define mp_is_neg(z)		(mp_sign(z) < 0)

#define mp_expt(z)		((z)->mp_xpt)
#define mp_set_sign(z, s)	(mp_sign(z) = (s))


/*
The size of a digit may be less than that of an int, but all calculations
with digits are done using ints.  Consequently, the macro to obtain the value
of a digit casts its value to an int, and a separate macro is needed to set
a digit, since the cast is not wanted then (MP_DIGIT is a macro which is
private to the header file).
*/

#define MP_DIGIT(z, i)		((z)->mp_dgts[i])
#define mp_digit(z, i)		((mp_int)MP_DIGIT(z, i))
#define mp_set_digit(z, i, x)	(MP_DIGIT(z, i) = (x))

#define mp_digit_ptr(z, i)	(mp_digit_ptr_type)(z->mp_dgts + (i))


/*
The following macros reference accumulator objects.
*/

#define mp_acc_ptr(i)		(mem_access(mp_priv_accum) + (i))
#define mp_acc_digit_ptr(i)	((mp_digit_ptr_type)mp_acc_ptr(i))
#define mp_acc_float_ptr(f)	mp_ptr(f)
#define mp_acc_expt(x)		mp_expt(mp_acc_float_ptr(x))


/*
The following returns a pointer to the i'th element of the array a (origin 0).
*/

#if 0
#define mp_array_element_ptr(a, i)	\
	    ((mp_ptr_type)(mem_access(a) + MP_ARRAY_HDR_LEN + \
		(i) * (((mp_float_array_hdr *)mem_access(a))->mp_arr_el_size)))
#endif

#define mp_array_element(a, i)	\
	    ((mem_access(a) + MP_ARRAY_HDR_LEN)[i])

#define mp_array_element_ptr(a, i)	\
	    mp_ptr(mp_array_element(a, i))



/*
The following macro is used in the Karatsuba function.	It converts a digit
index pointing into the accumulator into a digit pointer.
*/

#define mp_karatsuba_ptr(o)	(((mp_digit_ptr_type)mem_access(mp_accum)) + (o))



/*
Convert a sign and rounding type to the correct rounding type
for directed rounding.
*/

#define mp_fix_directed(s, r)	(((r) == MP_RND_DOWN || (r) == MP_RND_UP) && \
				    (s) < 0? MP_RND_DOWN + MP_RND_UP - (r): (r))



/*****************************************************************************
			ACCUMULATOR VARIABLES AND MACROS
*****************************************************************************/


#define ACC_SIZE		50		/* Initial accumulator size */
#define ACC_EXTEND_CONST	5


GLOBAL	t_handle		mp_priv_accum;		/* accumulator t_handle */
#define mp_accum	mp_priv_accum


/*
Change variables:  when an extend operation is performed by mp_acc_alloc(),
the change_flag is modified to reflect this, so the caller and the functions
above it will know that their pointers may be now undefined.
*/

GLOBAL	long		mp_priv_change_flag;
GLOBAL	long		mp_priv_change_mask;
#define mp_change_mask	mp_priv_change_mask
#define mp_change_flag	mp_priv_change_flag



/*
NOTE:	in the following allocating macros, an accumulator index is passed
	as a REFERENCE.


Sets the REFERENCE p to an accumulator index for a space of n ints.
*/

#define mp_acc_alloc(nn, p)						\
{									\
    register mp_length	n = nn;						\
    p = mp_acc_sp;							\
    DEBUG_PRINTF_3("acc_alloc: n = %d, p = %d\n", n, p);		\
									\
    if ((mp_acc_sp += n) > mem_words(mp_accum))				\
    {									\
	mem_extend_words_zero(mp_accum, mp_acc_sp + n * ACC_EXTEND_CONST);		\
    }									\
    if (mp_acc_sp > max_sp) max_sp = mp_acc_sp; \
}


/*
Sets the REFERENCE p to an accumulator index for a space of n digits.
*/

#define mp_acc_digit_alloc(n, p)	mp_acc_alloc(digit_to_int(n), p)



/*
Sets z to an allocated space for a float with base b and digits t.
z is a REFERENCE.
*/

#define mp_acc_float_alloc(b, t, z)					\
{									\
    z = mp_alloc(b, t);							\
}



/*
Allocates space for z1 and z2 as in mp_acc_float_alloc().
*/

#define mp_acc_float_alloc_2(b, t, z1, z2)				\
{									\
    z1 = mp_alloc(b, t);						\
    z2 = mp_alloc(b, t);						\
}


/*
Allocates space for z1, z2, and z3 as in mp_acc_float_alloc().
*/

#define mp_acc_float_alloc_3(b, t, z1, z2, z3)				\
{									\
    z1 = mp_alloc(b, t);						\
    z2 = mp_alloc(b, t);						\
    z3 = mp_alloc(b, t);						\
}


/*
Allocates space for z1, z2, z3, and z4 as in mp_acc_float_alloc().
*/

#define mp_acc_float_alloc_4(b, t, z1, z2, z3, z4)			\
{									\
    z1 = mp_alloc(b, t);						\
    z2 = mp_alloc(b, t);						\
    z3 = mp_alloc(b, t);						\
    z4 = mp_alloc(b, t);						\
}



/*
Allocates space for z1, z2, z3, z4, and z5 as in mp_acc_float_alloc().
*/

#define mp_acc_float_alloc_5(b, t, z1, z2, z3, z4, z5)			\
{									\
    z1 = mp_alloc(b, t);						\
    z2 = mp_alloc(b, t);						\
    z3 = mp_alloc(b, t);						\
    z4 = mp_alloc(b, t);						\
    z5 = mp_alloc(b, t);						\
}


/*
The following frees space by truncating the accumulator stack pointer to p.
*/

#ifdef NO_DEBUG
#define mp_acc_delete(p)	(mp_acc_sp = (p))
#else
#define mp_acc_delete(p)					\
{								\
    if ((p) < 0 || (p) > mp_acc_sp)				\
	mp_error("illegal pointer passed to acc_delete\n");	\
								\
    DEBUG_PRINTF_3("acc_delete: p = %d\n", p);			\
    mp_acc_sp = (p);						\
}
#endif


/*
The following deletes the mp_acc_float z.
*/

#define mp_acc_float_delete(z)	mp_delete_float(z)



/*
Change macros:
	mp_change_up()		increases the level of function invocation,
	mp_change_down()	decreases the level of function invocation,
	mp_has_changed()	returns whether a change has occurred in a
				lower level, and resets the flag for this
				level, in either case.
*/

#define mp_change_up()		(mp_change_mask <<= 1)
#define mp_change_down()	(mp_change_mask >>= 1)
#define mp_has_changed()	((mp_change_flag & mp_change_mask) && \
				    (mp_change_flag &= ~mp_change_mask, 1))



/******************************************************************************
			    MISCELLANEOUS MACROS
******************************************************************************/


/*
Set the digits p[0..n-1] to zero.
*/

#define mp_set_digits_zero(p, n)				\
{								\
    register nn = n; register mp_digit_ptr_type q = p;		\
    while (--nn >= 0) *q++ = 0;					\
}



#define unsigned_mod(x, y)	((x) > 0? (x) % (y): (y) - (-(x) % (y)))
#define int_sign(x)		((x) < 0? -1: ((x) > 0))
#define check()			bh_check(0, 1000000)
#define mp_times_log2_b(j, b)	mp_change_base(2, b, j)
#define mp_swap_vars(x, y, temp) (temp = x, x = y, y = temp)



/*
Check macros - the following check whether the float bodies have
compatible parameters.
*/

#define mp_check_2(name, x, y) 		\
		    DENY(mp_t(x) != mp_t(y) || mp_b(x) != mp_b(y))

#define mp_check_3(name, x, y, z)					\
		    DENY(mp_t(x) != mp_t(y) || mp_b(x) != mp_b(y) ||	\
		    mp_t(x) != mp_t(z) || mp_b(x) != mp_b(z));



/******************************************************************************
				COPY MACROS
******************************************************************************/

/*
Copy n ints/digits from s to d.
*/

#define mp_int_copy(d, s, n)	byte_cpy(d, s, (n) * sizeof(t_word))
#define mp_dig_copy(d, s, n)	byte_cpy(d, s, (n) * sizeof(mp_digit_type))


/*
Copy the mp body pointed to by x to that pointed to by y.
*/

#define mp_copy_ptr(x, y)	mp_int_copy(y, x, mp_body_size(mp_t(x)))




/******************************************************************************
			ABBREVIATION MACROS
******************************************************************************/


/*
The following macros abbreviate a common operation - calling a mp function
with the result argument the same as a given argument.	This is similar
to the C "+=", "*=" operators, etc.
*/

#define mp_add_eq(x, y)		mp_add(x, y, x)
#define mp_sub_eq(x, y)		mp_sub(x, y, x)
#define mp_mul_eq(x, y)		mp_mul(x, y, x)
#define mp_div_eq(x, y)		mp_div(x, y, x)
#define mp_add_int_eq(x, i)	mp_add_int(x, i, x)
#define mp_mul_int_eq(x, i)	mp_mul_int(x, i, x)
#define mp_div_int_eq(x, i)	mp_div_int(x, i, x)
#define mp_add_q_eq(x, i, j)	mp_add_q(x, i, j, x)
#define mp_mul_q_eq(x, i, j)	mp_mul_q(x, i, j, x)




/******************************************************************************
			    DEBUG MACROS
******************************************************************************/


#ifdef NO_DEBUG

#define DEBUG_PRINTF_1(a)
#define DEBUG_PRINTF_2(a, b)
#define DEBUG_PRINTF_3(a, b, c)
#define DEBUG_PRINTF_4(a, b, c, d)
#define DEBUG_PRINTF_5(a, b, c, d, e)
#define DEBUG_1(a, b)
#define DEBUG_2(a, b, c)
#define DEBUG_DUMP(a, b, c)
#define DEBUG_END()
#define DEBUG_BEGIN(a)
#define DEBUG_ON(a)
#define DEBUG_OFF(a)

#else


GLOBAL	mp_debug_mask	mp_debug_flags;
GLOBAL	mp_int		mp_priv_debug_level,
			mp_priv_debug_sp,
			mp_priv_debug_stack[32];


/*
Given a mask m, turns debugging on for this function and its immediate
children, if the mask allows it thus.
*/

#define DEBUG_BEGIN(m)	\
    if (mp_debug_flags & (m))					\
    {									\
	mp_priv_debug_stack[mp_priv_debug_sp++] = mp_priv_debug_level;	\
	mp_priv_debug_level = 2;					\
    }									\
    else								\
	mp_priv_debug_level--



/*
Called when a function exits, restores the debugging state of the
parent function.
*/

#define DEBUG_END()	(mp_priv_debug_level++ == 2 && \
	    (mp_priv_debug_level = mp_priv_debug_stack[--mp_priv_debug_sp]))



#define DEBUG_PRINTF_1	mp_priv_debug_printf
#define DEBUG_PRINTF_2	mp_priv_debug_printf
#define DEBUG_PRINTF_3	mp_priv_debug_printf
#define DEBUG_PRINTF_4	mp_priv_debug_printf
#define DEBUG_PRINTF_5	mp_priv_debug_printf
#define DEBUG_1		mp_priv_debug_1
#define DEBUG_2		mp_priv_debug_2
#define DEBUG_DUMP	mp_priv_debug_dump
#define DEBUG_ON	mp_debug_on
#define DEBUG_OFF	mp_debug_off

#endif




/******************************************************************************
			PRIVATE FUNCTIONS
******************************************************************************/


void
mp_priv_bes2	HAS_3_ARGS(
	mp_float,	x,
	mp_int,		nu,
	mp_float,	y
);
/*
Uses the backward recurrence method to evaluate y = J(nu, x) as in
mp_besj().  Assumes that nu > 0 (not too large) and x > 0.
*/


mp_bool
mp_priv_hank	HAS_3_ARGS(
	mp_float,	x,
	mp_int, nu,
	mp_float,	y
);
/*
Tries to compute y = J(nu, x) using Hankel's asymptotic series.	 nu is a
non-negative integer not too large.  The return value is whether the
computation is successful.  The time taken is O(t^3).
*/


void
mp_priv_break_out	HAS_4_ARGS(
	char *,		s,
	mp_sign_type,	sign,
	mp_length,	width,
	void_func,	act /* (char) */
);
/*
Sends the string in s through act(), with optional sign prefixed,
and with suitable formatting spaces specified by width.
*/


mp_int
mp_priv_change_base	HAS_3_ARGS(
	mp_base_type,	b1,
	mp_base_type,	b2,
	mp_int,		n
);
/*
Returns j such that b1^abs(j) >= b2^abs(n),
i.e. abs(j) >= abs(n)*log(b2)/log(b1)
     and sign(j) == sign(n),
assuming that b1 > 1, b2 >= 1, and b1 * b2 <= MAX_INT.
Usually the value of j returned is close to minimal.
*/


void mp_priv_debug_1	HAS_2_ARGS(
	char *,		msg,
	mp_ptr_type,	z
);
/*
Display message, followed by the mp float z
*/


void mp_priv_debug_dump HAS_3_ARGS(
	char *,			msg,
	mp_digit_ptr_type,	p,
	mp_int,			n
);


void
mp_priv_divl	HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_float,	z
);
/*
Sets z = x / y.	 Only called by mp_div() (q.v.).
Accumulator operations are performed.
*/


void
mp_priv_erf2		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Returns y = erf(x) using the power series for small x.	Called by mp_erf().
*/


mp_bool
mp_priv_erf3	HAS_3_ARGS(
	mp_float,	x,
	mp_float,	y,
	mp_bool,	to_infinity
);
/*
If to_infinity is true, sets:

    y = exp(x^2) * integral from x to infinity of exp(u^2) du,

otherwise sets:

    y = exp(-x^2) * integral from 0 to x of exp(u^2) du.


In both cases the asymptotic series is used.  The return value of the function
is whether x is large enough for the asymptotic series to give full accuracy.
The condition on x for the return value to be true is approximately that
x > sqrt(t * log(b)).
*/


mp_float
mp_priv_exp1		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Sets y = exp(x) - 1 and returns y given that abs(x) < 1.
The time taken is O(sqrt(t)*M(t)).  Accumulator operations are performed.
*/


void
mp_priv_int_gcd		HAS_2_ARGS(
	mp_int *,	k,
	mp_int *,	l
);
/*
Sets *k /= gcd and *l /= gcd where gcd is the greatest common divisor
of *k and *l.
*/


mp_length
mp_priv_guard_digits	HAS_2_ARGS(
	mp_int,		n,
	mp_base_type,	b
);
/*
This function (called mpgd in Brent) is useful for computing the number of
guard digits required for various multiple-precision calculations.
It returns the ceiling of the log to the base b of max(1, abs(n));
i.e. the minimum value of j such that b^j >= abs(n) (with j >= 0).  
*/


mp_length
mp_priv_extra_guard_digits	HAS_2_ARGS(
	mp_int,		n,
	mp_base_type,	b
);
/*
Returns 1 + mp_guard_digits(100 * n, b) which can be added to t.  The base is
just passed on to mp_guard_digits.
*/


mp_float
mp_priv_atn1	HAS_2_ARGS(
	mp_int,		n,
	mp_float,	y
);
/*
Sets y = arctan(1 / n) and returns y.  The usual power series is used.
Accumulator operations are performed.
*/


mp_float
mp_priv_log_235		HAS_4_ARGS(
	mp_int,		i,
	mp_int,		j,
	mp_int,		k,
	mp_float,	x
);
/*
Returns x = log(2^i * 3^j * 5^k).  x must be already allocated.	 The method
requires time O(t^2).  It is assumed that i, j, and k are not too large.
Accumulator operations are performed.
*/


void
mp_priv_update	HAS_1_ARG(
	mp_ptr_type,	zp
);
/*
Updates the maximum and minimum exponent indicators of the package with
respect to zp.	No check is made for overflow or underflow.
*/


void
mp_priv_move	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Copies the sign, exponent, and digits of x into y and either pads with zeros
or rounds appropriately.  x & y need not have the same number of digits, but
their base must be equal.  Rounding is determined by the rounding type of y.
This routine is mainly used to copy one mp number to another which has a
different precision (e.g. a temporary variable).  Accumulator operations are
performed if t of y is less than t of x.
*/


void
mp_priv_move_round	HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Moves x + s * (b^-(t of y)) * abs(x) into y using mp_move(), where s is:
	 0	if rounding is truncating or round to nearest,
	-1	if rounding is round down,
	+1	if rounding is round up.

The restrictions on x and y are the same as those in mp_move() (q.v.).
*/


void
mp_priv_karatsuba	HAS_5_ARGS(
	mp_digit_index, x,
	mp_digit_index, y,
	mp_digit_index, z,
	mp_length,	n,
	mp_base_type,	b
);
/*
Karatsuba multiplication: multiplies the n digits in x by the n digits in
y and places the result in 2 * n digits of z.  x, y, and z give a digit
index into the accumulator.  b is the base.
*/


void
mp_priv_nzr	HAS_5_ARGS(
	mp_sign_type,		sign,
	mp_expt_type,		expt,
	mp_ptr_type,		zp,
	mp_digit_ptr_type,	digs,
	mp_length,		guard
);
/*
Normalizes the digits digs containing the specified number of guard digits, and
stores the result in zp, with the sign and exponent.  The contents of digs are
destroyed.  See the notes for a description of the different rounding types.
*/


mp_float
mp_priv_abs_int_power	HAS_3_ARGS(
	mp_float,	x,
	mp_int,		n,
	mp_float,	y
);
/*
Sets y = x ^ abs(n) for mp x and y, and returns y.  0^0 is defined to be 1.
The routine uses no guard digits with truncated rather than rounded arithmetic.
The rounding is not the best possible, but with directed rounding true upper
and lower bounds are obtained.	Accumulator operations are performed.
*/


mp_float
mp_priv_scale	HAS_3_ARGS(
	mp_float,	x,
	mp_base_type,	base,
	mp_int,		j
);
/*
Sets x *= base^j and returns x.	 Accumulator operations are performed.
*/


void
mp_priv_cis	HAS_5_ARGS(
	mp_float,	x,
	mp_float,	c,
	mp_float,	s,
	mp_bool,	do_cos,
	mp_bool,	do_sin
);
/*
Sets c = cos(x) if do_cos is true, and s = sin(x) if do_sin is true.
If do_cos is false, the cos argument is completely ignored, and likewise
for do_sin.  Accumulator operations are performed.
*/


void
mp_priv_shift_r		HAS_3_ARGS(
	mp_digit_ptr_type,	p,
	mp_length,		off,
	mp_length,		n
);
/*
Shifts n digits starting at p right off places.
*/


void
mp_priv_shift_l		HAS_3_ARGS(
	mp_digit_ptr_type,	p,
	mp_length,		off,
	mp_length,		n
);
/*
Shifts n digits starting at p left off places.
*/


mp_int
mp_priv_all_zero	HAS_2_ARGS(
	mp_digit_ptr_type,	p,
	mp_length,		n
);
/*
Returns whether p[0..n-1] are all zero.
*/


mp_int
mp_priv_comp_digits	HAS_2_ARGS(
	mp_ptr_type,	x,
	mp_ptr_type,	y
);
/*
Returns -1, 0, 1 according to whether the digits of x <, ==, > those of y.
x and y are mp pointers.
*/


void
mp_priv_copy		HAS_2_ARGS(
	mp_float,	x,
	mp_float,	y
);
/*
Copies the mp number x to y.
*/




#define mp_shift_r		mp_priv_shift_r
#define mp_debug_dump		mp_priv_debug_dump
#define mp_all_zero		mp_priv_all_zero
#define mp_break_out		mp_priv_break_out
#define mp_karatsuba		mp_priv_karatsuba
#define mp_extra_guard_digits	mp_priv_extra_guard_digits
#define mp_debug_1		mp_priv_debug_1
#define mp_bes2			mp_priv_bes2
#define mp_erf2			mp_priv_erf2
#define mp_erf3			mp_priv_erf3
#define mp_hank			mp_priv_hank
#define mp_cis			mp_priv_cis
#define mp_comp_digits		mp_priv_comp_digits
#define mp_scale		mp_priv_scale
#define mp_guard_digits		mp_priv_guard_digits
#define mp_change_base		mp_priv_change_base
#define mp_log_235		mp_priv_log_235
#define mp_atn1			mp_priv_atn1
#define mp_int_gcd		mp_priv_int_gcd
#define mp_divl			mp_priv_divl
#define mp_exp1			mp_priv_exp1
#define mp_move_round		mp_priv_move_round
#define mp_abs_int_power	mp_priv_abs_int_power
#define mp_move			mp_priv_move
#define mp_copy			mp_priv_copy
#define mp_update		mp_priv_update
#define mp_shift_l		mp_priv_shift_l
#define mp_nzr			mp_priv_nzr

#endif /* INCLUDED_MP_PRIV_INTERNALS */

