/* Dynamic array specifications for use in extended integer package */

#ifndef _dyn_arr_internals_h_
#define _dyn_arr_internals_h_

#include "dyn_arr.e"

typedef
	struct
	{
		t_block_header		block_header;
		dyn_arr_length		dyn_arr_sz;
		dyn_arr_type		dyn_arr_elt[VARIABLE_LENGTH];
	}
	dyn_arr_header;

/*
The following gives the length of the header measured in units of one t_word.
It must be check when porting to new machines
*/

#define DYN_ARR_HDR_LEN	(BLOCK_HEADER_WORDS + 1)

/*
Tracked block type for blocks storing dynamic array - the value of this may be
defined by the calling package, for example, in defs.h or trbl.e which 
are included before this file:
*/

#ifndef DYN_ARR_BLOCK_TYPE

#endif

/*
The following gives the number of bh_ints required to store a block containing
a dynamic array of n objects
*/

#define dyn_arr_required_space(n)	(DYN_ARR_HDR_LEN + (n))

/*
Macro for accessing individual dynamic array elements, numbered from 0 to length-1
*/

#if 0
#define	dyn_arr_element(h,k) \
	(((dyn_arr_header *)mem_access(h))->dyn_arr_elt[k])
#endif

#define dyn_arr_element(h,k) \
        (*mem_check_ptr(h, (((dyn_arr_header *)mem_access(h))->dyn_arr_elt + k)))

/*
The following is the CURRENT number of elements stored in the dynamic array
*/

#define dyn_arr_curr_length(h)	(((dyn_arr_header *)mem_access(h))->dyn_arr_sz)

/*
The following is the maximum number of elements the dynamic array can hold
*/

#define dyn_arr_avail_length(h)	(mem_words(h) - DYN_ARR_HDR_LEN)

/*
The following macro assures that there is enough room for at least req elements
in the sequence with t_handle hdl, by extending it as necessary; if required, at
least minext extra elements will be allocated space
 WARNING!!! - Beware of the "dangling else" problem if used in an if statement
 NOTE:		- Arguments are only evaluated once
*/

#define dyn_arr_assure_space( hdl, req, minext ) \
	{	dyn_arr_handle	_h = hdl; \
		dyn_arr_length	_r = req, _m = minext, _a = dyn_arr_avail_length(_h); \
		if (_r > _a) \
			mem_extend_words_zero( _h, dyn_arr_required_space( (_r > (_a+_m)) ? _r : (_a+_m) ) ); \
	}

/*
A pointer to the first element, for use in tight loops.
 WARNING!!!	- ANY Blockhandler operation might cause a garbage collection,
	moving the block and invalidating the pointer
*/

#define dyn_arr_elt0_ptr(h)	(& (((dyn_arr_header *)mem_access(h))->dyn_arr_elt[0]) )

/*
The following macro will copy sequence elements from sequence with t_handle srchdl,
starting at element srcpos, into space referenced by desthdl, starting at element
dstpos.  A total of len digits are copied.

Both srcpos, dstpos are numbered from 0.

It is permissible for dsthdl to equal srchdl ONLY IF dstpos is equal to srcpos
*/

#define dyn_arr_copy_elements( srchdl, srcpos, len, dsthdl, dstpos ) \
{ \
	register dyn_arr_type	*p = dyn_arr_elt0_ptr( srchdl ) + srcpos; \
	register dyn_arr_type	*q = dyn_arr_elt0_ptr( dsthdl ) + dstpos; \
	register dyn_arr_length	k = len; \
	if (q != p) \
	while (k-- > 0) \
		*q++ = *p++; \
}

/*
The following macro will set all elements of the given t_handle to zero up to the
supplied length
*/

#define dyn_arr_set_zero( h, len ) \
{ \
	register dyn_arr_type	*p = dyn_arr_elt0_ptr( h ); \
	register dyn_arr_length	k = len; \
	while (k-- > 0) \
		*p++ = 0; \
}


/*
** duplicates of above for double precision arrays.
*/

typedef
	struct
	{
		t_block_header		block_header;
		dyn_arr_length		dyn_arr_sz;
		double			dyn_real_arr_elt[VARIABLE_LENGTH];
	}
	dyn_real_arr_header;

#define DYN_REAL_ARR_HDR_LEN	DYN_ARR_HDR_LEN

#define dyn_real_arr_required_space(n)	(DYN_ARR_HDR_LEN + (n)*sizeof(double)/sizeof(t_word))

#define	dyn_real_arr_element(h,k) \
	(((dyn_real_arr_header *)mem_access(h))->dyn_real_arr_elt[k])

#endif /* _dyn_arr_internals_h_ */

