/******************************************************************************
  anf_elt_abs.c
******************************************************************************/

#include "kant.h"
#include "vector.h"
#include "real.e"
#include "anf.h"
#include "conv.e"

vector
anf_elt_abs WITH_2_ARGS(
	order,		ord,
	anf_elt,	alpha
)
/*******************************************************************************
 
Description:

    Computes the absolute values of the conjugate vector of an algebraic number 
    in the actual precision.
    These values are given back in a vector of r1+r2 entries. 
 
    alpha might be given as integer, via coefficients or via conjugates. 
                                        
 
Calling sequence:
 
	vec = anf_elt_abs(ord, alpha);
 
      	order         	ord     = t_handle of order 
        anf_elt         alpha   = algebraic number
        vector          vec     = vector with absolute values
  
      
History:
 
	92-04-01 JS	square root if complex place
	92-03-24 JS	first version
  
*******************************************************************************/
{
	block_declarations;
 
	anf_elt		beta;
	integer_small	i, j, r1, r12;
	t_handle		R;
        vector          vec;
	t_real		temp, temp1, tempr, tempi, two;
 
 
	order_must_be_over_z(ord);
 
	r1    = order_r1(ord);
	r12   = r1+order_r2(ord);
	R     = order_reals(ord);
	two   = conv_int_to_real(R, 2);
 
/*
    beta will contain the conjugate vector
*/ 
	if (anf_elt_is_con(alpha))
		beta = anf_elt_incref(alpha);
	else
		beta = anf_elt_con(ord, alpha);
  
        vec = vec_new(r12);

/*
    first the real conjugates
*/ 
	for (i=1; i<=r1; ++i)
	{
		if (real_sign(R, anf_con(beta, i)) < 0)
			temp = real_negate(R, anf_con(beta, i));
		else
			temp = real_incref(anf_con(beta, i));
 
		vec_entry(vec, i) = temp;
	}
 
/*
    now the non-real conjugates
*/
 
	for (i=r1+1, j=r12+1; i<=r12; ++i, ++j)
	{
		tempr = real_mult(R, anf_con(beta, i), anf_con(beta, i));
		tempi = real_mult(R, anf_con(beta, j), anf_con(beta, j));
		temp1 = real_add(R, tempr, tempi);
		temp  = real_divide(R, temp1, two);
 
		vec_entry(vec, i) = real_sqrt(R, temp);
 
		real_delete(&tempr);
		real_delete(&tempi);
		real_delete(&temp1);
		real_delete(&temp);
	} 
 
	anf_elt_delete(ord, &beta);
        real_delete(&two);
 
	return vec;	
}
