/******************************************************************************
  anf_elt_is_mth_power.c
******************************************************************************/
#include "kant.h"

t_logical
anf_elt_is_mth_power WITH_4_ARGS(
	order,		ord,
	anf_elt,	alpha,
	integer_small,	m,
	anf_elt*,	beta
)

/******************************************************************************
 
Description:	Looks for beta in ord satisfying alpha = beta^m.

Calling sequence:
 
	bool = anf_elt_is_mth_power(ord,alpha,m,&beta);
	
	bool				: TRUE = beta exists
					  FALSE = else
	order		ord		: see above
	anf_elt		alpha		: see above
	m		integer_small	: see above
	anf_elt		beta		: if bool gets true beta is returned here	

	Careful! This routine does not care for any denominators (i.e.
	the denominator of anf_elt_simplify(ord,alpha) must be 1).
 
History:
 
	92-09-04 KW	written
 
******************************************************************************/
{
	block_declarations;

	anf_elt		alpha1,alpha2,alpha3,alpha4,alpha5;
	t_handle	C,R;
	integer_big	a,b,c,d,e;
	integer_small	i,j,n,r1,r2,r12;
	lattice		lat;
	lat_enum_env	env;
	t_logical	bool,finis;
	matrix		mat,mrb;
	t_comp		ctemp;
	t_real		tempu,tempv,tempw;
	vector		lambda;

	if (anf_print_level > 0) printf("Entering anf_elt_is_mth_power...\n");

	n = order_rel_degree(ord);
	R = order_reals(ord);

	r1 = order_r1(ord);
	r2 = order_r2(ord);
	r12 = r1+r2;
	mrb = order_basis_real(ord);

	alpha1 = anf_elt_simplify(ord,alpha);	
	if (!anf_elt_is_integer(alpha1) && (anf_elt_den(alpha1) != 1))
		error_internal("anf_elt_is_mth_power: Cannot t_handle denominators.");
	alpha2 = anf_elt_con(ord,alpha1);

/*
**	Is N(alpha) a m'th-power in Z ?
*/
	bool = FALSE;

	anf_norm(ord,alpha1,&a,&b);

        if ((m & 1) || (integer_compare(a,0) != -1))
	{
		c = integer_abs(a);
		tempu = conv_int_to_real(R,c);
		tempv = real_root(R,tempu,m);
		d = conv_real_to_int_round(R,tempv);
		e = integer_power(d,m);
		bool = integer_compare(c,e) == 0;
		real_delete(&tempu);
		real_delete(&tempv);
		integer_delref(c);
		integer_delref(d);
		integer_delref(e);
	}
	if (!bool) goto end;

/*
**	Let w1,...,wn be a Z-basis of ord.
**	All x = \sum_{i=1}^n x_i*w_i in ord satisfying x^m = alpha are
**	among the elements satisfying T2L(x) = n where 
**	
**	(a) T2L = \sum_{i=1}^n \lambda_j^{-2} |\sum_{i=1}^n x_i*w_i^{(j)}|^2
**
**	(b) \lambda_j = |alpha^(j)|^{1/m} (1<=j<=n).
**
**	Let's enumerate.........
*/	
	C = comp_create(real_dec_prec(R));

/*
**	Set lambda
*/
	lambda = vec_new(n);
	for (i=1;i<=r12;i++)
	{
		ctemp = anf_elt_ith_con(ord,alpha2,i);
		tempu = comp_norm(C,ctemp);
		tempv = real_sqrt(R,tempu);
		tempw = real_root(R,tempv,m);
		vec_entry(lambda,i) = real_inverse(R,tempw);
		comp_elt_delete(C,&ctemp);
		real_delete(&tempu);
		real_delete(&tempv);
		real_delete(&tempw);
	}
	for (i=r1+1;i<=r12;i++)
	{
		vec_entry(lambda,i+r2) = real_incref(vec_entry(lambda,i));
	}
	mat = mat_new(n,n);
	for (i=1;i<=n;i++)	
	{		
		for (j=1;j<=n;j++)
		{
			mat_elt(mat,i,j) = real_mult(R,vec_entry(lambda,i),mat_elt(mrb,i,j));
		}
	}

	lat = lat_create();
	lat_rank(lat) = n;
	lat_basis(lat) = mat;
	lat_basis_ring(lat) = ring_incref(R);
	env = lat_enum_create(lat);
	lat_enum_request_set_next(env);

/*
**	We look for elements with n-1/100 <= T2L(x) <= n+1/100
*/
	tempu = conv_int_to_real(R,n);
	tempv = conv_double_to_real(R,0.01);
	lat_enum_lbound(env) = real_subtract(R,tempu,tempv);
	lat_enum_ubound(env) = real_add(R,tempu,tempv);
	real_delete(&tempu);
	real_delete(&tempv);
	bool = FALSE;
	while (!bool && lat_enum(lat,env))
	{
		alpha3 = lat_elt_to_anf_elt(lat,lat_enum_act_coefs(env),ord);
		alpha4 = anf_power(ord,alpha3,m);
		bool = anf_elt_equal(ord,alpha1,alpha4);
		if (bool)
		{
			*beta = anf_elt_incref(alpha3);
		}
		else
		{
/*
**		Because of T2L(x) = T2L(-x) we must test -alpha if m is odd
**		(the enumeration routine never returns -x for reasons of efficiency).
*/
			if (m & 1)
			{
				alpha5 = anf_negate(ord,alpha4);
				bool = anf_elt_equal(ord,alpha1,alpha5);
				if (bool)
				{
					*beta = anf_negate(ord,alpha3);

				}				
				anf_elt_delete(ord,&alpha5);
			}
		}
		anf_elt_delete(ord,&alpha3);
		anf_elt_delete(ord,&alpha4);
	}
		
	lat_enum_delete(lat,&env);
	lat_delete(&lat);
	vec_delete(R,&lambda);
	ring_delete(&C);

end:	anf_elt_delete(ord,&alpha1);
	anf_elt_delete(ord,&alpha2);		
	integer_delref(a);

	if (anf_print_level > 0) printf("Leaving anf_elt_is_mth_power.\n");                                    

	return(bool);
}
