/******************************************************************************
  anf_ideal_cebotarev.c
******************************************************************************/
#include "kant.h"

anf_ideal
anf_ideal_cebotarev WITH_5_ARGS (
	order,		ord,
	order,		ordmax,
	integer_big,	p,
	anf_elt,	alpha,
	integer_big*,	piq
)
/******************************************************************************
 
Description:	Let alpha be an algebraic number in ord which is not
		a p'th power. By Frobenius/Cebotarev there exists a non-zero
		prime ideal pi of ord which does not contain the discriminant
		of ord such that alpha is not a p'th power in ord modulo pi.
		    
Calling sequence:
 
	pi = anf_ideal_cebotarev(order,ordmax,p,alpha,piq)

	pi		: anf_ideal	= a prime ideal with the properties
					  described above
	ord		: order		= t_handle of order
	ordmax		: order		= maximal overorder of ord.
					  Following features are implemented:
					  (a) ord = ordmax (the handles must be equal)
					  (b) ord is the suborder of ordmax.
					  (c) ordmax = MEM_NH. In that case
					      the programm computes the maximal
					      overorder.
	p		: integer_big	= a prime
	alpha		: anf_elt	= handle of an algebraic number
	piq		: integer_big	= norm of pi is a power of the prime piq

History:

	92-10-12 KW	written
 
******************************************************************************/
{
	block_declarations;

	anf_elt		beta,gamma;
	anf_ideal	pi;
	dyn_arr_handle	lst;
	integer_big	q,tempa,tempb,tempc;
	integer_small	i,n,r;
	t_logical	flag;

	if (anf_print_level > 0) printf("Entering anf_ideal_cebotarev...\n");

	order_disc_assure(ord);
	n = order_abs_degree(ord);

	if (!integer_is_single(p)) error_internal("anf_ideal_cebotarev: p must be a small int at the moment.");

	q = p;
	do
	{
/*
**		Search pi.
**
**		(i)   norm(id) is a power of a prime q,
**		      i.e norm(id) = q^s (s \in {1,...,n}).
**		(ii)  q is the smallest integer in id.
**		(iii) p divides (norm(id)-1) ==> q > p.
**		(iv)  q does not divide disc(ord).
*/
		tempa = q;
		q = i_next_prime(q,TRUE);
		integer_delref(tempa);

/*
**		Check (iv).
*/

		tempa = integer_rem(order_disc(ord),q);
		flag = !tempa;
		integer_delref(tempa);
		if (flag) continue;

/*
**		Check (iii).
*/
		tempa = integer_subtract(q,1);
		flag = tempb = integer_rem(tempa,p);
		integer_delref(tempa);
		integer_delref(tempb);

		if (flag) continue;
/*
**		Compute all prime ideals of ord lying over q.
*/
		lst = order_prime_over(ord,ordmax,q);
		r = dyn_arr_curr_length(lst);

/*
**		Isn't alpha a p'th power in ord modulo a prime ideal of lst ?
*/
		flag = TRUE;
		for (i=1;(flag) && (i<=r);i++)
		{
 			pi = dyn_arr_element(lst,i-1);
			anf_ideal_norm(ord,pi,&tempa,&tempb);
			tempb = integer_subtract(tempa,1);
			tempc = integer_div(tempb,p);
			beta  = anf_elt_power_mod(ord,q,alpha,tempc);
			gamma = anf_elt_subtract(ord,beta,1);
			flag  = anf_elt_in_ideal(ord,gamma,pi);
			integer_delref(tempa);
			integer_delref(tempb);
			integer_delref(tempc);
			anf_elt_delete(ord,&beta);
			anf_elt_delete(ord,&gamma);
		}
		if (!flag) pi = anf_ideal_incref(pi);
		for (i=1;i<=r;i++)
		{
			anf_ideal_delete(ord,&dyn_arr_element(lst,i-1));
		}
		dyn_arr_delete(&lst);			
	} while(flag);

	if (anf_print_level > 0) printf("Leaving anf_ideal_cebotarev.\n");

	*piq = q;
	return(pi);
}
