/******************************************************************************
  order_relation_eval_units.c                                                           
******************************************************************************/
 
#include "kant.h"
    
#define UNCNTFAC  10
#define LOGTEN    2.302585
#define PRECLLL   15

t_void
order_relation_eval_units WITH_1_ARG(
	order,		ord
)
/*******************************************************************************

Description: 
 
        Uses the relations ("S-units") found in the order for computing
        a system of independent units. Here first the relation matrix is
        Hermite reduced. Each column of the reduced matrix which consists
        merely of zeroes refers to a unit in the order.
        In the second step the logarithms of the units are computed and
        these column vectors MLLL reduced. 
        The result is a system of independent units.
 
        Still missing:
 
        (1) Looking for 0-columns in this routine (not in MLLL)
        (2) Handling if status != 0
        (3) Logs should be computed in ..factorize
        (4) mat_ring_mlll_reduce instead of HNF

Calling sequence: 
	
	order_relation_eval_units(ord);
                               
	order		ord    = t_handle of the order 
 
History:
                           
        92-09-09 JS     anf_elt_norm_abs                
        92-09-01 JS     try not setting relation_status to one
 	92-06-10 JS 	LLL reduction of transh after MLLL of unit logs
 	92-05-15 JS 	case no relation matrix at all
 	92-05-06 JS 	first version
  
*******************************************************************************/
{
	block_declarations;
  
        t_handle          Z, R, RL;
        matrix          mat, mat1, mat2, transh, transl, logs, transr, id;
        matrix          mat3, logsh, logsl, transhsub, transhl;
        vector          vec, vec1;
        integer_small   i, j, idcnt, rcnt, pcnt, r, r12, ucnt, rank, prec, d1;
        integer_big     norm, den, maxi;
        t_real          two, eps, logmax;
        anf_elt         unit;
        dyn_arr_handle  marr;
 
/*
    at the moment only status = 0 is supported
*/
        if (order_relation_status(ord))
                error_internal("order_relation_eval_units: Bad status!");
 
/*
    using the units which are known already 
*/
 
        if (order_reg_known(ord) && anf_print_level > 0)
        {
                printf("order_relation_eval_units: Old Regulator = ");
                real_write(order_reals(ord), order_reg(ord), 30);
                puts("");
        }
 
        for (i=1; i<=order_units_count(ord); i++)
                anf_elt_factorize(ord, order_unit(ord, i));
 
	idcnt	= order_fac_basis_ideals_count(ord);
	rcnt  	= order_relation_count(ord);
	pcnt  	= order_fac_basis_len(ord);
        mat   	= order_relation_mat(ord);
        vec   	= order_relation_numbers(ord);
	r12   	= order_r1(ord) + order_r2(ord);
        r       = r12 - 1;
        ucnt  	= (rcnt-idcnt > r*UNCNTFAC) ? r*UNCNTFAC : rcnt-idcnt;
 
        if (anf_print_level > 0)
                printf("order_relation_eval_units: Matrix is %d by %d.\n", idcnt, rcnt);
                   
/*
    Trivial case: No relations
*/ 
        if (!rcnt || ucnt < 1) return;
 
        Z = m_z_str_incref(structure_z);
	R = ring_incref(order_reals(ord));
	RL= real_str_create(PRECLLL);
                                            
/* We do not set the status. That means that in future calls of anf_elt_factorize
   the existing relations will not be deleted. */
/*        order_relation_status(ord) = 1; */
 
/*
    The relation matrix might be too long
*/
            
        if (mat)
        {
	        mat1 = mat;
	        mat  = mat_ring_submat(Z, mat1, 1, 1, idcnt, rcnt);
	        mat_delref(Z, &mat1);
	        order_relation_mat(ord)	= mat_incref(mat);
        }
                                              
        vec1 = vec;
        vec  = vec_new(rcnt);
        for (i=1; i<=rcnt; ++i)
               vec_entry(vec, i) = anf_elt_incref(vec_entry(vec1, i));
        vec_delete(ord, &vec1);
 
        order_relation_numbers(ord) = vec; 
 
/*
    Now HNF reduction (should be MLLL)
*/
        
        transh = mat_ring_create_id(Z, rcnt);
 
        if (mat)
        {
	        if (anf_print_level > 0) 
		puts("Starting HNF reduction of relation matrix...");
	        mat1 = mat; mat  = 0;
        	mat_ring_hnf_col_sub(Z, mat1, &mat, &transh, 1);
	        mat_delref(Z, &mat1);
 
	        if (anf_print_level > 2)
	        {
	                puts("HNF reduced relation matrix:");
	                mat_anf_write(Z, mat); 
	                puts("Trafo matrix:");
	                mat_anf_write(Z, transh); 
                }
        }
 
                      
/*
    Now we extract the correct part of the HNF transformation matrix
    (and LLL-reduce it!)
*/
        if (mat)
	{
	        mat2   = mat_ring_submat(Z, transh, 1, idcnt+1, rcnt, ucnt);
	        if (anf_print_level > 0)
	        {
		        maxi   = mat_z_max(mat2);
		        cay_print("Maximal entry of Trafo: %d   ", maxi);
		        integer_delref(maxi);
		}

	        transl = 0; mat3 = 0;
	        lll_int_d_reduce(Z, mat2, 0.75, &mat3, &transl);
	        mat_delref(Z, &mat2);
	        mat_delref(Z, &transl);
 
	        if (anf_print_level > 0)
	        {
		        maxi   = mat_z_max(mat3);
		        cay_print("... and after LLL:      %d\n", maxi);
		        integer_delref(maxi);
	        }
        
		/*  move in the LLL reduced part of the transformation matrix */

	        mat2   = transh;
	        transh = mat_ring_insert(Z, mat2, mat3, 1, idcnt+1);
	        mat_delref(Z, &mat2);
        }
        else
        {
                mat3 = mat_incref(transh);
        }

/*
    Computing the logs 
    (should perhaps be done in anf_elt_factorize and stored in ord)
*/
        if (anf_print_level > 0) puts("Starting Log calculation...");
 
        logs = mat_new(r12, rcnt);
        for (i=1; i<=rcnt; ++i)
                anf_elt_logs_mat(ord, vec_entry(vec, i), logs, i);
                    
/*
    logsh will hold the logarithms of the possibly dependent units)
*/
        transr = mat_z_to_mat_real(R, mat3); 
        logsh  = mat_ring_mult(R, logs, transr);
 
/*
    We have to look whether the precision is sufficient
*/
         
        logmax = mat_real_max(R, logsh);
        if (anf_print_level > 0) 
	{
		printf("Maximal logarithm entry: ");
                real_write(R, logmax, 10); puts("");
        }
        d1   = (int) (conv_real_to_double(R, logmax)/LOGTEN);
        prec = real_dec_prec(R);      
         
        real_delete(&logmax);
        
        if (prec - d1 < PRECLLL)
        {             
                /* recalculate everything ... */
 
                order_reals_set(ord, d1 + PRECLLL);
                order_reals_create(ord);

	        mat_delref(R, &transr);
	        mat_delref(R, &logsh);
	        mat_delref(R, &logs);
	        ring_delete(&R);
 
                R = ring_incref(order_reals(ord));
 
        	if (anf_print_level > 0) 
			printf("Restart of Log calculation with precision %d...\n", d1 + PRECLLL);
 
	        logs = mat_new(r12, rcnt);
	        for (i=1; i<=rcnt; ++i)
	                anf_elt_logs_mat(ord, vec_entry(vec, i), logs, i);

	        transr = mat_z_to_mat_real(R, mat3); 
	        logsh  = mat_ring_mult(R, logs, transr);
        } 
               
/*
    casting the log matrix to precision LLLPREC
*/
        mat2 = logsh;
        logsh = mat_real_to_mat_real(R, mat2, RL);
 
        mat_delref(R, &mat2);
        mat_delref(Z, &mat3);

/*
    MLLL reduction of Log matrix (referring only to units)
*/
        if (anf_print_level > 0) 
		printf("Starting MLLL reduction (%d by %d) ...\n", mat_row(logsh), mat_col(logsh));

        eps  = conv_double_to_real(RL, 1.0e-10);
        transl = 0;
        mat_ring_mlll_reduce(RL, logsh, eps, &rank, &logsl, &transl); 
        real_delete(&eps);

        if (anf_print_level > 0) 
        {       
                if (rank > 1)
                   printf("%d independent units found (unit rank: %d).\n",rank, r);        
                else
                   printf("1 independent unit found (unit rank: %d).\n", r);        
		puts("MLLL reduced logarithms:");
                mat_real_write(RL, logsl);
        } 
/*        if (anf_print_level > 0) 
        {       
                printf("Trafo matrix:\n");
                mat_anf_write(Z, transl);
                cay_print("Determinant: %d\n", mat_ring_det(Z, transl));
                mat1 = mat_z_to_mat_real(RL, transl);
                printf("Product:\n");
                mat_real_write(RL, mat_ring_mult(RL, logsh, mat1));
        } */
 
/*
    Multiplication of the HNF Trafo matrix by the MLLL Trafo matrix
*/
        if (anf_print_level > 0) puts("Starting exponent calculation...");
 
        id     = mat_ring_create_id(Z, rcnt);
        mat1   = mat_ring_insert(Z, id, transl, idcnt+1, idcnt+1);
        mat2   = transh;
        transh = mat_ring_mult(Z, mat2, mat1);
 
        mat_delref(Z, &id);
        mat_delref(Z, &mat1);
        mat_delref(Z, &mat2);
 
/*
    Computation of the units
*/        
        order_units_delete(ord);
 
	order_units_count_set(ord, rank);
 
/*
    we first excerpt the exponent arrays and LLL reduce them
*/
 
        transhsub = mat_ring_submat(Z, transh, 1, idcnt+1, rcnt, rank);
 
        if (anf_print_level > 0)
        {
	        maxi   = mat_z_max(transhsub);
	        cay_print("Maximal entry of exponent matrix: %d   ", maxi);
	        integer_delref(maxi);
	}

        transhl = 0; mat3 = 0;
        lll_int_d_reduce(Z, transhsub, 0.75, &transhl, &mat3);
        mat_delref(Z, &mat3);
        mat_delref(Z, &transhsub);
 
        if (anf_print_level > 0)
        {
	        maxi   = mat_z_max(transhl);                    
	        cay_print("... and after LLL:                %d\n", maxi);
	        integer_delref(maxi);
        }                               
                     
/*
    now compute the units as power products of the original algebraic numbers
*/
        for (i=1; i<=rank; ++i)
        {        
                vec1 = vec_new(rcnt);
                for (j=1; j<=rcnt; ++j)
                {
                        vec_entry(vec1, j) = 
			integer_incref(mat_elt(transhl,j,i));
                }
                 
                if (anf_print_level > 0)
                {
                        printf("Computing unit #%d, exponent vector:\n", i);
                        mat_anf_write(Z, vec1);
                }
 
                unit = anf_elt_power_product(ord, vec, vec1);
 
                if (anf_print_level > 0)
                {
                        printf("Unit #%d: ", i);
                        anf_elt_write(ord, unit);puts("");
                }

	        anf_elt_norm_abs(ord, unit, &norm, &den);
                if (den != 1 || norm != 1)
			  error_internal("This is no unit!");
 
                integer_delref(norm);
                integer_delref(den);
	        vec_delete(Z, &vec1);
 
                order_unit(ord, i) = unit;
        }

	order_units_logs_calc(ord);
	if (order_units_are_maximal(ord)) 
	{
		order_reg_calc(ord);
		
                if (anf_print_level > 0)
                {
                        printf("Regulator: ");
                        real_write(R, order_reg(ord), 30);
                        puts("");
                } 
        }
                            
        if (mat) mat_delref(Z, &mat);
 
        mat_delref(RL, &logsl);
        mat_delref(RL, &logsh);
        mat_delref(R, &logs);
        mat_delref(R, &transr);
        mat_delref(Z, &transh);
        mat_delref(Z, &transhl);
        mat_delref(Z, &transl);
 
        ring_delete(&Z);
        ring_delete(&R);
        ring_delete(&RL);
 
        return;
}

