/******************************************************************************
  order_write.c                                                           
******************************************************************************/

#include "kant.h"

void
order_write WITH_1_ARG(
	order,		ord
)
/*******************************************************************************
 
Description:
 
       Writes an order to stdout. This routine is semi-compatible to the 
       "(FLD)"-format of KANT V1. If no integer_bigs occur KANT V1 can read
       the data. This also applies to the V2 function order_read_v1.
 
       On the other hand this routine is completely compatible to order_read.
 
       The order must be defined over Z. It should be either defined as equation
       order with power basis or (eventually) via a suborder which is 
       equation order with power basis.  
 
       Class group structure is not supported yet.
 
 
Calling sequence:
 
	order_write(ord);
 
      	order       ord   = t_handle of order to write
      
 
History:
 
	92-05-07 JS	Regulator
	92-03-20 JS	class group
	92-03-10 JS    	first version
 
*******************************************************************************/
{
	block_declarations;
	
	order		ordcoef, ordact, subord;
        anf_ideal       id;
        anf_elt         alpha;
        integer_small   deg, ucnt, ex[11], i, j, k, w, num, numid;
        integer_big     den, h;
        matrix          trans;
        t_poly       pol;
	dyn_arr_handle  polarr;
 

	order_must_be_over_z(ord);
 
        ordcoef = order_coef_order(ord);
	deg 	= order_rel_degree(ord);
        ucnt    = order_units_count(ord); 
        w       = order_torsion_rank(ord);
 
        ordact  = order_simplify_trans(ord);
 
/*
    We need: Polynomial and transformation matrix
*/
	if (order_basis_is_power(ordact))
        {       
                pol   = order_poly(ordact);
                trans = 0;
        }
	else if (order_basis_is_rel(ordact))
	{                                
		subord = order_suborder(ordact);
		if (!order_basis_is_power(subord))
                {
	               	error_internal
			("order_write: Cannot deal with this order.");
                }

                pol    = order_poly(subord);
                trans  = order_tran(ordact);
                den    = order_tran_den(ordact);
	}
        else
        {
                error_internal("order_write: Cannot deal with this order.");
        }
 
/*
   Now we set the Ex-Flags
*/ 
        for (i=10; i; ex[i--]=0);
 
        ex[1] = (trans) ? 2 : 1; if (!order_is_maximal(ordact)) ex[1] = -ex[1];
 
        ex[2] = ucnt;
                                           
        ex[4] = w;

        if (order_units_are_fund(ordact)) ex[5] = ex[2];
 
        if (order_class_group_structure_known(ord))
        {
                h = order_class_number(ord);
                if (integer_compare(h, 999) > 0)
	                ex[7] = -1;
                else
                        ex[7] = h;
        }
        else
        {
                h = 0;
        }

 
        ex[10] = order_r1(ordact);
 
/*
   Output of first line
*/      
         
        printf("(FLD=)%4d,",deg);
        for (i=1; i<=10; ++i) printf("%3d,",ex[i]);
        printf(" * ");
        if (order_disc_known(ordact)) 
		integer_write(order_disc(ordact));
        if (order_reg_known(ordact))
        {
 		printf("  ");
		real_write(order_reals(ordact), order_reg(ordact), 20);
        }
        printf("\n");
 
/*
   Output of polynomial
*/
 
        polarr = poly_to_dyn_arr(ordcoef, pol);
        for (i=deg-1; i>=0; --i) 
        {
                integer_write(dyn_arr_element(polarr, i));
                printf(" ");
                integer_delref(dyn_arr_element(polarr, i));
        }
        printf("\n");
                           
        integer_delref(dyn_arr_element(polarr, deg));
	dyn_arr_delete(&polarr);
 
/*
   Output of transformation matrix (if necessary)
*/  

        if (trans)
        {
                for (j=1; j<=deg; ++j)
                {
                        for (i=1; i<=deg; ++i)
                        {
                                integer_write(mat_elt(trans, i, j));
                                printf(" ");
                        }
                        integer_write(den);
                        printf("\n");
                }
        } 
        
/*
   Output of units 
*/  

        if (ucnt)
        {
                for (i=1; i<=ucnt; ++i)
                {
                        for (j=1; j<=deg; ++j)
                        {
                             integer_write(anf_elt_coef(order_unit(ordact, i), j));
                             printf(" ");
                        }
                        printf("\n");
                }
        } 
 
/*
   Output of torsion unit
*/
        
        if (w > 2)
        {
                for (j=1; j<=deg; ++j)
                {
                     integer_write(anf_elt_coef(order_torsion_unit(ordact), j));
                     printf(" ");
                }
                printf("\n");
        }
 
/*
   ... and output of class group
*/
        if (h > 1)
        { 
                integer_write(h); printf(" ");
		num=order_class_group_order(ord);
                integer_write(num); printf(" ");
                for (j=1; j<=num; j++)
                {
                        integer_write(order_class_group_factor_order(ord, j));
                        printf(" ");
                }
                printf("\n");
                numid=order_class_group_gen_count(ord);
                integer_write(numid); printf("\n");
                for (i=1; i<=numid; i++)
                {                       
                        id = order_class_group_gen(ord, i);
                        integer_write(anf_ideal_degree(id)); printf(" ");
                        integer_write(anf_ideal_gen1(id)); printf(" ");
                        alpha = anf_ideal_gen2(id);
                        for (k=1; k<=deg; k++)
                        {
	                        integer_write(anf_elt_coef(alpha, k)); 
				printf(" ");
                        }
	                printf("\n");
                }  
                for (j=1; j<=num; j++)
                {
                        for (i=1; i<=numid; i++)
                        {
                                integer_write(order_class_group_factor_gen_exp(ord, i, j));
                                printf(" ");
                        }
                        printf("\n");
                }
        }
 
        order_delete(&ordact);
 
        return;
 
}
