#include "defs.h"
#include "structure.e"
#include "error.e"

#ifndef KANT

/*
 *    Generic code for structures
 */

#include "system.h"
#include "pow_struct.e"
#include "alg.h"
#include "dyn_arr.h"
#include "debug.e"

/*
Structure create machinery: Allan, 7/4/92.
*/
 
#define MAX_ARGS    8
#define CREATE_TABLE_SIZE  REP_NUMREPS
#define DELETE_TABLE_SIZE  103
 
typedef struct
{
    t_handle	next;
    t_handle	str;
    t_int	nargs;
    t_handle	args[1];
} t_str_entry;
 
#define ENTRY_SIZE  (sizeof(t_str_entry) - sizeof(t_handle))
 
/* 
 *  The structure table 
 */ 
   
private t_handle structure_table = NH; 
    
/*
 *    Initialise the structure remembering code
 */

public t_void structure_init_table()
{
    structure_table = mem_alloc_words_zero(CREATE_TABLE_SIZE);
}

/*
 *    Create a structure and remember it in the table of remembered
 *    structures.
 *
 *    structure_create(repnum, hard_ptr, comp_ptr, nargs, arg1, ...)
 */

public t_handle structure_create VARARGS_DECL((t_int repnum, ...))
{
    va_list    va, argsp, save_argsp;
    VARARGS_LOCALS(t_int repnum;)
    t_pfh    hard_ptr;
    t_pfl    comp_ptr;
    t_int    nargs, i;
    t_int    hash;
    t_handle	h, args[MAX_ARGS], str;
    t_str_entry	   *hp;

    VARARGS_START(va, t_int, repnum);
    hard_ptr = va_arg(va, t_pfh);
    comp_ptr = va_arg(va, t_pfl);
    nargs = va_arg(va, t_int);

    ASSERT(nargs >= 0 && nargs < MAX_ARGS);

    if (nargs < 0)
    {
	argsp = va_arg(va, va_list);
	save_argsp = argsp;
    }
    else
	for (i = 0; i < nargs; i++)
	    args[i] = va_arg(va, t_handle);

    va_end(args);

    hash = repnum;
    h = mem_check_ptr(structure_table, mem_access(structure_table))[hash];

    while (h)
    {
	hp = (t_str_entry *)mem_access(h);
	if (hp->nargs == nargs)
	{
	    if (nargs < 0)
	    {
		for (i = 0; i < -nargs; i++)
		{
		    t_handle arg = va_arg(argsp, t_handle);

		    if (!(*comp_ptr)(i, arg, hp->args[i]))
		    break;
		}

		if (i == -nargs)
		{
		    IF_DEBUG_FLAG
		    (
			DEBUG_STRUCTURE,
			cay_print(
			    "structure_create: re-using old structure %d\n",
			    hp->str
			)
		    );
		    return ALG_STR_INCREF(hp->str);
		}
	    }
	    else
	    {
		for (i = 0; i < nargs; i++)
		    if (!(*comp_ptr)(i, args[i], hp->args[i]))
		    break;

		if (i == nargs)
		{
		    IF_DEBUG_FLAG
		    (
			DEBUG_STRUCTURE,
			cay_print(
			    "structure_create: re-using old structure %d\n",
			    hp->str
			)
		    );
		    return ALG_STR_INCREF(hp->str);
		}
	    }
	}

	h = hp->next;
    }

    /*
    Make structure first in case it dies and we leave stuff lying around.
    */

    if (nargs < 0)
    {
	str = (*hard_ptr)(save_argsp);
    }
    else
    {
	switch (nargs)
	{
	case 0: str = (*hard_ptr)(); break;
	case 1: str = (*hard_ptr)(args[0]); break;
	case 2: str = (*hard_ptr)(args[0], args[1]); break;
	case 3: str = (*hard_ptr)(args[0], args[1], args[2]); break;
	case 4: str = (*hard_ptr)(args[0], args[1], args[2], args[3]); break;
	case 5: str = (*hard_ptr)(args[0], args[1], args[2], args[3], args[4]);
		break;
	case 6: str = (*hard_ptr)(args[0], args[1], args[2], args[3], args[4],
		    args[5]);
		break;
	case 7: str = (*hard_ptr)(args[0], args[1], args[2], args[3], args[4],
		    args[5], args[6]);
		break;
	default:
		DIE();
	}
    }

    m_structure_set_stored(str);

    h = mem_alloc(ENTRY_SIZE + (nargs >= 0? nargs: -nargs) * sizeof(t_handle));
    hp = (t_str_entry *)mem_access(h);

    hp->str = str;
    hp->nargs = nargs;
    hp->next = mem_access(structure_table)[hash];
    for (i = 0; i < nargs; i++)
    hp->args[i] = args[i];

    mem_access(structure_table)[hash] = h;

    IF_DEBUG_FLAG
    (
	DEBUG_STRUCTURE,
	cay_print(
	    "structure_create: made new structure %d, table t_handle %d\n",
	    str, h
	)
    );

    return str;
}

#endif /* KANT not defined */

/*
 *  Allocate and initialise a structure with all the necessary 
 *    structure information.
 */
 
public t_handle f_structure_alloc MEM_DECL_5(size, trb_type, level, repnum, is_finite)
t_int size;
t_int trb_type, level, repnum;
t_logical is_finite;
{
    t_handle s;

    s = mem_alloc_zero_INHERIT(size);

    block_init(s, trb_type);
    m_structure_put_level(s, level);
    m_structure_put_repnum(s, repnum);
    m_structure_put_parent(s, NH);

    m_structure_clear_stored(s);
    m_structure_set_implicit(s);
    if (is_finite)
	m_structure_set_finite(s);
    else
	m_structure_clear_finite(s);

    return s;
}

#ifndef KANT 
/*
 *    Remove a structure from the table.
 */

public t_void structure_remove(s)
t_handle s;
{
    t_handle	h, *hp;
    t_int    hash;
    t_str_entry	   *ctp;

    ASSERT(structure_is_stored(s));
    ASSERT(block_ref_count(s) == 1);

    IF_DEBUG_FLAG
    (
	DEBUG_STRUCTURE,
	cay_print("structure_remove: biffing structure %d\n", s);
    )

    hash = m_structure_repnum(s);
    hp = mem_access(structure_table) + hash;
    mem_check_ptr(structure_table, hp);

    while (*hp)
    {
	ctp = (t_str_entry *)mem_access(*hp);
	if (ctp->str == s)
	    break;

	hp = &ctp->next;
    }

    ASSERT(*hp);

    h = *hp;
    IF_DEBUG_FLAG
    (
	DEBUG_STRUCTURE,
	cay_print(
	    "structure_remove: removing block %d from create table\n",
	    h
	)
    );
    *hp = ctp->next;
    mem_delete_hptr(&h);
}

/*
 *    Try to find a structure in the structure table which matches
 *    the given argument.
 */

public t_logical structure_match(repnum, comp_ptr, arg, match_str)
t_int repnum;
t_pfl comp_ptr;
t_handle arg, *match_str;
{
    t_int hash;
    t_handle h;

    hash = repnum;
    h = mem_check_ptr(structure_table, mem_access(structure_table))[hash];

    while (NOT_NH(h))
    {
	t_str_entry *hp;

	hp = (t_str_entry *) mem_access(h);
	if ((*comp_ptr)(arg, hp->str))
	{
	    IF_DEBUG_FLAG
	    (
		DEBUG_STRUCTURE,
		cay_print("structure_match: match structure %d\n", 
		    hp->str);
	    )
	    *match_str = hp->str;
	    return TRUE;
	}
	h = hp->next;
    }

    return FALSE;
}

/*
 *    Iterate over all structures of the given repclass in the structure table
 *    calling the given function pointer with each structure.
 */

public t_void structure_iterate_repclass(repnum, func_ptr)
t_int repnum;
t_pfv func_ptr;
{
    t_int hash;
    t_handle h;

    hash = repnum;
    h = mem_check_ptr(structure_table, mem_access(structure_table))[hash];

    while (NOT_NH(h))
    {
	t_str_entry *hp;

	hp = (t_str_entry *) mem_access(h);
	(*func_ptr)(hp -> str);
	h = hp->next;
    }
}

/*
 *    Iterate over all structures of the given variety in the structure table
 *    calling the given function pointer with each structure.
 */

public t_void structure_iterate_variety(varnum, func_ptr)
t_int varnum;
t_pfv func_ptr;
{
    t_int	i;
    t_repclass	*rep;

    for (i = REP_NOTHING + 1; i < REP_NUMREPS; i++)
    {
	rep = m_repclass_num_to_ptr(i);
	if (rep && m_repclass_varnum(rep) == varnum)
	    structure_iterate_repclass(i, func_ptr);
    }
}

/* 
 *  Return the parent of a structure. 
 *    Create it if no such parent exists.
 */ 
 
public t_handle structure_parent_create(s) 
t_handle s; 
{
    t_handle par; 
 
    if (NOT_NH(par = m_structure_parent(s)))
    { 
	return par;
    } 
 
    par = (*(alg_parent_create(s)))(s);
    m_structure_put_parent(s,par);
    return par;
} 

/*
 *  Delete the parent of *a structure* s.
 *  Note: the structure may not have a parent as it may never have
 *  been pushed onto the stack (eg: it was created internally as a
 *  temporary).
 */
 
public t_void structure_parent_delete(s)
t_handle    s;
{
    t_handle pow_str;
 
    pow_str = m_structure_parent(s);
    if (IS_NH(pow_str))
    {
	return;
    }
    pow_struct_delete(0, &pow_str);
    m_structure_parent(s) = NH;
}

/* 
 *    Structure access functions
 */
 
public t_structure_hdr *structure_access(s)
t_handle s; 
{ return m_structure_access(s); } 

public t_int structure_level(s) 
t_handle s; 
{ return m_structure_level(s); } 

public t_handle structure_parent(s) 
t_handle s; 
{ return m_structure_parent(s); } 

public t_int structure_repnum(s) 
t_handle s; 
{ return m_structure_repnum(s); } 

public t_int structure_varnum(s) 
t_handle s; 
{ 
    extern t_repclass *alg_repclasses[];

    return m_repclass_num_to_ptr(m_structure_repnum(s)) -> varnum; 
} 

public t_bool structure_is_stored(s) 
t_handle s; 
{ return m_structure_is_stored(s); }

public t_bool structure_is_finite(s) 
t_handle s; 
{ return m_structure_is_finite(s); }

public t_bool structure_is_explicit(s) 
t_handle s; 
{ return m_structure_is_explicit(s); } 

public t_void structure_set_stored(s) 
t_handle s; 
{ m_structure_set_stored(s); } 

public t_void structure_set_finite(s) 
t_handle s; 
{ m_structure_set_finite(s); } 

public t_void structure_set_explicit(s) 
t_handle s; 
{ m_structure_set_explicit(s); } 

public t_void structure_set_implicit(s) 
t_handle s; 
{ m_structure_set_implicit(s); } 

public t_void structure_clear_stored(s) 
t_handle s; 
{ m_structure_clear_stored(s); } 

public t_void structure_clear_finite(s) 
t_handle s; 
{ m_structure_clear_finite(s); } 

/*
 *    Structure store functions
 */

public t_void structure_put_parent(s, p)
t_handle s, p;
{
    m_structure_put_parent(s,p);
}

public t_void structure_put_level(s, l)
t_handle s;
t_int l;
{
    m_structure_put_level(s,l);
}

public t_char *structure_print_separator_before(s)
t_handle s;
{
    switch (m_structure_repnum(s))
    {
    case REP_MATR:
    case REP_MODHOM:
    case REP_GRP_MAT:
    return "\n";

    case REP_MODTUPLE:
    case REP_MODKG:
    case REP_CODE:
    return "";

    default:
    if (m_structure_level(s) > 1)
	return "";

    return "";
    }
}

public t_char *structure_print_separator_before_setq(s)
t_handle s;
{
    switch (m_structure_repnum(s))
    {
    case REP_MATR:
    case REP_MODHOM:
    case REP_GRP_MAT:
#if 0
    return "\n	  ";
#endif
    return "\n";

    case REP_MODTUPLE:
    case REP_MODKG:
    case REP_CODE:
    return "	";

    case REP_ECART:
    return "";

    default:
    if (m_structure_level(s) > 1)
	return "    ";

    return "";
    }
}

public t_char *structure_print_separator_after(s)
t_handle s;
{
    switch (m_structure_repnum(s))
    {
    case REP_MATR:
    case REP_MODHOM:
    case REP_GRP_MAT:
    return "\n";

    case REP_MODTUPLE:
    case REP_MODKG:
    case REP_CODE:
    return "\n";

    case REP_ECART:
    return " ";

    default:
    if (m_structure_level(s) > 1)
	return "\n";

    return " ";
    }
}

/*
 *    Save and restore the structure table.
 */

public t_void structure_save_workspace(file, write_func)
t_file file;
t_pfi write_func;
{
    (*write_func)(file, &structure_table, sizeof(t_handle));
}

public t_void structure_restore_workspace(file, read_func)
t_file file;
t_pfi read_func;
{
    (*read_func)(file, &structure_table, sizeof(t_handle));
}

#endif /* KANT not defined */
