#include "defs.h"
#include "integer.e"

Logical
integer_is_prime_25E9 WITH_1_ARG(
    integer_big,    p
)
/*
** Given an integer p which is less than 25 * 10^9, return TRUE if p is prime.
** This uses the strong pseudoprime test outlined in H. Riesel,
** "Prime Numbers and Computer Methods for Factorization", Birkhauser,
** 1985, pp 98-99.
*/
{
	block_declarations;

	t_int    s;
	t_int    base;
	t_int    r;
	integer_big      k;
	integer_big      d;
	integer_big      temp;
	integer_big      quot;
	integer_big      rem;
	integer_big      pmin1;



	if (integer_is_single(p))
		return integer_is_prime_small(p);

	/*
	** Test for small cases, and do simple trial division. Note that
	** we can assume that p > 5, so the following gcd test is correct.
	*/
	if (integer_gcd(p, 30) > 1)
		return FALSE;

	/*
	 * Calculate p - 1 = d * 2^s, d odd
	 */

	pmin1 = integer_add(p, -1);
	integer_shift_right_maximal(pmin1, &d, &s);

	for (base = 2; base <= 5; base++)
	{
		if (base == 4)
			continue;

		k = modint_exp(p, base, d);
		if (k == 1)
		{
			continue;
		}

		for (r = 0; r < s; r++)
		{
			if (integer_compare(k, pmin1) == 0)
				break;
			if (r < s - 1)
			{
				temp = k;
				k = modint_exp(p, temp, 2);
				integer_delref(temp);
			}
		}
		integer_delref(k);

		if (r >= s)
		{
			integer_delref(pmin1);
			integer_delref(d);
			/* Failed pseudoprime test */
			return FALSE;
		}
	}

	integer_delref(pmin1);
	integer_delref(d);

	/*
	 * Check the exceptions.
	 *
	 * Note that Riesel gives the exceptions for p < 25 * 10^9:
	 *
	 *	    25,326,001
	 *	   161,304,001
	 *	   960,946,321
	 *	 1,157,839,381
	 *	 3,215,031,751
         *       3,697,278,427
	 *	 5,764,643,587
	 *	 6,770,862,367
	 *	14,386,156,093
	 *	15,579,919,981
	 *	18,459,366,157
	 *	19,887,974,881
	 *	21,276,028,621
	 */

	integer_quot_rem(p, 1000000, &quot, &rem);
	/*
	** quot, rem both single precision for p < 25 * 10^9. Therefore,
	** only have to check with single precision. Note also that the
	** first four tests correspond to single precision integers, and
	** are not really necessary here.
	*/
	if (
		(quot ==    25		&& rem == 326001) ||
		(quot ==   161		&& rem == 304001) ||
		(quot ==   960          && rem == 946321) ||
		(quot ==  1157		&& rem == 839381) ||
		(quot ==  3215		&& rem ==  31751) ||
                (quot ==  3697          && rem == 278427) ||
		(quot ==  5764		&& rem == 643587) ||
		(quot ==  6770		&& rem == 862367) ||
		(quot == 14386		&& rem == 156093) ||
		(quot == 15579		&& rem == 919981) ||
		(quot == 18459		&& rem == 366157) ||
		(quot == 19887		&& rem == 974881) ||
		(quot == 21276		&& rem ==  28621)
	  )
		/* quot is single */
		return FALSE;

	integer_delref(quot);

	/*
	 * Passed all pseudoprime tests, and not one of the exceptions
	 */
	return TRUE;
}
