#include "defs.h"
#include "debug.e"
#include "integer.e"
#include "poly.h"
#include "zm.e"
#include "error.e"

t_poly
modpoly_interpolate WITH_7_ARGS(
    t_handle,         pring,
    t_int,    pdig,
    t_poly,        Bpoly,
    t_int,    b,
    t_int,    bprime,
    t_poly,        Apoly,
    t_poly,        A1poly
)
/*
** modpoly_interpolate() : modular polynomial interpolation
** pdig: prime beta-integer
** Bpoly is a univariate polynomial over Z_pdig
** b is an element of Z_pdig such that Bpoly(b) != 0 and bprime = Bpoly(b)^-1
** A a polynomial in r (>=1) variables, A1 constant in A's main variable,
** returns the unique polynomial A*( x1,..,xr ) s.t.
** A*( x1,..,xr ) = A( x1,..,xr ) modulo B( x1 ) and
** A*( b,x2,..,xr ) = A1( x2,..,xr ) and \curl_1 (A*) <= deg( B )
*/
{
    block_declarations;
    t_handle           Adph;
    t_handle           A1dph;
    t_handle           Aph;
    t_handle           A1ph;
    t_handle           resph;
    t_int    c;
    t_int    d;
    t_int    e;
    t_int    e1;
    t_int    exp;
    t_int    n;
    t_int    noterms;
    t_int    termno;
    t_poly        Adash;
    t_poly        A1dash;
    t_poly        a;
    t_poly        a1;
    t_poly        coef;
    t_poly        respoly;
    t_poly        temp;

    if ( ! integer_is_single( pdig ))
    {
	error_internal( "cannot interpolate for large modulus" );
    }

#ifdef DEBUG
    if ( DEBUG_FLAG( DEBUG_MODPOLY_INTERPOLATE ))
    {
	cay_print("modpoly_interpolate: b = %d, bprime = %d, Bpoly = ", b, bprime);
	poly_z_write(pring,Bpoly);
	cay_print(", Apoly = ");
	poly_z_write(pring,Apoly);
	cay_print(", A1poly = ");
	poly_z_write(pring,A1poly);
	cay_print("\n");
    }
#endif /* DEBUG */

    /* STEP 1 : deg( B ) == 0 */

    n = poly_deg( Bpoly );

    if ( n == 0 )
    {
        /* lift A1poly into world of Apoly.             */
        /* we need to add the LEAST principal variable. */

#ifdef DEBUG
	if ( DEBUG_FLAG( DEBUG_MODPOLY_INTERPOLATE ))
	{
	    cay_print( "degree( B ) = 0, return A1poly\n" );
	}
#endif /* DEBUG */

        poly_z_lift (pring, Apoly, A1poly, &temp, &respoly);
        m_modpoly_delref (pring, temp);
	return  respoly;
    }

    /* Step 2 (r == 1) */

    Aph = m_poly_poly_to_handle (Apoly);
    if (m_poly_univariate (Aph))
    {
        a = modpoly_eval_princvar (pring, pdig, Apoly, b);
        d = modint_subtract( pdig, A1poly, a );
        if (d == 0)
        {
            return( m_modpoly_incref( pring, Apoly ));
        }
        else
        {
            c = modint_mult( pdig, d, bprime);
            temp = modpoly_integer_mult( pring, pdig, Bpoly, c );
            respoly = modpoly_add( pring, pdig, temp, Apoly );
            m_modpoly_delref(pring, temp);
            return( respoly );
        }
    }

    /* Step 3 (r > 1) */

    A1dash = m_modpoly_incref (pring, A1poly);
    Adash  = m_modpoly_incref (pring, Apoly);
    A1dph = m_poly_poly_to_handle (A1dash);
    Adph  = m_poly_poly_to_handle (Adash);
    A1ph  = m_poly_poly_to_handle (A1poly);
    noterms = poly_deg( Apoly) + poly_deg( A1poly) + 2;
    m_poly_create_empty (&resph, m_poly_princvar (Aph),
                                       m_poly_least_pvar (Aph), noterms);
    termno = noterms - 1;

    while (!poly_z_is_zero_poly(pring, Adash) || 
                                    !poly_z_is_zero_poly(pring, A1dash))
    {
        if (poly_z_is_zero_poly(pring, Adash))
        {
            temp = m_poly_coefft (Adph, 0);
            a = poly_z_constant_poly (pring, temp, 0);
            poly_z_advance2 (pring, A1dash, &exp, &a1, &temp);
            m_modpoly_delref (pring, A1dash);
            A1dash = temp;
        }
        else if (poly_z_is_zero_poly (pring, A1dash))
        {
            temp = m_poly_coefft (A1dph, 0);
            a1 = poly_z_constant_poly (pring, temp, 0);
            poly_z_advance2 (pring, Adash, &exp, &a, &temp);
            m_modpoly_delref (pring, Adash);
            Adash = temp;
        }
        else
        {
            e = poly_deg( Adash);
            e1 = poly_deg( A1dash);
            exp = (e > e1 ? e : e1);
            a  = poly_z_zero_poly (pring, m_poly_coefft(Aph,0));
            a1 = poly_z_zero_poly (pring, m_poly_coefft(A1ph,0));
            if (e == exp)
            {
                m_modpoly_delref (pring, a);
                poly_z_advance2 (pring, Adash, &e, &a, &temp);
                m_modpoly_delref (pring, Adash);
                Adash = temp;
            }
            if (e1 == exp)
            {
                m_modpoly_delref (pring, a1);
                poly_z_advance2 (pring, A1dash, &e1, &a1, &temp);
                m_modpoly_delref (pring, A1dash);
                A1dash = temp;
            }
        }
        coef = modpoly_interpolate(pring,pdig,Bpoly,b,bprime,a,a1);
        m_modpoly_delref(pring, a);
        m_modpoly_delref(pring, a1);
        m_poly_coefft( resph, termno) = coef;
        m_poly_expt( resph, termno) = exp;
        termno --;
    }
 
    return (poly_z_clean (pring, m_poly_handle_to_poly (resph)));
}
