#include "defs.h"
#include "mp.e"
#include "mp.h"
#include "mp_build.h"


mp_build_context
mp_build_start		WITH_2_ARGS(
	mp_float,	z,
	mp_base_type,	inbase
)
/*
Sets up a build_context for use with z.  inbase is the base used
for subsequent build input calls.
*/
{
    mp_ptr_type		zp = mp_ptr(z);
    mp_build_context	bc = build_alloc();


    /*
    Store the base and float.
    */

    in_base(bc) = inbase;
    final_z(bc) = z;


    /*
    Allocate a new temporary variable with increased precision.
    Use truncation except for directed rounding.
    */

    temp_z(bc) = mp_alloc(
	mp_b(zp), mp_t(zp) + mp_guard_digits(MAX_INT, mp_b(zp))
    );

    /*
    Initialize the variables which will be used in the build process.
    */

    /* CHECK: not needed: alloc sets z to 0 */
    mp_set_sign(mp_ptr(temp_z(bc)), 0);

    sign(bc) = 1;
    point(bc) = -1;

    expt(bc) = 0;
    expt_sign(bc) = 1;

    state(bc) = MANT_SIGN;

    return bc;
}



void
mp_build_point		WITH_1_ARG(
	mp_build_context,	bc
)
/*
Build process only: inserts a fractional point in the digits of the mp
number being built.  See the documentation for details.
*/
{
    /*
    The variable point counts how many digits have been inserted after the
    fractional point.  If the point has not been encountered yet, the variable
    will have value -1.
    */

    if (state(bc) == MANT || state(bc) == MANT_SIGN)
    {
	state(bc) = MANT;

	if (point(bc) >= 0)
	    mp_error("mp_build_point: decimal point encountered twice");

	point(bc) = 0;
    }

    else
       mp_error("mp_build_point: illegal decimal point");
}



void
mp_build_minus		WITH_1_ARG(
	mp_build_context,	bc
)
/*
Build process only: inserts a minus sign in front of the mantissa/exponent
of the mp number being built.  See the documentation for details.
*/
{
    if (state(bc) == MANT_SIGN)
    {
	state(bc) = MANT;

	sign(bc) = -1;
    }

    else if (state(bc) == EXPT_SIGN)
    {
	state(bc) = EXPT;

	expt_sign(bc) = -1;
    }

    else
       mp_error("mp_build_minus: illegal minus sign");

}



void
mp_build_expt		WITH_1_ARG(
	mp_build_context,	bc
)
/*
Build process only: signifies the beginning of the exponent in the mp
number which is being built.  See notes for details.
*/
{
    state(bc) = EXPT_SIGN;
}



void
mp_build_digit		WITH_2_ARGS(
	mp_build_context,	bc,
	mp_digit_type,		dig
)
/*
Build process only: inserts the next digit into the mp number which is being
built.  This is called for digits of both the mantissa and the exponent.
See notes for details.
*/
{
    mp_base_type	inbase = in_base(bc);


    if (dig < 0 || dig >= inbase)
	mp_error("mp_build_digit: digit (%d) out of range", dig);

    if (state(bc) == MANT_SIGN || state(bc) == MANT)
    {
	mp_float	temp;

	temp = temp_z(bc);
	state(bc) = MANT;


	/*
	Shift left the number and add in the new digit.
	*/

	mp_mul_int_eq(temp, inbase);

	if (dig)
	    mp_add_int_eq(temp, dig);


	/*
	Count the number of places past the fractional point if it
	has been encountered.
	*/

	if (point(bc) >= 0)
	    point(bc)++;
    }

    else /* state(bc) == EXPT_SIGN || state(bc) == EXPT */
    {
	state(bc) == EXPT;

	if (expt(bc) >= MAX_EXPT / (2 * inbase))
	    mp_error("mp_build_digit: exponent overflow");

	expt(bc) *= inbase;
	expt(bc) += dig;
    }

}


mp_float
mp_build_finish		WITH_1_ARG(
	mp_build_context *,	bptr
)
/*
Finishes a build process and returns the completed mp number.
*/
{
    mp_length	locus = point(*bptr);
    mp_float    z;
    mp_float	temp;
    
    z = final_z(*bptr);
    temp = temp_z(*bptr);

    /*
    Account for optional initial sign.
    */

    mp_sign(mp_ptr(temp)) *= sign(*bptr);


    /*
    If no fractional point was found, set number of fractional places to 0.
    */

    if (locus < 0)
	locus = 0;


    /*
    Adjust number for exponent and number of fractional places and copy into
    final mp number.  Delete the temporary mp number and the build context.
    */

    mp_scale(temp, in_base(*bptr), expt(*bptr) * expt_sign(*bptr) - locus);
    mp_move(temp, z);

    mp_delete_float(temp);
    build_delete(bptr);

    DEBUG_BEGIN(DEBUG_BUILD);
    DEBUG_PRINTF_1("+build {\n");
    DEBUG_1("final = ", mp_ptr(z));
    DEBUG_PRINTF_1("-}\n");
    DEBUG_END();

    return z;
}
