#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"
#include "poly.h"
#include "dyn_arr.h"
#include "poly_z_faclst.h"
#include "bit_vector.h"
#include "debug.e"

t_handle	poly_u_z_sqfree_fact ( pring,apoly)
t_handle	pring;
t_poly	apoly;
/*
** SAC IUSFPF
** FACTOR A SQUARE FREE POLY
** apoly is an integral univariate sqfree poly which is positive, 
** primitive and of positive degree. Return  an array of the 
** positive irreducible factors of apoly.
*/
{
	t_handle	res;
	t_int		deg;
	t_int		pdig;
	t_bit_vector	degrees;
	t_handle	ddfact;
	t_poly	ppoly;
	t_int		pdeg;
	t_int		len;
	t_handle	pfactors;
	t_handle	modpfact;
	t_int		lenp;
	integer_big	a, b, h, M, temp;
	t_handle	modMfact;
	t_int		i, j;
	block_declarations;
	

	/* 
	** Step 1 : Compute the distinct degree factorisation
	** for suitable prime p and a set C of possible degrees
	** for factors.
	*/

	deg = poly_deg( apoly);
	
	poly_u_z_fact_deg_set(pring, apoly, &pdig, &ddfact, &degrees, 5);
	/*
	** Returns a list of tuples (pi, ei) where pi is the product of all
	** irreducible factors of apoly of degree ei.
	*/
	if (!pdig)
	{
		/* apoly is irreducible. */
		res = poly_array_alloc(1);
		poly_array_append(res, apoly);
		return res;
	}

	/* Step 2 : Factor distinct degree factors. */
	len = m_poly_z_faclst_len(ddfact);
	modpfact = poly_array_alloc(deg);
	for(i=0; i<len; i++)
	{
		ppoly = m_poly_z_faclst_factor(ddfact, i);
		pdeg = m_poly_z_faclst_power(ddfact, i);
		if ( pdeg == poly_deg( ppoly))
			poly_array_append(modpfact, ppoly);
		else
		{
			pfactors = poly_u_zm_fact(pring, pdig, ppoly);
			poly_z_elt_delete(pring, &ppoly);
			lenp = m_poly_z_faclst_len(pfactors);
			for (j = 0; j < lenp; j++)
				poly_array_append(modpfact, m_poly_z_faclst_factor(pfactors, j));
			mem_delete_hptr(&pfactors); 
			/* We have effectively delreffed each entry already. */
		}
		/* Reference has now been transferred away from all 
		** polynomials in the ddfact faclst, so we can delete the faclst 
		** structure.
		*/
	}
	mem_delete_hptr(&ddfact);
	/* 
	** Now modpfact is a poly_array containing, the complete factorisation of
	** apoly mod pdig.
	*/
	IF_DEBUG_FLAG(DEBUG_POLY,
	{
		cay_print("The complete factorisation mod %d is ...\n", pdig);
		poly_array_write(pring, modpfact);
	}
	);

	/* 
	** Step 3 : Compute the coefficient bound. 
	*/
	a = poly_z_lead_coefft(pring, apoly);
	h = poly_z_sum_norm(pring, apoly);
	b = integer_mult(a, h);
	temp = b;
	b = integer_shift_left(temp, deg/ZETA, deg%ZETA);
	integer_delref(temp);

	M = pdig;
	while (integer_compare(M, b) <= 0)
	{
		temp = M;
		M = integer_mult(pdig, temp);
		integer_delref(temp);
	}
	integer_delref(b);
	integer_delref(a);
	integer_delref(h);
	
	/* Step 4: Lift the factorisation to a mod M factorisation.  */
	modMfact = poly_u_zm_hensel_list(pring, pdig, modpfact, M, apoly);
	poly_array_delete(pring, &modpfact);
	integer_delref(pdig);
	IF_DEBUG_FLAG(DEBUG_POLY,
	{
		cay_print("The power of prime and factorisation is  \n");
		integer_write(M);
		cay_print("\n");
		poly_array_write(pring, modMfact);
	}
	);

	/* Step 5 : Combine the lifted factors to a factorisation over Z. */
	res = poly_u_z_faclst_combine(pring, M, apoly, modMfact, degrees);
	poly_array_delete(pring, &modMfact);
	bit_vector_delete(&degrees);
	integer_delref(M);
	return res;
}

