#ifndef _BLOCK_E_
#define _BLOCK_E_

/*  block.e

Specifications for Blocks with refcounts in Cayley

Most mem blocks in Cayley are "tracked blocks", with a BLOCK TYPE and a
REFERENCE COUNTER.  They start with a header of the following form:
*/

#define BLOCK_REFCOUNT_BITS	16
#define BLOCK_TYPE_BITS		15

#define BLOCK_SHIFT		BLOCK_REFCOUNT_BITS
#define BLOCK_MASK		((1 << BLOCK_SHIFT) - 1)

typedef unsigned int t_block_header;
typedef t_block_header *t_block_p;

/*
Note that direct access to the fields of the above struct should NEVER be made
by Cayley routines which use tracked blocks: the fields are PRIVATE to the
tracked block module.  All access must be through the macros below.
*/

/*
The following gives the length of the tracked block header measured in units of
one t_word.  It must be checked when porting to new machines.
*/

#define BLOCK_HEADER_WORDS	MEM_BYTES_TO_WORDS(sizeof(t_block_header))

/*
The following is the maximum permissible value for a reference counter: it is
used for checking against overflow:
*/

#define	BLOCK_MAX_REFCNT	0xffff

/*
Initialize the type and set the reference counter to one (for use ONLY
immediately after allocation)
*/

extern void block_init P_((t_handle, unsigned int));
extern int block_decref_delete P_((t_handle));
extern t_handle block_copy P_((t_handle, unsigned int));
extern t_handle block_incref P_((t_handle));

/*
Use the usual macros block_decref(h), etc. when you have a handle h; if you
want to do more than one block operation use pointers for efficiency - e.g.

	t_block_p	p;

	p = block_access(h);
	if (block_has_other_refs_p(p))
	    block_decref_p(p);
	else
	    ....
*/

#define block_access(h)		(((t_block_header *)mem_access(h)))

#define block_type_p(p)		(*(p) >> BLOCK_SHIFT)
#define block_incref_p(p)	((*(p))++)
#define block_decref_p(p)	((*(p))--)
#define block_ref_count_p(p)    (*(p) & BLOCK_MASK)
#define block_has_no_refs_p(p)	(block_ref_count_p(p) == 0)
#define block_has_other_refs_p(p) (block_ref_count_p(p) > 1)

#define block_type(h)		block_type_p(block_access(h))
#if 0
#define block_incref(h)		block_incref_p(block_access(h))
#endif
#define block_decref(h)		block_decref_p(block_access(h))
#define block_ref_count(h)	block_ref_count_p(block_access(h))
#define block_has_no_refs(h)	block_has_no_refs_p(block_access(h))
#define block_has_other_refs(h)	block_has_other_refs_p(block_access(h))

/*
Include file listing types in Cayley
*/

#include "block_types.e"

/*
Old-style "stacks", maintained by the Stackhandler, are a special case.
They are indicated by a flag bit in the block type, the remainder of
which is used for special Stackhandler flags.
*/

#define BLOCK_BLK_IS_STK_FLG	0x8000

#define block_is_stack(h)	(block_type(h) & BLOCK_BLK_IS_STK_FLG)


#if 0
#define block_init(h,typ)	\
	{ block_type(h) = typ; block_access(h)->block_rfr_ctr = 1; }
#endif


/*
The following macro defines a temporary variable for use in macros for
reference counters (see integer.e for an example).  This macro should
be used at the start of the body of any function which calls such macros,
so as to define an automatic incarnation of the temporary variable.  Note
that the value of the variable should never carry over from one macro which
uses it to another.
*/


#undef BLOCK_TEMP_VARIABLE
#define BLOCK_TEMP_VARIABLE	\
	You_forgot_block_decl_dummy_This_message_care_of_Count_JS

#define	block_declarations	t_handle	BLOCK_TEMP_VARIABLE
#define	block_decl		block_declarations

#endif /* _BLOCK_E_ */
