#ifndef _INTBIG_INTERNALS_H_

#define _INTBIG_INTERNALS_H_

#if defined(DEVELOP) || defined(DEVELOPMENT)
#undef INTBIG_NO_SANITY_CHECKS
#else
#define INTBIG_NO_SANITY_CHECKS
#endif

/*  intbig_internals.h

Internal private definitions for the multi-precision integer arithmetic
package:  these should not be used from outside the package.

The prefix integer_ indicates part of the public interface; other prefixes
such as intbig_ are used for private parts of the package.
*/

/*
First we consider the radix base, integer_BETA, used for the representation of
large integers.  It is chosen to be the largest power of 2 which is less than
one-third of gamma, where gamma is the smallest positive integer such that
gamma or -gamma (or both) cannot be represented as a machine integer.
Multiplication of multiple-precision integers requires addition of
three terms (two products and a carry); choosing integer_BETA this way
guarantees that such additions can be performed without overflow.

Note that the definition of integer_BETA is MACHINE-DEPENDENT.  For a
32-bit machine with twos complement arithmetic, gamma is 2 to the power
31, so integer_IBETA is two to the power 29.

The value is defined in integer.e.  Inside the package, we abbreviate its
name to BETA.  We allow for the possibility that either or both integer.e
and intbig_internals.h (this file) may be included in .c files.
*/

#define BETA	536870912

#ifdef	integer_BETA
#if	integer_BETA != BETA
ERROR: definitions of BETA and integer_BETA must be identical
#endif
#endif

/*
The following additional constants are used within the package.  They are
also MACHINE-DEPENDENT:

ZETA = log2( BETA )
DELTA = 2 to the power ( floor( ZETA/2 ) )
EPSILON = 2 to the power ( floor( (ZETA+1) / 2 ) )
THETA is the largest power of 10 not greater than BETA
ETA = log10( THETA ) = floor( log10( BETA ) )
IOTA = # of lower bits in a digit
KAPPA = # of higher bits in a digit
EPSILON = 2^IOTA
DELTA = 2^KAPPA
*_MASK = a mask for obtaining the bits covered by *
*/

#define ZETA	29
#define IOTA	15
#define KAPPA	14
#define EPSILON	(1 << IOTA)
#define DELTA	(1 << KAPPA)
#define THETA	100000000
#define ETA	8
#define BETA_MASK (BETA - 1)
#define EPSILON_MASK (EPSILON - 1)
#define DELTA_MASK (DELTA - 1)

/*
Multiple-precision integers are stored in a block as a sequence of digits 
with a sign field which can have the following numbers:
	 0:	the integer is 0 and the length and digits are undefined;
	 1:	the integer is positive;
	-1:	the integer is negative.

If the integer is non-zero, the digits are stored with the least significant
digit first.  Thus the sequence b0, b1, b2, ...  (where for each j,
0 <= bj < BETA) represents the integer

	b0 + b1 * BETA + b2 * BETA*BETA + ...

As well as the t_block_header (type and reference counter), the block has an
additional header entry specifying its current number of beta-digits.
The block also has a "buf index" - if this is -1, the block is simply an
allocated block which is used like any other integer block; it is deleted
after use.  If, however, this index has a value of 0 or more, it is an index
into a fixed array of pre-allocated blocks (called "buffers") which are used
by the inthdl_buf_alloc() and inthdl_buf_delete() macros.  This removes the
overhead of allocating a lot of temporary buffers.

Here is the definition of the header:
*/

typedef
    struct
	{
	    t_block_header		block_header;
	    inthdl_length	intbig_sz;
	    inthdl_sign		intbig_sgn;
#if 0
	    inthdl_buf_index	intbig_bfndx;
#endif
	}
    intbig_header;

/*
After the length field comes the array of beta-digits - the most significant
digit must be non-zero.
The overall structure of the block is:
*/

typedef
    struct
	{
	    intbig_header	intbig_hdr;
	    t_int	intbig_dgt[VARIABLE_LENGTH];
	}
    intbig_struct;

/*
The following gives the length of the header measured in units of one t_word:
*/

#define INTBIG_HDR_LEN	MEM_BYTES_TO_WORDS( sizeof( intbig_header ) )

/*
Tracked block type for blocks storing big integers - the value of this may
be defined by the calling package, for example, in defs.h or trbl.e
which are included before this file:
*/

#ifndef INTBIG_BLOCK_TYPE

#endif

/*
The following gives the number of bh_ints required to store a block containing
n beta_digits:
*/

#define intbig_required_space(n)	(INTBIG_HDR_LEN + (n))


/*
The following gives the current size in bh_ints of the block h:
*/

#define intbig_current_space(h)		mem_words(h)


/*
Macro for accessing individual digits - the first digit is numbered zero:
*/


/* GET A BRAIN SOMEONE!!!!!!!!!!!!!!!!!!!!!!! */

#ifdef DEVELOP

/*  Version with sanity check for debugging   */


#define intbig_digit(h,k) \
(  \
    ((intbig_struct *)mem_access(h))->intbig_dgt[ \
	( (k) < 0 ? intbig_digit_error( h, k ) : (k) )  \
    ] \
)

#else

/*
>>>>> Production <<<<< version 
:-) :-) Did V3 *ever* have a "production" version? - AKS
*/

#define intbig_digit(h,k) \
	(((intbig_struct *)mem_access(h))->intbig_dgt[k])

#endif


/*
A pointer to the first digit, for use in ANY!!!! loops
*/

#define intbig_dig0_ptr(h) (& (((intbig_struct *)mem_access(h))->intbig_dgt[0]) )

/*
The following macros access the fields of the header:
*/

#define intbig_curr_size(h)	(((intbig_header *)mem_access(h))->intbig_sz)
#define intbig_sign(h)		(((intbig_header *)mem_access(h))->intbig_sgn)
#define intbig_buf_index(h)	(((intbig_header *)mem_access(h))->intbig_bfndx)

/*
Although the current digits extend only up to intbig_curr_size(h), there may
be extra room in the block.  The maximum number of digits the block has space
for is given by:
*/

#define intbig_avail_size(h)	(mem_words(h) - INTBIG_HDR_LEN)

/*
Blocks containing big integers returned by the integer_ functions are
often "reduced"; that is,

	mem_reduce_words( h, intbig_required_space( intbig_curr_size(h) ) )

has been applied to them.  This means that intbig_avail_size(h) is often the
same as intbig_curr_size(h): there may in fact be a few extra words.
*/

/*
The following type will be used for intermediate working data: it represents
an integer which fits into a t_word (machine integer) but may or may not be
small enough to be a beta-digit:
*/

typedef t_word	intbig_medium;

/*
The following assures that there is room for at least req beta_digits in integer
block hdl, by extending it as necessary; if extension is required, at least
minext extra beta-digits will be provided (setting minext to a reasonable value,
say 10, cuts down on repeated extensions).  WARNING: Beware of the "dangling
else" problem if this macro is used in an if statement.  The arguments of the
macro are placed in local variables so are only evaluated once each.
*/

#define intbig_assure_space( hdl, req, minext ) \
    {   inthdl_handle h = hdl; \
	inthdl_length r = req, m = minext, a = intbig_avail_size(h); \
	if( r > a ) \
	    mem_extend_words_zero( h, \
		intbig_required_space( ( r > ( a+m ) ) ? r : ( a+m ) ) ); \
    }

/*
The following macro will copy digits from the number with t_handle srchdl,
starting at digit srcpos, into the space referenced by desthdl starting at
position dstpos.  A total of len digits are copied.

Note that srcpos & dstpos are numbered from 0, so to copy the whole block,
set these to zero.

It is permissible for dsthdl to equal srchdl, but ONLY if also dstpos is
equal to srcpos.
*/

/* GET A BRAIN SOMEONE!!!!!!!!!!!!!!!!!!!!!!! */

#ifndef	DEVELOP
#define	intbig_debug_copy_digits(a,b,c,d,e)
#endif

#define intbig_copy_digits( srchdl, srcpos, len, dsthdl, dstpos )	\
{				\
    register t_int	*_p = intbig_dig0_ptr( srchdl ) + srcpos;\
    register t_int	*_q = intbig_dig0_ptr( dsthdl ) + dstpos;\
    register inthdl_length	_k = len;				\
				\
    intbig_debug_copy_digits( srchdl, srcpos, len, dsthdl, dstpos );	\
				\
    if( _q != _p )		\
	while( _k-- > 0 )	\
	    *_q++ = *_p++;	\
}


/*
The following macro will set all betadigits for the given t_handle to zero up
to the supplied length.
*/
#define intbig_set_zero( h, len )		\
{				\
    register t_int	*_p = intbig_dig0_ptr( h );	\
    register inthdl_length	_k = len;			\
				\
    while( _k-- > 0 )		\
	*_p++ = 0;		\
} 


#if 0

/*
Used to be the following drivel to get round the "memory management":

The following macros are to do with the inthdl buffer: inthdl_buf_alloc(len)
returns a t_handle to a block with len digits; the block should be then deleted
by a call to inthdl_buf_delete() with the t_handle passed.

In inthdl_buf_alloc(), if the length is short enough and there is a
free buffer, that buffer is returned; otherwise inthdl_alloc() is called.
In inthdl_buf_delete(), if the t_handle refers to a block which is not a
buffer, the t_handle is simply deleted; otherwise inthdl_buf_delete_func() is
called which adjusts the free list and index accordingly.
*/

#define inthdl_buf_alloc(len)						\
(									\
    (len) <= INTHDL_BUF_DIGITS && inthdl_buf_next_index < INTHDL_TOTAL_BUFS? \
	inthdl_buf[inthdl_buf_next_index++]:				\
	inthdl_alloc(len)						\
)

#define inthdl_buf_delete(h)					\
    (intbig_buf_index(h) >= 0?					\
	    inthdl_buf_delete_func(h):				\
	    (mem_delete_h(h), 0)				\
    )

/*
Return whether a t_handle refers to a buffer (used in inthdl_standardize()):
*/

#define inthdl_is_buf(h)	(intbig_buf_index(h) >= 0)

#endif


#define inthdl_buf_alloc(len)	inthdl_alloc(len)
#define inthdl_buf_delete(h)	mem_delete_h(h)




/*
The following macro, given beta-digits diga and digb, calculates the
unique beta-digits carry and prod such that

	diga * digb = carry*BETA + prod

It is assumed that all digits are positive.

If your machine has reasonably fast floating point arithmetic, define
the preprocessor constant USE_DOUBLE_MULT.
*/

#define USE_DOUBLE_MULT


#ifdef USE_DOUBLE_MULT

extern double BETA_INV;

#define ib_mult(diga, digb, carry, prod)				\
{									\
    register unsigned long _da = (diga), _db = (digb), _p;		\
    *(prod) = (_p = (_da * _db) & BETA_MASK);				\
    *(carry) = 0.25 + BETA_INV * ((double)_da * (double)_db - (double)_p); \
}

#define ib_sqr(diga, carry, prod)					\
{									\
    register unsigned long _da = (diga), _p;				\
    *(prod) = (_p = (_da * _da) & BETA_MASK);				\
    *(carry) = 0.25 + BETA_INV * ((double)_da * (double)_da - (double)_p); \
}

#else

#define ib_mult(diga, digb, carry, prod)				\
{									\
    register unsigned long _da = (diga), _db = (digb), _a1, _a0, _b1, 	\
			   _b0, _chi, _p;				\
    _a1 = _da >> IOTA; _a0 = _da & EPSILON_MASK;			\
    _b1 = _db >> IOTA; _b0 = _db & EPSILON_MASK;			\
    _chi = _a1 * _b0 + _b1 * _a0;					\
    *(prod) = (_p = _a0 * _b0 + ((_chi & DELTA_MASK) << IOTA)) & BETA_MASK; \
    *(carry) = ((_a1*_b1) << (IOTA - KAPPA)) + (_chi >> KAPPA) + (_p >> ZETA); \
}

#define ib_sqr(diga, carry, prod)					\
{									\
    register unsigned long _da = (diga), _a1, _a0, _chi, _p;		\
    _a1 = _da >> IOTA; _a0 = _da & EPSILON_MASK;			\
    _chi = (_a1 * _a0) << 1;						\
    *(prod) = (_p = _a0 * _a0 + ((_chi & DELTA_MASK) << IOTA)) & BETA_MASK; \
    *(carry) = ((_a1*_a1) << (IOTA - KAPPA)) + (_chi >> KAPPA) + (_p >> ZETA); \
}

#endif

/*
The following macro, given *positive* beta-digits a1, a0 and b with a1 < b,
so that b is nonzero and

	a = a1 * BETA + a0

is a two-digit (or one-digit if a1 is zero) beta-representation of an integer
a with a < BETA * b, calculates the quotient q and dividend-signed remainder r
on dividing a by b, that is, the unique integers q and r such that 

	a = q * b + r  with  0 <= r < b (with q >= 0).

It returns q in *qptr and r in *rptr.

Note that a < BETA * b implies (a/b) < BETA, so that q really
can be represented as a single beta-digit.

If your machine has reasonably fast floating point arithmetic, define
the preprocessor constant USE_DOUBLE_QUOT_REM.
*/

#define USE_DOUBLE_QUOT_REM


#ifdef USE_DOUBLE_QUOT_REM

extern double DOUBLE_BETA;

#define ib_quot_rem(a1, a0, b, qptr, rptr)				\
{									\
    long a1_ = (a1), a0_ = (a0), b_ = (b), qq, rr, low, high;		\
    qq = (DOUBLE_BETA * (double)a1_ + (double)a0_) / (double)b_;	\
    ib_mult(qq, b_, &high, &low);					\
    rr = ((a1_ - high) << ZETA) + a0_ - low;				\
    if (rr < 0)								\
	do { qq--; } while ((rr += b_) < 0);				\
    else								\
	while (rr >= b_) rr -= b_, qq++;					\
    *qptr = qq; *rptr = rr;						\
}

#else

#define ib_quot_rem(a1, a0, b, qptr, rptr)	\
	ib_quot_rem_func(a1, a0, b, qptr, rptr)

#endif


#endif /* _INTBIG_INTERNALS_H_ */
