/*******************************************************************************
 
rel_lattice.h
 
AJ March 1992 
Last modification: -


This header file includes structure definitions for lattices
over relative extension and the "enumeration environment" of such a lattice.
 
*******************************************************************************/

#ifndef INCLUDED_REL_LATTICE_H
#define INCLUDED_REL_LATTICE_H
 
#include "kant.h"

/*******************************************************************************
/
/  Introducing new types
/
*******************************************************************************/
 
typedef t_handle  rel_lattice;
typedef t_handle  rel_lat_elt;
typedef t_handle  rel_lat_enum_env;
 
/*
    temporarily (until update of block_type.h)
*/
 
#ifndef REL_LATTICE_BLOCK_TYPE
 
#define REL_LATTICE_BLOCK_TYPE       11567
#define REL_LAT_ELT_BLOCK_TYPE       11568
#define REL_LAT_ENUM_ENV_BLOCK_TYPE  11569
 
#endif  REL_LATTICE_BLOCK_TYPE

  
/*******************************************************************************
/
/  Structure definition for lattices
/
*******************************************************************************/
  
typedef struct
	{
		t_block_header	trbl_hdr;
                                      
                integer_small   rank;   
		                /* rank of the lattice over the subfield      */    

                integer_small   number_con;   
		                /* number of the conjugates of the subfields
			 	   = absolute degree of the subfield          */

		order		coef_ord;
				/* coefficient ring of the lattice	      */

		matrix		coef_ord_gram;
				/* coefficient ring of the lattice	      */

		vector          basis;
				/* basis of the lattice and all conjugates of
				   this basis (vec_length = number_con)       */
                t_handle          basis_ring;                                     
				/* normally the complex numbers               */
 
                vector          gram;
				/* gram matrix of the lattice and all 
				   conjugates of this gram matrix 
				   (vec_length = number_con)                  */
                t_handle          gram_ring;                                      
				/* normally equal to basis_ring               */

                vector          chol;
				/* pos. def. quadr. form of the lattice and  
				   all conjugates of this quadr. form 
				   (vec_length = number_con)                  */
                t_handle          chol_ring;
				/* normally equal to basis_ring               */

                t_handle          disc; 
                t_handle          disc_ring;
				/* normally equal to basis_ring               */
 
	}
	rel_lattice_struct;
 
  
/*******************************************************************************
/
/   Macros for accessing structure items of lattices
/
*******************************************************************************/
 
/*
    general access macro for table element (internal use)
*/
  
#define	rel_lat_access(h)	    ((rel_lattice_struct *) mem_access(h))
 
 
/*
    rank of lattice
*/
 
#define rel_lat_rank(h)		    (rel_lat_access(h) -> rank)
  
                                                                     
/*
    number_con  of lattice
*/
 
#define rel_lat_number_con(h)       (rel_lat_access(h) -> number_con)
  
/*
    coefficient ring of the lattice
    Is the ring known?                       
    Gram-matrix
*/
 
#define rel_lat_coef_ord(h)  	    (rel_lat_access(h) -> coef_ord)
#define rel_lat_coef_ord_known(h)   (rel_lat_coef_ord(h) != MEM_NH)

#define rel_lat_coef_ord_gram(h)    (rel_lat_access(h) -> coef_ord_gram)
 

/*
    vector with the basis of lattice
    coefficient ring of basis
    Is the matrix known?
    Is the basis given over Z?
*/
 
#define rel_lat_basis(h)  	    (rel_lat_access(h) -> basis)
#define rel_lat_basis_con(h,i)      (vec_entry(rel_lat_basis(h),i))
#define rel_lat_basis_ring(h)	    (rel_lat_access(h) -> basis_ring)
#define rel_lat_basis_known(h)      (rel_lat_basis(h) != MEM_NH)
 
   
/*
    vector with Gram matrices
    coefficient ring of Gram matrix
    Is the matrix known?
*/
  
#define rel_lat_gram(h)		    (rel_lat_access(h) -> gram)
#define rel_lat_gram_con(h,i)       (vec_entry(rel_lat_gram(h),i))
#define rel_lat_gram_ring(h)  	    (rel_lat_access(h) -> gram_ring)
#define rel_lat_gram_known(h)       (rel_lat_gram(h) != MEM_NH)
 
 
/*
    vector with the Matrices of Cholesky decompositions
    Is the matrix known?

*/
 
#define rel_lat_chol(h)		    (rel_lat_access(h) -> chol)
#define rel_lat_chol_con(h,i)       (vec_entry(rel_lat_chol(h),i))
#define rel_lat_chol_ring(h)	    (rel_lat_access(h) -> chol_ring)
#define rel_lat_chol_known(h)       (rel_lat_chol(h) != MEM_NH)


/*
    Lattice discriminant
    Ring over which discriminant is defined (at the moment this is the
             same ring as the coefficient ring of the Cholesky matrix)
    Is the Discriminant known?
*/
 
#define rel_lat_disc(h)	 	    (rel_lat_access(h) -> disc)
#define rel_lat_disc_ring(h)	    (rel_lat_access(h) -> disc_ring)
#define rel_lat_disc_known(h)       (rel_lat_disc(h) != MEM_NH)
 
 
  
/*******************************************************************************
/
/  Structure definition for lattices elements
/
*******************************************************************************/
  
typedef struct
	{
		t_block_header	trbl_hdr;
		anf_elt	        rel_lat_elt_coefs[VARIABLE_LENGTH];
	} 	
	rel_lat_elt_struct;
 
/*******************************************************************************
/
/   Macros for accessing structure items of lattice elements
/
*******************************************************************************/

/*   
    length of header (internal use)
    total size of space required by the number (internal use)
*/
  
#define REL_LAT_ELT_HDR_LEN		BLOCK_HEADER_WORDS
#define rel_lat_elt_required_space(k)	((k) + REL_LAT_ELT_HDR_LEN)
   
/*
    accessing 
       ... a certain place within the structure (internal use)
       ... a certain component
*/
  
#define rel_lat_elt_access(e)    \
			((rel_lat_elt_struct *) mem_access((e)))
 
#define rel_lat_elt_coef(e, i) \
	                (rel_lat_elt_access(e) -> rel_lat_elt_coefs[(i)-1])

  
/*
    allocating space
*/
 
#define rel_lat_elt_alloc(e, k)	\
	{ \
		e = (rel_lat_elt) mem_alloc_words_zero(rel_lat_elt_required_space(k)); \
		block_init(e, REL_LAT_ELT_BLOCK_TYPE); \
	}
  
/*                                
    rel_lat_elt-/reference counter maintenance
       - reassigning a lattice element
       - increment reference counter 
       - has the element no references? (internal use)
       - has the element other references? (internal use)
*/
  
#define rel_lat_elt_reassign(lat, a, b)	\
        { rel_lat_elt _rel_lat_elt_temp = a; a = b; \
	  rel_lat_elt_delete(lat, &_rel_lat_elt_temp ); }
 
#define rel_lat_elt_incref(a) 		((rel_lat_elt) block_incref(a))
#define rel_lat_elt_has_no_refs(a)	block_has_no_refs(a)
#define rel_lat_elt_has_other_refs(a)	block_has_other_refs(a)
 
 
/*******************************************************************************
/
/  Structure definition for enumeration environments of lattices
/
*******************************************************************************/
  
typedef struct
	{
		t_block_header	trbl_hdr;
        
                /*  for all calculation we use the ring chol_ring              */
		
                vector          ubound;
				/* upper bounds for the conjugated quad. forms */
 
                integer_small   request;
				/* 1= next vector, 2= shortest vector          */
 
                integer_small   status;     
				/* -1= from beginning, 0= no vector more,
				    1= next vector there                       */

		vector          level_flag;
				/* intern flag for the enumeration             */

		vector          termination_flag;
				/* intern flag for the enumeration             */
 
                rel_lat_elt     act_coefs;
				/* a founded vector with algebraic elements    
				   (representation as an integer vector)       */
                rel_lat_elt     act_coefs_con;
				/* a founded vector with algebraic elements    
				   in conjugated form                          */
                vector          act_length;         
				/* the actual length of all conj. quad. forms  */
 
                vector		part_sums;
				/* partial sum in step i:
				   for each conjugated form one vector         */
                vector		tmp_bounds;
				/* rest length in step i:                      
				   for each conjugated form one vector         */

		vector		coefs_lat;
				/* the real lattices for each  act_coefs       */

		vector		coefs_env;
				/* the real enum. env for each  act_coefs      */

		int		counter;
	}
	rel_lat_enum_env_struct;
 
  
/*******************************************************************************
/
/   Macros for accessing structure items of enumeration environments
/
/   All real values have to refer to the complex field of the Cholesky matrix 
/   of the lattice under consideration.
/
*******************************************************************************/
 
/*
    general access macro for table element (internal use)
*/

#define	rel_lat_enum_access(h)	    ((rel_lat_enum_env_struct *) mem_access(h))

  
/*
    Upper bound 
*/
 
#define rel_lat_enum_ubounds(h)		(rel_lat_enum_access(h) -> ubound)
#define rel_lat_enum_ubounds_con(h,i)	(vec_entry(rel_lat_enum_ubounds(h),i))

/*
    Accessing the request flag 
       ... "shortest vector"
       ... "next vector"
       ... setting to "shortest vector"
       ... setting to "next vector"
*/
 
#define rel_lat_enum_request(h)             (rel_lat_enum_access(h) -> request)
#define rel_lat_enum_request_is_shortest(h)  (rel_lat_enum_request(h) == 1)
#define rel_lat_enum_request_is_next(h)      (rel_lat_enum_request(h) == 2)
#define rel_lat_enum_request_set_shortest(h) (rel_lat_enum_request(h) = 1)
#define rel_lat_enum_request_set_next(h)     (rel_lat_enum_request(h) = 2)
 
 
/*
    Flag for enumeration routines
*/
 
#define rel_lat_enum_status(h)         	(rel_lat_enum_access(h) -> status)
#define rel_lat_enum_status_is_new(h)	(rel_lat_enum_status(h) == -1)
#define rel_lat_enum_status_is_done(h)	(rel_lat_enum_status(h) == 0)
#define rel_lat_enum_status_is_busy(h)	(rel_lat_enum_status(h) == 1)
#define rel_lat_enum_status_set_new(h)	(rel_lat_enum_status(h) = -1) 
#define rel_lat_enum_status_set_done(h)	(rel_lat_enum_status(h) = 0)
#define rel_lat_enum_status_set_busy(h)	(rel_lat_enum_status(h) = 1)

/*
    Intern Flags for enumeration
*/
 
#define rel_lat_enum_level_flag(h)       \
	                           (rel_lat_enum_access(h) -> level_flag)

#define rel_lat_enum_level_flag_level(h,i) \
	                        (vec_entry(rel_lat_enum_level_flag(h),i))

#define rel_lat_enum_termination_flag(h)     \
	          	        (rel_lat_enum_access(h) -> termination_flag)

#define rel_lat_enum_termination_flag_level(h,i) \
	                     (vec_entry(rel_lat_enum_termination_flag(h),i))

                                            
/*
    Accessing the actual vector
       ... its square length in the i-th conjugated form
*/
 
#define rel_lat_enum_act_coefs(h)     (rel_lat_enum_access(h) -> act_coefs)
#define rel_lat_enum_act_coefs_entry(h,i)   \
	              	              (rel_lat_elt_coef(rel_lat_enum_act_coefs(h),i))

#define rel_lat_enum_act_coefs_con(h)  	\
 	                              (rel_lat_enum_access(h) -> act_coefs_con)
#define rel_lat_enum_act_coefs_con_entry(h,i) \
 	                           (rel_lat_elt_coef(rel_lat_enum_act_coefs_con(h),i))

#define rel_lat_enum_act_lengths(h)	(rel_lat_enum_access(h) -> act_length)
#define rel_lat_enum_act_lengths_con(h,i)  \
			  (vec_entry(rel_lat_enum_act_lengths(h), i))
 
/*
    Internal auxiliary arrays 
*/

#define rel_lat_enum_part_sums(h)  	(rel_lat_enum_access(h) -> part_sums)
#define rel_lat_enum_part_sums_con(h,i)      \
		          (vec_entry(rel_lat_enum_part_sums(h), i)) 
			  /* i-th conjugated form: vector with sums (vector) */
#define rel_lat_enum_part_sums_con_level(h,i,j)   \
	  ((t_comp) vec_entry(vec_entry(rel_lat_enum_part_sums(h), i),j))
			  /* i-th conjugated form, j-th sum                  */

   
#define rel_lat_enum_tmp_bounds(h)  	(rel_lat_enum_access(h) -> tmp_bounds) 
#define rel_lat_enum_tmp_bounds_con(h,i)      \
			   (vec_entry(rel_lat_enum_tmp_bounds(h),i)) 
			  /* i-th conjugated form: the rest of upper bound
			     (t_comp)                                       */
#define rel_lat_enum_tmp_bounds_con_level(h,i,j)   \
	  ((t_real) vec_entry(vec_entry(rel_lat_enum_tmp_bounds(h), i),j))
			  /* i-th conjugated form, j-th level               */
                      

#define rel_lat_enum_coefs_lat(h)   \
			  ((vector) (rel_lat_enum_access(h) -> coefs_lat) )
			  /* i-th conjugated coeff.: the approp. lattice    */
#define rel_lat_enum_coefs_lat_level(h,i)  \
			  ((lattice) vec_entry(rel_lat_enum_coefs_lat(h), i)) 
			  /* i-th conjugated coeff.: the approp. lattice    */
    
#define rel_lat_enum_coefs_env(h)   \
			  ((vector) (rel_lat_enum_access(h) -> coefs_env) )
			  /* i-th conjugated coeff.: the approp. lattice    */
#define rel_lat_enum_coefs_env_level(h,i)  \
		  ((lat_enum_env) vec_entry(rel_lat_enum_coefs_env(h), i)) 
			  /* i-th conjugated coeff.: the approp. lattice    */
                                                                              

#define rel_lat_enum_counter(h)		(rel_lat_enum_access(h) -> counter)

/*******************************************************************************
/
/   End
/
*******************************************************************************/
 
#endif INCLUDED_REL_LATTICE_H
