/* mrandom.h 1.9 6/11/92 */
/*
 *		mrandom.h
 *
 *	A wrapper for random(), to allow safe restart.
 *
 *	Original Implementation:
 *		Clark Thomborson, September 1991
 *
 *	Modifications by Clark Thomborson, May 1992
 *	 	- call RNGs other than 4.3bsd random()
 *		- allow multiple RNGs to be active simultaneously
 *
 *	This material is based upon work supported by the National
 *	Science Foundation under grant number MIP-9023238.  The
 *	Government has certain rights in this material.
 *
 *	Any opinions, findings, and conclusions or recommendations
 *	expressed in this material are those of the author and do
 *	not necessarily reflect the view of the National Science
 *	Foundation.
 *
 *	This code is neither copyrighted nor patented.
 */

#define RNGIDSTRLEN 80 /* max length of string written by describe_rng() */

/* max # of long ints in any RNG's state table */
#define MAXRNGTABLESIZE 57

/* Data describing the current RNG */
struct rngdata {
  /* The following data is taken directly from the statefile */
  long	rngalg;	    /* algorithm used to generate pseudorandoms:
		     * 0 = trivial rng (long state=seed1; state += seed2)
		     * 1 = 4.3bsd random.c,
		     * 2 = bentley.c,
		     * 3 = pcrand.c,
		     * 4 = 4.3bsd nrand48.c,
		     * 5 = 4.3bsd rand.c,
		     * other values currently undefined */
  long	rngseed1;    /* the seed originally used to initialize rngstate */
  long	rngseed2;    /* the second seed (if any) */
  long	rngcount1;  /* mod-BILLION counter of random() calls */
  long	rngcount2;  /* div-BILLION counter */
  long	rngnextval; /* random()'s next output */
  /* The next three values are inferred from the rngalg value */
  long	rngstatesize;
  long	rngrangem1; /* one less than the range, so we can use signed longs
		     * (some code efficiency might be gained with unsigned
		     * longs, but I wouldn't trust most compilers on these) */
  double rngrange;  /* the RNG's range, expressed as a d.p. float */
  /* Here is the RNG's state-table (from the statefile) */
  long	rngstate[MAXRNGTABLESIZE];
};
typedef struct rngdata RNGdata;

char *describe_rng (/* RNGdata *rng; char rngid[RNGIDSTRLEN] */);
/* Write a short ASCII description of rng into the user-supplied string
 * rngid, which must be of length at least RNGIDSTRLEN.  If the user has
 * not initialized the rng with init_rng() or restart_rng(), abort with
 * an error message to stderr.  Otherwise return the value of rngid.
 */

int init_rng (/* RNGdata *rng; long alg, seed1, seed2, count1, count2 */);
/* Initialize the general-purpose rng data area so that it holds the
 * state and other data required for the selected algorithm "alg", where
 *	alg = 0 is a trivial generator that returns state += seed2,
 *	  where state is initially set to seed1.  Use for debugging only!
 *	alg = 1 is 4.3bsd random.c (non-linear additive feedback)
 *	alg = 2 is Bentley's remarkable RNG (lagged Fibonnacci)
 *	alg = 3 is the portable combined (multiplicative) RNG
 *	alg = 4 is 4.3bsd nrand48.c (48-bit multiplicative)
 *	alg = 5 is 4.3bsd rand.c, for amusement.  It fails most tests.
 *
 * Note: before calling init_rng(&myrng, ...), the user must allocate space
 * for this generator, for example, with the storage declaration
 *		RNGdata myrng;
 *
 * The seed1 parameter is used by all RNG algorithms to initialize its
 * state table.  The seed2 parameter is only used by algorithms 3 and 4.
 *
 * The count1 parameter indicates how many times the selected RNG algorithm
 * should be called prior to returning control to the calling routine.
 * We recommend a high value, at least 10000, for this parameter, to minimize
 * the ill effects of seeding.  The count2 parameter can be given a non-zero
 * value if you wish to "cycle" the generator a huge number of times: it 
 * will be called (count2*1e9 + count1) times.  Probably count2 should always
 * be 0 until computers become much, much faster than today.
 *
 * init_rng() returns 1 unless an out-of-range argument is detected
 * (rngalg >4 or <0; count1 >1e9 or <0; or count2 <0), in which case
 * it returns 0.
 *
 * Note: a single program may call init_rng() any number of times, to set up
 * multiple generators (possibly using more than one RNG algorithm), e.g with 
 *		RNGdata myrngs[8];
 *		long i,sum=0;
 *		for (i=0; i<7; i++) {
 *		  /* generator #i gets seed1 = i
 *		  init_rng(&myrngs[i],2,i,0,100000,0);
 *		}
 *		/* our eighth RNG uses algorithm #3, just for kicks
 *		init_rng(&myrngs[7],3,7,0,100000,0);
 *		/* add 1-bit numbers, one from each generator
 *		for (i=0; i<7; i++) {
 *		  sum += mrandom(&myrngs[i],2);
 *		}
 *
 * Warning: do not attempt to use multiple rngdata areas for algorithm #1.
 * The 4.3bsd random.c code has internal state that will not be modified
 * correctly when you switch generators (this was a bug in the original
 * implementation and it would be very difficult to fix here).
 * 
 * We recommend that init_rng() be used very sparingly.  Except when
 * replicating experiments or debugging, it is better to restart an
 * existing generator (stored in a statefile) than to initialize a new one.
 */

int restart_rng (/* RNGdata *rng; char *filename; */);
/* Restart a generator from a statefile, copying data into the user-supplied
 * rng data area.  Print a message on stderr and return 0 if the restart
 * failed due to a garbled or nonexistent statefile.  Otherwise return 1.
 */

double frandomrv (/* RNGdata *rng; long n; double v[]; */);
/* Generate a length-n vector v[] of uniformly-distributed variates, where
 * 0.0 <= v[i] < 1.0 for i=0,...n-1.  Return the value of v[0]. 
 * 
 * Special-case parameter values: if rng==0, use the most recently
 * initialized or restarted RNG; if n==0, return one random variate
 * and don't write into v[].
 */

# define  frandomr(rng)	frandomrv(rng,0,0)
/* Like frandomrv(), except return just one uniform variate in the range
 * [0.0, 1.0).
 */

# define  frandom()	frandomrv(0,0,0)
/* Like frandomr(rng), except use the RNG that was most recently initialized
 * or restarted.  Abort with an error message if no RNG has been initialized
 * or restarted.
 */ 

long mrandomrv (/* RNGdata *rng; long m,n,v[]; */);
/* Generate a length-n vector v of random longs, uniformly distributed
 * in the range 0..m-1, using the indicated rng.  Return a copy of the
 * first random variate, v[0].
 *
 * Special-case parameter values: if rng==0, use the RNG that was
 * the most-recently initialized or restarted; if n==0, return one
 * random variate and don't write into v[]. 
 *
 * Our code does not have a deterministic bias for any m, unlike the
 * typical "good" code
 *		(long) floor( frandom() * (double) m )
 * or the commonly-used, but hazardous (because it exposes the flaws
 * in many RNGs) code 
 *		random()%m
 * We remove the bias by making multiple calls (on rare occasions)
 * to the underlying generator.  The expected number of RNG calls
 * is upper-bounded by n/(1 - (RNGrange%m)/RNGrange) < 2n.
 * 
 * The program is aborted, with an error message, if m == 0 or if
 * m exceeds the range of the RNG.  Note: m is treated as an unsigned long,
 * so mrandomr(rng,1<<31) returns a full-range (31-bit) integer if the
 * currently-selected RNG has a 31-bit range.
 *
 * The program will also abort, again with an error message to stderr,
 * if the generator is behaving so non-randomly that our multiple-call
 * bias-removal algorithm makes an excessive number of calls to the
 * underlying generator.
 */

# define  mrandomr(rng,m)	mrandomrv(rng,m,0,0)
/* Like mrandomrv(rng,n,v), except returns just one uniform variate in
 * the range 0..m-1.
 */

# define  mrandom(m)	mrandomrv(0,m,0,0)
/* Like mrandomr(rng,m), except use the RNG that was most recently initialized
 * or restarted.  Abort with an error message if no RNG has been initialized
 * or restarted.
 */

void xrandomrv(/* RNGdata rng; long n,v[]; */);
/* Call the appropriate RNG n times, keeping track of the number of calls.
 * Writes a vector v[] of the long ints returned by these RNG calls.
 */

int save_rng(/* RNGdata *rng; char *filename; */);
/* Save the RNG data to a statefile.
 * Check to be sure the RNG can be restarted by calling restart_rng().
 * Return 0 if a problem is detected, printing an error message on stderr.
 * Otherwise return 1.
 */

/* end mrandom.h */
