/*
This code is copyright (C) 1998 Robert O'Callahan.
This code is free for non-commercial use. Modification in all forms is permitted.
This license continues to apply to any modified versions. This license text must be
reproduced and distributed with any modified versions.
As a matter of courtesy I (Robert O'Callahan) would like to be informed of
any potentially useful modifications.
*/

#include "ttssh.h"

#define WM_SOCK_ACCEPT (WM_APP+9999)
#define WM_SOCK_IO     (WM_APP+9998)
#define WM_SOCK_GOTNAME (WM_APP+9997)

#define CHANNEL_READ_BUF_SIZE 30000

static int find_request_num(struct _TInstVar FAR * pvar, SOCKET s) {
  int i;

  if (s == INVALID_SOCKET) return -1;

  for (i = 0; i < pvar->fwd_state.num_requests; i++) {
    if (pvar->fwd_state.requests[i].listening_socket == s) {
      return i;
    }
  }

  return -1;
}

static int find_channel_num(struct _TInstVar FAR * pvar, SOCKET s) {
  int i;

  if (s == INVALID_SOCKET) return -1;

  for (i = 0; i < pvar->fwd_state.num_channels; i++) {
    if (pvar->fwd_state.channels[i].local_socket == s) {
      return i;
    }
  }

  return -1;
}

static int find_request_num_from_async_request(struct _TInstVar FAR * pvar, HANDLE request) {
  int i;

  if (request == 0) return -1;

  for (i = 0; i < pvar->fwd_state.num_requests; i++) {
    if (pvar->fwd_state.requests[i].to_host_lookup_handle == request) {
      return i;
    }
  }

  return -1;
}

static int check_local_channel_num(struct _TInstVar FAR * pvar, int local_num) {
  if (local_num < 0 || local_num >= pvar->fwd_state.num_channels
    || pvar->fwd_state.channels[local_num].status == 0) {
    notify_nonfatal_error(pvar, "The server attempted to manipulate a forwarding channel that does not exist.\n"
      "Either the server has a bug or is hostile. You should close this connection.");
    return 0;
  } else {
    return 1;
  }
}

static void request_error(struct _TInstVar FAR * pvar, int request_num, int err) {
  SOCKET s = pvar->fwd_state.requests[request_num].listening_socket;

  if (s != INVALID_SOCKET) {
    closesocket(s);
    pvar->fwd_state.requests[request_num].listening_socket = INVALID_SOCKET;
  }

  notify_nonfatal_error(pvar, "Communications error while listening for a connection to forward.\n"
    "The listening port will be terminated.");
}

static void send_local_connection_closure(struct _TInstVar FAR * pvar, int channel_num) {
  FWDChannel FAR * channel = pvar->fwd_state.channels + channel_num;

  if ((channel->status & (FWD_REMOTE_CONNECTED | FWD_LOCAL_CONNECTED))
    == (FWD_REMOTE_CONNECTED | FWD_LOCAL_CONNECTED))  {
    SSH_channel_input_eof(pvar, channel->remote_num);
    SSH_channel_output_eof(pvar, channel->remote_num);
    channel->status |= FWD_CLOSED_LOCAL_IN | FWD_CLOSED_LOCAL_OUT;
  }
}

static void closed_local_connection(struct _TInstVar FAR * pvar, int channel_num) {
  FWDChannel FAR * channel = pvar->fwd_state.channels + channel_num;

  if (channel->local_socket != INVALID_SOCKET) {
    closesocket(channel->local_socket);
    channel->local_socket = INVALID_SOCKET;

    send_local_connection_closure(pvar, channel_num);
  }
}

static void free_channel(struct _TInstVar FAR * pvar, int i) {
  FWDChannel FAR * channel = &pvar->fwd_state.channels[i];

  channel->status = 0;
  if (channel->local_socket != INVALID_SOCKET) {
    closesocket(channel->local_socket);
    channel->local_socket = INVALID_SOCKET;
  }
  channel->request_num == -1;
}

void FWD_channel_input_eof(struct _TInstVar FAR * pvar, uint32 local_channel_num) {
  FWDChannel FAR * channel;

  if (!check_local_channel_num(pvar, local_channel_num)) return;

  channel = pvar->fwd_state.channels + local_channel_num;

  if (channel->local_socket != INVALID_SOCKET) {
    shutdown(channel->local_socket, 1);
  }
  channel->status |= FWD_CLOSED_REMOTE_IN;
  if ((channel->status & FWD_CLOSED_REMOTE_OUT) == FWD_CLOSED_REMOTE_OUT) {
    closed_local_connection(pvar, local_channel_num);
    free_channel(pvar, local_channel_num);
  }
}

void FWD_channel_output_eof(struct _TInstVar FAR * pvar, uint32 local_channel_num) {
  FWDChannel FAR * channel;

  if (!check_local_channel_num(pvar, local_channel_num)) return;

  channel = pvar->fwd_state.channels + local_channel_num;

  if (channel->local_socket != INVALID_SOCKET) {
    shutdown(channel->local_socket, 0);
  }
  channel->status |= FWD_CLOSED_REMOTE_OUT;
  if ((channel->status & FWD_CLOSED_REMOTE_IN) == FWD_CLOSED_REMOTE_IN) {
    closed_local_connection(pvar, local_channel_num);
    free_channel(pvar, local_channel_num);
  }
}

static void channel_error(struct _TInstVar FAR * pvar, int channel_num, int err) {
  closed_local_connection(pvar, channel_num);
  notify_nonfatal_error(pvar, "Communications error while reading from a forwarded local port.\n"
    "The forwarded connection will be closed.");
}

static void channel_opening_error(struct _TInstVar FAR * pvar, int channel_num, int err) {
  char buf[1024];

  SSH_fail_channel_open(pvar, pvar->fwd_state.channels[channel_num].remote_num);
  _snprintf(buf, sizeof(buf), "The server attempted to forward a connection through this machine.\n"
    "It requested a connection to machine %s.\n"
    "The machine could not be found. The forwarded connection will be closed.",
    pvar->fwd_state.requests[pvar->fwd_state.channels[channel_num].request_num].to_host);
  notify_nonfatal_error(pvar, buf);
  free_channel(pvar, channel_num);
}

static int validate_IP_number(struct _TInstVar FAR * pvar, uint32 addr) {
  int i;

  if (pvar->fwd_state.local_host_IP_numbers == NULL) {
    static char loopback[] = { 127, 0, 0, 1 };
    HOSTENT FAR * hostent = gethostbyaddr(loopback, 4, AF_INET);

    for (i = 0; hostent->h_addr_list[i] != NULL; i++) {
    }

    pvar->fwd_state.local_host_IP_numbers = (uint32 FAR *)malloc(sizeof(uint32)*(i + 1));
    
    for (i = 0; hostent->h_addr_list[i] != NULL; i++) {
      pvar->fwd_state.local_host_IP_numbers[i] = ntohl(*(uint32 FAR *)hostent->h_addr_list[i]);
    }
    pvar->fwd_state.local_host_IP_numbers[i] = 0;
  }

  for (i = 0; pvar->fwd_state.local_host_IP_numbers[i] != 0; i++) {
    if (pvar->fwd_state.local_host_IP_numbers[i] == addr) {
      return 1;
    }
  }

  return 0;
}

static int alloc_channel(struct _TInstVar FAR * pvar) {
  int i, new_num_channels;

  for (i = 0; i < pvar->fwd_state.num_channels; i++) {
    if (pvar->fwd_state.channels[i].status == 0) {
      return i;
    }
  }

  new_num_channels = pvar->fwd_state.num_channels*2 + 1;
  pvar->fwd_state.channels = (FWDChannel FAR *)realloc(pvar->fwd_state.channels,
    sizeof(FWDChannel)*new_num_channels);

  for (; i < new_num_channels; i++) {
    FWDChannel FAR * channel = pvar->fwd_state.channels + i;

    channel->status = 0;
    channel->local_socket = INVALID_SOCKET;
    channel->request_num = -1;
  }
  i = pvar->fwd_state.num_channels;
  pvar->fwd_state.num_channels = new_num_channels;
  return i;
}

static void accept_local_connection(struct _TInstVar FAR * pvar, int request_num) {
  int channel_num;
  SOCKET s;
  struct sockaddr addr;
  int addrlen = sizeof(addr);
  char buf[1024];
  char FAR * IP;

  s = accept(pvar->fwd_state.requests[request_num].listening_socket, &addr, &addrlen);
  if (s == INVALID_SOCKET) return;

  IP = (char FAR *)&((struct sockaddr_in *)(&addr))->sin_addr.s_addr;

  if (!validate_IP_number(pvar, htonl(((struct sockaddr_in *)(&addr))->sin_addr.s_addr))) {
    _snprintf(buf, sizeof(buf), "Host with IP number %d.%d.%d.%d tried to connect to a forwarded local port.\n"
      "This could be some kind of hostile attack.",
      IP[0], IP[1], IP[2], IP[3]);
    notify_nonfatal_error(pvar, buf);
    closesocket(s);
    return;
  }

  _snprintf(buf, sizeof(buf), "%d.%d.%d.%d", IP[0], IP[1], IP[2], IP[3]);

  channel_num = alloc_channel(pvar);
  pvar->fwd_state.channels[channel_num].status = FWD_LOCAL_CONNECTED;
  pvar->fwd_state.channels[request_num].request_num = request_num;
  pvar->fwd_state.channels[channel_num].local_socket = s;

  SSH_open_channel(pvar, channel_num, pvar->fwd_state.requests[request_num].to_host,
    pvar->fwd_state.requests[request_num].to_port, buf);
}

static void connected_local_connection(struct _TInstVar FAR * pvar, int channel_num) {
  SSH_confirm_channel_open(pvar, pvar->fwd_state.channels[channel_num].remote_num,
    channel_num);
  pvar->fwd_state.channels[channel_num].status |= FWD_LOCAL_CONNECTED;
}

static void make_local_connection(struct _TInstVar FAR * pvar, int channel_num) {
  FWDChannel FAR * channel = pvar->fwd_state.channels + channel_num;
  FWDRequest FAR * request = pvar->fwd_state.requests + channel->request_num;
  struct sockaddr_in addr;
  
  addr.sin_family = AF_INET;
  addr.sin_port = htons((unsigned short)request->to_port);
  addr.sin_addr.s_addr = htonl(request->to_host_addr);

  if ((channel->local_socket = socket(AF_INET, SOCK_STREAM, 0)) != INVALID_SOCKET
    && WSAAsyncSelect(channel->local_socket, pvar->fwd_state.accept_wnd, WM_SOCK_IO,
      FD_CONNECT | FD_READ | FD_CLOSE) != SOCKET_ERROR) {
    if (connect(channel->local_socket, (struct sockaddr FAR *)&addr, sizeof(addr)) != SOCKET_ERROR) {
      connected_local_connection(pvar, channel_num);
      return;
    } else if (WSAGetLastError() == WSAEWOULDBLOCK) {
      /* do nothing, we'll just wait */
      return;
    }
  }

  channel_opening_error(pvar, channel_num, WSAGetLastError());
}

static void read_local_connection(struct _TInstVar FAR * pvar, int channel_num) {
  FWDChannel FAR * channel = pvar->fwd_state.channels + channel_num;

  if ((channel->status & (FWD_REMOTE_CONNECTED | FWD_LOCAL_CONNECTED))
    != (FWD_REMOTE_CONNECTED | FWD_LOCAL_CONNECTED))  {
    return;
  }

  while (channel->local_socket != INVALID_SOCKET) {
    char buf[CHANNEL_READ_BUF_SIZE];
    int amount = recv(channel->local_socket, buf, sizeof(buf), 0);
    int err;
    
    if (amount > 0) {
      if ((channel->status & FWD_CLOSED_REMOTE_OUT) == 0) {
        SSH_channel_send(pvar, channel->remote_num, buf, amount);
      }
    } else if (amount == 0 || (err = WSAGetLastError()) == WSAEWOULDBLOCK) {
      return;
    } else {
      channel_error(pvar, channel_num, err);
      return;
    }
  }
}

static void failed_to_host_addr(struct _TInstVar FAR * pvar, int request_num, int err) {
  int i;

  for (i = 0; i < pvar->fwd_state.num_channels; i++) {
    if (pvar->fwd_state.channels[i].request_num == request_num) {
      channel_opening_error(pvar, i, err);
    }
  }
}

static void found_to_host_addr(struct _TInstVar FAR * pvar, int request_num) {
  int i;

  pvar->fwd_state.requests[request_num].to_host_addr =
    ntohl(*(uint32 FAR *)((HOSTENT FAR *)pvar->fwd_state.requests[request_num].to_host_hostent_buf)->h_addr_list[0]);

  for (i = 0; i < pvar->fwd_state.num_channels; i++) {
    if (pvar->fwd_state.channels[i].request_num == request_num) {
      make_local_connection(pvar, i);
    }
  }
}

static LRESULT CALLBACK accept_wnd_proc(HWND wnd, UINT msg, WPARAM wParam, LPARAM lParam) {
  struct _TInstVar FAR * pvar = (struct _TInstVar FAR *)GetWindowLong(wnd, GWL_USERDATA);

  if (msg == WM_SOCK_ACCEPT
    && (LOWORD(lParam) == FD_READ || LOWORD(lParam) == FD_CLOSE)) {
    msg = WM_SOCK_IO;
  }

  switch (msg) {
  case WM_SOCK_ACCEPT: {
    int request_num = find_request_num(pvar, (SOCKET)wParam);

    if (request_num < 0) return TRUE;

    if (HIWORD(lParam) != 0) {
      request_error(pvar, request_num, HIWORD(lParam));
    } else {
      switch (LOWORD(lParam)) {
      case FD_ACCEPT:
        accept_local_connection(pvar, request_num);
        break;
      }
    }
    return TRUE;
  }

  case WM_SOCK_GOTNAME: {
    int request_num = find_request_num_from_async_request(pvar, (HANDLE)wParam);

    if (request_num < 0) return TRUE;

    if (HIWORD(lParam) != 0) {
      failed_to_host_addr(pvar, request_num, HIWORD(lParam));
    } else {
      found_to_host_addr(pvar, request_num);
    }
    pvar->fwd_state.requests[request_num].to_host_lookup_handle = 0;
    free(pvar->fwd_state.requests[request_num].to_host_hostent_buf);
    pvar->fwd_state.requests[request_num].to_host_hostent_buf = NULL;
    return TRUE;
  }

  case WM_SOCK_IO: {
    int channel_num = find_channel_num(pvar, (SOCKET)wParam);

    if (channel_num < 0) return TRUE;

    if (HIWORD(lParam) != 0) {
      if (LOWORD(lParam) == FD_CONNECT) {
        channel_opening_error(pvar, channel_num, HIWORD(lParam));
      } else {
        channel_error(pvar, channel_num, HIWORD(lParam));
      }
    } else {
      switch (LOWORD(lParam)) {
      case FD_CONNECT:
        connected_local_connection(pvar, channel_num);
        break;
      case FD_READ:
        read_local_connection(pvar, channel_num);
        break;
      case FD_CLOSE:
        read_local_connection(pvar, channel_num);
        closed_local_connection(pvar, channel_num);
        break;
      }
    }
    return TRUE;
  }
  }

  return CallWindowProc(pvar->fwd_state.old_accept_wnd_proc, wnd, msg, wParam, lParam);
}

void FWD_prep_forwarding(struct _TInstVar FAR * pvar) {
  int i;

  for (i = 0; i < pvar->fwd_state.num_requests; i++) {
    if (pvar->fwd_state.requests[i].type == FWD_REMOTE_TO_LOCAL) {
      SSH_request_forwarding(pvar, pvar->fwd_state.requests[i].from_port,
        pvar->fwd_state.requests[i].to_host, pvar->fwd_state.requests[i].to_port);
    } else if (pvar->fwd_state.requests[i].type == FWD_REMOTE_X11_TO_LOCAL) {
      char FAR * auth_protocol = getenv("TTSSH_XAUTH_PROTOCOL");
      char FAR * auth_data = getenv("TTSSH_XAUTH_DATA");

      if (auth_protocol == NULL) {
        notify_nonfatal_error(pvar,
          "Environment variable TTSSH_XAUTH_PROTOCOL must be set to use X11 forwarding.");
      } else if (auth_data == NULL) {
        notify_nonfatal_error(pvar,
          "Environment variable TTSSH_XAUTH_DATA must be set to use X11 forwarding.");
      } else {
        SSH_request_X11_forwarding(pvar, auth_protocol, auth_data, 0);
      }
    }
  }
}

void FWD_enter_interactive_mode(struct _TInstVar FAR * pvar) {
  int i;

  for (i = 0; i < pvar->fwd_state.num_requests; i++) {
    FWDRequest FAR * request = pvar->fwd_state.requests + i;

    if (request->type == FWD_LOCAL_TO_REMOTE) {
      SOCKET s = socket(AF_INET, SOCK_STREAM, 0);
      struct sockaddr_in addr;

      addr.sin_family = AF_INET;
      addr.sin_port = htons((unsigned short)request->from_port);
      addr.sin_addr.s_addr = htonl(INADDR_ANY);

      request->listening_socket = s;
      if (s == INVALID_SOCKET
        || WSAAsyncSelect(s, pvar->fwd_state.accept_wnd, WM_SOCK_ACCEPT, FD_ACCEPT | FD_READ | FD_CLOSE) == SOCKET_ERROR
        || bind(s, (struct sockaddr FAR *)&addr, sizeof(addr)) == SOCKET_ERROR
        || listen(s, SOMAXCONN) == SOCKET_ERROR) {
        notify_nonfatal_error(pvar, "Some socket(s) required for port forwarding could not be initialized.\n"
          "Some port forwarding services may not be available.");
        return;
      }
    }
  }
}

void FWD_open(struct _TInstVar FAR * pvar, uint32 remote_channel_num,
              char FAR * local_hostname, int local_port, char FAR * originator, int originator_len) {
  int i;
  char buf[1024];

  for (i = 0; i < pvar->fwd_state.num_requests; i++) {
    FWDRequest FAR * request = pvar->fwd_state.requests + i;

    if (request->type == FWD_REMOTE_TO_LOCAL
      && request->to_port == local_port
      && strcmp(request->to_host, local_hostname) == 0) {
      int channel_num;

      if (request->to_host_addr == 0
        && request->to_host_lookup_handle == 0) {
        HANDLE task_handle;
        
        if (request->to_host_hostent_buf == NULL) {
          request->to_host_hostent_buf = (char FAR *)malloc(MAXGETHOSTSTRUCT);
        }
        
        task_handle = 
          WSAAsyncGetHostByName(pvar->fwd_state.accept_wnd, WM_SOCK_GOTNAME,
          request->to_host, request->to_host_hostent_buf, MAXGETHOSTSTRUCT);
        
        if (task_handle == 0) {
          SSH_fail_channel_open(pvar, remote_channel_num);
          _snprintf(buf, sizeof(buf), "The server attempted to forward a connection through this machine.\n"
            "It requested a connection to machine %s on port %d.\n"
            "An error occurred while processing the request, and it has been denied.",
            local_hostname, local_port);
          notify_nonfatal_error(pvar, buf);
          free(request->to_host_hostent_buf);
          request->to_host_hostent_buf = NULL;
          return;
        } else {
          request->to_host_lookup_handle = task_handle;
        }
      }
      
      channel_num = alloc_channel(pvar);
      pvar->fwd_state.channels[channel_num].status = FWD_REMOTE_CONNECTED;
      pvar->fwd_state.channels[channel_num].request_num = i;
      pvar->fwd_state.channels[channel_num].remote_num = remote_channel_num;

      if (request->to_host_addr != 0) {
        make_local_connection(pvar, channel_num);
      }

      return;
    }
  }

  /* this forwarding was not prespecified */
  SSH_fail_channel_open(pvar, remote_channel_num);
  _snprintf(buf, sizeof(buf), "The server attempted to forward a connection through this machine.\n"
    "It requested a connection to machine %s on port %d.\n"
    "You did not specify this forwarding to TTSSH in advance, and therefore the request was denied.",
    local_hostname, local_port);
  notify_nonfatal_error(pvar, buf);
}

void FWD_X11_open(struct _TInstVar FAR * pvar, uint32 remote_channel_num,
  char FAR * originator, int originator_len) {
  int i;

  for (i = 0; i < pvar->fwd_state.num_requests; i++) {
    FWDRequest FAR * request = pvar->fwd_state.requests + i;

    if (request->type == FWD_REMOTE_X11_TO_LOCAL) {
      int channel_num = alloc_channel(pvar);

      pvar->fwd_state.channels[channel_num].status = FWD_REMOTE_CONNECTED;
      pvar->fwd_state.channels[channel_num].request_num = i;
      pvar->fwd_state.channels[channel_num].remote_num = remote_channel_num;

      make_local_connection(pvar, channel_num);

      return;
    }
  }

  /* this forwarding was not prespecified */
  SSH_fail_channel_open(pvar, remote_channel_num);
  notify_nonfatal_error(pvar, "The server attempted to forward a connection through this machine.\n"
    "It requested a connection to the local X server.\n"
    "You did not specify this forwarding to TTSSH in advance, and therefore the request was denied.");
}

void FWD_confirmed_open(struct _TInstVar FAR * pvar, uint32 local_channel_num,
                        uint32 remote_channel_num) {
  SOCKET s;
  FWDChannel FAR * channel;

  if (!check_local_channel_num(pvar, local_channel_num)) return;

  channel = pvar->fwd_state.channels + local_channel_num;
  s = channel->local_socket;
  if (s != INVALID_SOCKET) {
    channel->remote_num = remote_channel_num;
    channel->status |= FWD_REMOTE_CONNECTED;

    read_local_connection(pvar, local_channel_num);
  } else {
    SSH_channel_input_eof(pvar, remote_channel_num);
    SSH_channel_output_eof(pvar, remote_channel_num);
    channel->status |= FWD_CLOSED_LOCAL_IN | FWD_CLOSED_LOCAL_OUT;
  }
}

void FWD_failed_open(struct _TInstVar FAR * pvar, uint32 local_channel_num) {
  if (!check_local_channel_num(pvar, local_channel_num)) return;

  notify_nonfatal_error(pvar, "A program on the local machine attempted to connect to a forwarded port.\n"
    "The forwarding request was denied by the server. The connection has been closed.");
  free_channel(pvar, local_channel_num);
}

void FWD_received_data(struct _TInstVar FAR * pvar, uint32 local_channel_num,
                       unsigned char FAR * data, int length) {
  SOCKET s;
  u_long do_block = 0;
  FWDChannel FAR * channel;

  if (!check_local_channel_num(pvar, local_channel_num)) return;

  channel = pvar->fwd_state.channels + local_channel_num;
  s = channel->local_socket;
  if (s != INVALID_SOCKET) {
    if ((pvar->PWSAAsyncSelect)(s, pvar->fwd_state.accept_wnd, 0, 0) == SOCKET_ERROR
        || ioctlsocket(s, FIONBIO, &do_block) == SOCKET_ERROR
        || (pvar->Psend)(s, data, length, 0) != length
        || (pvar->PWSAAsyncSelect)(s, pvar->fwd_state.accept_wnd,
             WM_SOCK_ACCEPT, FD_READ | FD_CLOSE) == SOCKET_ERROR) {
      closed_local_connection(pvar, local_channel_num);
      notify_nonfatal_error(pvar, "A communications error occurred while sending forwarded data to a local port.\n"
        "The forwarded connection will be closed.");
    }
  }
}

static void parse_request(FWDRequest FAR * request, char FAR * str) {
  char FAR * host_start;

  request->listening_socket = INVALID_SOCKET;
  request->to_host_addr = 0;
  request->to_host_hostent_buf = NULL;
  request->to_host_lookup_handle = 0;

  if (str[0] == 'L' || str[0] == 'l') {
    request->type = FWD_LOCAL_TO_REMOTE;
  } else if (str[0] == 'R' || str[0] == 'r') {
    request->type = FWD_REMOTE_TO_LOCAL;
  } else if (str[0] == 'X' || str[0] == 'x') {
    request->type = FWD_REMOTE_X11_TO_LOCAL;
    request->to_host_addr = INADDR_LOOPBACK;
    request->to_port = 6000;
    return;
  } else {
    request->type = 0;
    return;
  }
  str++;

  request->from_port = atoi(str);
  while (*str >= '0' && *str <= '9') {
    str++;
  }

  if (*str != ':') {
    request->type = 0;
    return;
  }
  str++;

  host_start = str;
  while (*str != ':' && *str != 0 && *str != ';') {
    str++;
  }
  if (*str != ':') {
    request->type = 0;
    return;
  }
  *str = 0;
  strncpy(request->to_host, host_start, sizeof(request->to_host));
  *str = ':';
  str++;

  request->to_port = atoi(str);
  while (*str >= '0' && *str <= '9') {
    str++;
  }
  if (*str != ';' && *str != 0) {
    request->type = 0;
    return;
  }

  request->listening_socket = INVALID_SOCKET;
}

void FWD_init(struct _TInstVar FAR * pvar) {
  if (pvar->session_settings.DefaultForwarding[0] == 0) {
    pvar->fwd_state.requests = NULL;
    pvar->fwd_state.num_requests = 0;
  } else {
    int i, ch, j;

    j = 1;
    for (i = 0; (ch = pvar->session_settings.DefaultForwarding[i]) != 0; i++) {
      if (ch == ';') {
        j++;
      }
    }

    pvar->fwd_state.num_requests = j;
    pvar->fwd_state.requests = (FWDRequest FAR *)malloc(sizeof(FWDRequest)*j);

    parse_request(pvar->fwd_state.requests, pvar->session_settings.DefaultForwarding);
    j = 1;
    for (i = 0; (ch = pvar->session_settings.DefaultForwarding[i]) != 0; i++) {
      if (ch == ';') {
        parse_request(pvar->fwd_state.requests + j,
          pvar->session_settings.DefaultForwarding + i + 1);
        j++;
      }
    }
  }

  pvar->fwd_state.num_channels = 0;
  pvar->fwd_state.channels = NULL;
  pvar->fwd_state.local_host_IP_numbers = NULL;

  if (pvar->fwd_state.num_requests > 0) {
    pvar->fwd_state.accept_wnd = CreateWindow("STATIC", "TTSSH Port Forwarding Monitor",
      WS_DISABLED | WS_POPUP, 0, 0, 1, 1, NULL, NULL, hInst, NULL);
    if (pvar->fwd_state.accept_wnd != NULL) {
      pvar->fwd_state.old_accept_wnd_proc = (WNDPROC)SetWindowLong(pvar->fwd_state.accept_wnd, GWL_WNDPROC, (LONG)accept_wnd_proc);
      SetWindowLong(pvar->fwd_state.accept_wnd, GWL_USERDATA, (LONG)pvar);
    }
  } else {
    pvar->fwd_state.accept_wnd = NULL;
  }
}

void FWD_end(struct _TInstVar FAR * pvar) {
  int i;

  if (pvar->fwd_state.accept_wnd != NULL) {
    DestroyWindow(pvar->fwd_state.accept_wnd);
  }

  if (pvar->fwd_state.requests != NULL) {
    for (i = 0; i < pvar->fwd_state.num_requests; i++) {
      FWDRequest FAR * request = pvar->fwd_state.requests + i;
     
      if (request->listening_socket != INVALID_SOCKET) {
        closesocket(request->listening_socket);
      }
      if (request->to_host_hostent_buf != NULL) {
        free(request->to_host_hostent_buf);
      }
      if (request->to_host_lookup_handle != 0) {
        WSACancelAsyncRequest(request->to_host_lookup_handle);
      }
    }
    free(pvar->fwd_state.requests);
  }

  if (pvar->fwd_state.channels != NULL) {
    for (i = 0; i < pvar->fwd_state.num_channels; i++) {
      free_channel(pvar, i);
    }
    free(pvar->fwd_state.channels);
  }
  
  if (pvar->fwd_state.local_host_IP_numbers != NULL) {
    free(pvar->fwd_state.local_host_IP_numbers);
  }
}
