/**
   SSLContext.java

   Copyright (C) 1999, Claymore Systems, Inc.
   All Rights Reserved.

   ekr@rtfm.com  Tue May 18 09:43:47 1999

   This package is a SSLv3/TLS implementation written by Eric Rescorla
   <ekr@rtfm.com> and licensed by Claymore Systems, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All advertising materials mentioning features or use of this software
      must display the following acknowledgement:
      This product includes software developed by Claymore Systems, Inc.
   4. Neither the name of Claymore Systems, Inc. nor the name of Eric
      Rescorla may be used to endorse or promote products derived from this
      software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.

   $Id: SSLContext.java,v 1.4 1999/06/22 06:24:53 ekr Exp $

*/

package COM.claymoresystems.ptls;
import COM.claymoresystems.sslg.*;
import COM.claymoresystems.cert.WrappedObject;
import COM.claymoresystems.crypto.EAYEncryptedPrivateKey;

import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.util.Hashtable;
import java.util.Vector;
import java.io.*;

/** SSLContext holds all state relevant to more than one SSL Session/
    Connection. In particular, it's the access point for user keying
    material, user policy settings, and the session cache.
    <P>
    For clients, it is legal to merely create an SSLContext and
    use it immediately, though it is advisable to set the SSLPolicy.
    <P>
    Since servers must have keying material to operate, all SSLContexts
    which are to be used for servers must be initialized using
    loadEAYKeyFile() or loadPKS12KeyFile().
    <P>
    Currently, SSLContext supports only one key at a time and
    loading a new keyfile overrides any exiting keys/certificates.
    Future implementations may support multiple keys automatically
    selected based on the cipherSuite.
*/
public class SSLContext extends SSLContextInt
{
     private final static int SEED_BYTES=128;
     
     private Hashtable session_cache=new Hashtable();
     Vector root_list=new Vector();
     Vector certificates=null;
     PrivateKey privateKey=null;
     private int ephemeralDHKeyLength=1024;     
     SSLDHPrivateKey dhEphemeral=null;
     KeyPair rsaEphemeral=null;
     SecureRandom rng=new SecureRandom(); 

     /** Load keying material from the indicated PKCS12/PFX keyfile,
	 using the passphrase passed in

	 @param path the filename for the keyfile
	 @param passphrase the passphrase needed to decrypt/verify the keyfile

	 <B>Currently not implemented</B>
     */
     public void loadPKCS12File(String path, String passphrase){
       throw new InternalError("Not implemented");
     }

     /** Load a subset of SSLeay keyfiles.
	 <P>
	 We assume that the first key is bound to the first group
	 of certificates
	 <P>
	 We assume that any certificates we find are strictly ordered
	 from the user's certificate to the root.

	 @param path the filename for the keyfile
	 @param passphrase the passphrase needed to decrypt the private key

	 @exception IOException if the keyfile is badly formatted
	 @exception FileNotFoundException if the keyfile doesn't exist
     */
     public void loadEAYKeyFile(String path, String passphrase)
       throws IOException, FileNotFoundException {
       FileInputStream fis=new FileInputStream(path);
       BufferedReader br=new BufferedReader(new InputStreamReader(fis));

       PrivateKey tmpPrivateKey;
       StringBuffer keyType=new StringBuffer();

       SSLDebug.debug(SSLDebug.DEBUG_INIT,"Loading key file");

       if(!WrappedObject.findObject(br,"PRIVATE KEY",keyType))
	 throw new IOException("Couldn't find private key in this file");
       
       try {
	 tmpPrivateKey=EAYEncryptedPrivateKey.createPrivateKey(br,
	   keyType.toString(),passphrase.getBytes());
       } catch (IllegalArgumentException e){
	 throw new IOException(e.toString());
       }

       // Now reopen the file to get certs
       fis=new FileInputStream(path);
       br=new BufferedReader(new InputStreamReader(fis));

       Vector certs=new Vector();
       
       for(;;){
	 byte[] cert=WrappedObject.loadObject(br,"CERTIFICATE",null);

	 if(cert==null)
	   break;
	 SSLDebug.debug(SSLDebug.DEBUG_INIT,"Loading certificate",cert);	 
	 certs.addElement((Object)cert);
       }

       // Enforce a minimum of one certificate
       if(certs.size()!=0){
	 privateKey=tmpPrivateKey;
	 certificates=certs;
       }
     }


     /** Load a list of acceptable roots.
	 <P>
	 Roots are not used for verifying the keys found in the
	 keyfile. They are only used for verifying the certificates
	 of peer entities.
	 <P>
	 Roots are formatted in SSLeay "PEM" style

	 @param path the filename containing the root list
     */
     // Root certificates come in a file list bracketed by
     // -----BEGIN CERTIFICATE-----
     // -----END CERTIFICATE-----
     public void loadRootCertificates(String path)
       throws java.io.IOException,FileNotFoundException {
       FileInputStream fis=new FileInputStream(path);
       BufferedReader br=new BufferedReader(new InputStreamReader(fis));

       for(;;){
	 byte[] root=WrappedObject.loadObject(br,"CERTIFICATE",null);

	 if(root==null)
	   break;
	   
	 SSLDebug.debug(SSLDebug.DEBUG_INIT,"Loading root",root);
	 root_list.addElement((Object)root);
       }
     }

     // Everything from here on in is internal
     
     /** return the root list

	 @return a Vector of byte[]
     */
     Vector getRootList(){
       return root_list;
     }

     /** return our certificate list (corresponds to our one private key)

	 @return a Vector of byte[]
     */
     Vector getCertificateChain(){
       return certificates;
     }

     /** return our private key

	 @return a PrivateKey
     */
     PrivateKey getPrivateKey(){
       return privateKey;
     }

     synchronized SSLDHPrivateKey getEphemeralDHPrivateKey(){
       if(dhEphemeral==null){
	 dhEphemeral=new SSLDHPrivateKey(rng,ephemeralDHKeyLength);
       }

       return dhEphemeral;
     }

     synchronized private KeyPair getEphemeralRSAPair(){
       if(rsaEphemeral==null){
	 try {
	   KeyPairGenerator kg=KeyPairGenerator.getInstance("RSA");

	   kg.initialize(512,rng);
	   rsaEphemeral=kg.generateKeyPair();
	 } catch (Exception e){
	   throw new InternalError(e.toString()); // Nothing should go wrong
	 }
       }

       return rsaEphemeral;
     }

     synchronized RSAPrivateKey getEphemeralRSAPrivateKey(){
       return (RSAPrivateKey)rsaEphemeral.getPrivate();
     }

     synchronized RSAPublicKey getEphemeralRSAPublicKey(){
       return (RSAPublicKey)rsaEphemeral.getPublic();
     }
	 
	 
     // Use this to branch off each connection's PRNG
     synchronized byte[] getSeedBytes(){
       byte[] buf=new byte[SEED_BYTES];

       rng.nextBytes(buf);
       return buf;
     }

     protected synchronized void storeSession(String key,SSLSessionData sd){
       SSLDebug.debug(SSLDebug.DEBUG_STATE,"Storing session under key"+key);
       session_cache.put(key,(Object)sd);
     }

     protected synchronized SSLSessionData findSession(String key){
       SSLDebug.debug(SSLDebug.DEBUG_STATE,"Trying to recover session using key"+key);
       Object obj=session_cache.get(key);

       if(obj==null)
	 return null;
       
       return (SSLSessionData)obj;
     }

     protected synchronized void destroySession(String sessionLookupKey){
       session_cache.remove(sessionLookupKey);
     }
}

     

