/*********************************************************************
**
**     File name:               ssh_comm.c
**                                  
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 Handle Generic Communication over SSH
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**
**     Notes:
**	The functions in this file will allow us to communicate with
**	the remote host more easily.
**
**	For example, rather than doing recv(), decrypt(), unpack(),
**	we can just call ssh_recv(), which does it all.
**
**	Multi-threaded would also be nice, or having a "local" ssh-client
**	running in the background that WinSSH (or any other program that
**	wants to use SSH) can talk to).  That would be spiffy.
**
**     Functions:
**	ssh_recv		recv, decrypt, unpack a packet
**	ssh_send		pack, encrypt, send a packet
**
**	ssh_write_stdin		write stdin data to server
**	ssh_write_stdin_n	write n bytes of stdin to server
**	ssh_read_merge		read stderr, stdout data from server
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_comm.c,v 3.24 1998/05/09 16:30:34 kohno Exp $";
#endif

#include <assert.h>

#include "ssh.h"
#include "ssh_comm.h"
#include "ssh_packet.h"
#include "ssh_crypt.h"
#include "ssh_smsg.h"
#include "ssh_cmsg.h"

#include "ssh_util.h"


/*********************************************************************
**
**     Function:                ssh_recv()
**
**     Purpose:                 recv, decrypt, unpack a packet
**
**     Entry (pre) conditions:  encryption method initialized
**
**     Parameters:              sockfd		socket to listen to
**				ssh_info	connection-specific information
**
**				*data		data from socket
**				*data_len	length of *data
**
**     Return value:            message type in packet recv'd
**
**     Error codes:             SSH_MSG_NOTYPE	no message, or maybe recv err
**				SSH_MSG_ERROR	error
**
**				ssh_errno propagated or set to
**				SSH_ERRNO_RECV
**
**     Side effects:            packet recv'd from sockfd.
**				data contains decrypted packet data
**				data_len is size of data.
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					SSH_MSG_ERROR for too-big packets
**     Modified:		Tadayoshi Kohno, 11 April 1998
**					(compression)
**
**     Notes:
**	This function basically just recvs a packet, decrypts it,
**	and unpacks it.  There are some cross-overs between the last two
**	actions, so maybe I'll merge them to optimize speed?
**
**	Also, my first attempt was to just read a large chunk of data
**	and process that.  This is not a good idea because packets may
**	be broken and multiple packets may be sent back-to-back.
**
**	To help solve the problem this imposes, I first read the length
**	of the packet, then read a packet upto that length.  Then I decrypt
**	it and unpack it like normal.  Yippie.  (My main test was cat'ing
**	/etc/termcap [among other things]  :)
**
**	Potential problem in that this could loop forever.  *NEED* to time
**	it out -- right now calling code should time out the recv, sends().
**	[actually, as the DOCUMENTATION says, the time-outs should be
**	handled by the application]
**
**	Note that an ssh_errno of SSH_ERRNO_RECV does not necessarily imply
**	an error.  It can also mean that there just wasn't any data available.
**	But, an errno of SSH_ERRNO_PACKET_BIG means that the packet was
**	too big.  The return value for this will be SSH_MSG_ERROR
**
**	If we get a successful packet, we make sure it's not a debug message
**	if it is, we keep on looping (9 December 1997).  This should
**	also be changed and returned to the calling function to display. xxx
**	[now that ssh_read_interactive is in place, a debug message should
**	be returned and the application can deal with it -- or, we could
**	call the debug function from here].
**
*********************************************************************/

int ssh_recv
(
	socket_type sockfd,		/* socket to listen to */
	struct ssh_struct * ssh_info,	/* connection specific information */

	uint8_t * data,			/* data in packet */
	uint32_t * data_len		/* length of data in packet */
)
{
	uint8_t packet[SSH_MAX_PACKET];		/* packet from sockfd */
	uint8_t out_packet[SSH_MAX_PACKET];	/* decrypted packet */
	uint8_t glom_data[SSH_MAX_PACKET];	/* glommed data */

	uint8_t * tmp_ptr;			/* temporary data pointer */

	uint8_t type;				/* message type in packet */

	uint32_t length;		/* internal length of packet */
	uint32_t glom_len;		/* length of glommed type + data */

	int offset;			/* current offset into packet */
	int grabbed;			/* amount grabbed per recv() */
	int desired;			/* desired amount (full packet) */

	type = SSH_MSG_DEBUG;

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(SSH_MSG_ERROR);
	}

	while (type == SSH_MSG_DEBUG)
	{
		/*
		**	First let's determine the length of the packet to grab
		*/
		if ((offset = recv(sockfd, (char *) packet, sizeof(length),
			SR_FLAGS)) <= 0)
		{
			ssh_errno_set(SSH_ERRNO_RECV);
			return(SSH_MSG_NOTYPE);
		}

		my_bcopy((void *) packet, (void *) &length, sizeof(length));
		length = ntohl(length);

		desired = length + (8 - (length % 8)) + sizeof(length);

		/*
		**	uh-oh, trying to receive a packet that is too
		**	big (the max packet size I've encountered was
		**	around 532[??], but maybe X-packets, ... will
		**	be bigger)
		*/
		if (desired >= SSH_MAX_PACKET)
		{
			ssh_errno_set(SSH_ERRNO_PACKET_BIG);

			ssh_debug_int_new(&(ssh_info->debug_info),
				"packet too big (length %d)",
				desired, "ssh_recv");

			return(SSH_MSG_ERROR);
		}

		/*
		**	Now lets grab the packet
		**		-- xxx time this out -- could loop forever
		*/
		while (offset < desired)
		{
			grabbed = recv(sockfd, (char *)(packet + offset),
				desired - offset, SR_FLAGS);
			if (grabbed > 0)
			{
				offset = offset + grabbed;
			}
		}


		/*
		**	Finally we can decode and unpack it
		*/
		if (ssh_cipher_decode(ssh_info, packet, out_packet))
		{
			return(SSH_MSG_ERROR);
		}

/*
		old method replaced by that below (kohno, 11 April 1998)
		if (ssh_packet_unpack(ssh_info, out_packet, data_len, &type,
			data, SSH_MAX_PACKET))
		{
			return(SSH_MSG_ERROR);
		}
*/

		if (ssh_packet_unpack_block(ssh_info, out_packet, &glom_len,
			glom_data, SSH_MAX_PACKET))
		{
			return(SSH_MSG_ERROR);
		}

		/*
		**	uncompress the block (if it is compressed)
		*/
		if (ssh_compression_active(ssh_info))
		{
			if (ssh_compression_uncompress(ssh_info, glom_data,
				out_packet, glom_len, SSH_MAX_PACKET,
				&glom_len))
			{
				return(SSH_MSG_ERROR);
			}
			
			tmp_ptr = out_packet;
		} else
		{
			tmp_ptr = glom_data;
		}


		/*
		**	and extract the type and data
		*/
		if (ssh_packet_block_extract(&type, data, data_len, 
			tmp_ptr, glom_len, SSH_MAX_PACKET))
		{
			return(SSH_MSG_ERROR);
		}
	}

	return(type);
}


/*********************************************************************
**
**     Function:                ssh_send
**
**     Purpose:                 form a packet, encrypt it, and send it
**
**     Entry (pre) conditions:  socket open
**
**				encryption turned on with ssh_set_cipher()
**				data memory valid
**
**     Parameters:              sockfd		socket to talk over
**				ssh_info	connection-specific information
**
**				data		data for inside packet
**				data_len	length of data
**				type		message type
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**				ssh_errno propagated from called funcs
**
**     Side effects:            packet formed out of data and type
**				packet encrypted
**				packet sent to remote machine over sockfd
**
**     Author/Date:             Tadayoshi Kohno, 27 November 1997
**     Modified:		Tadayoshi Kohno, 11 April 1998
**					(compression)
**
**     Notes:
**	This functions rolls packing, encrypting, and sending all into one.
**	Using this really cleaned up some of my functions.
**
**	Again, make sure not overflowing buffer xxx
**
*********************************************************************/

int ssh_send
(
	socket_type sockfd,		/* socket to talk to */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	const uint8_t * data,		/* data to send */
	uint32_t data_len,		/* length of data */
	uint8_t type			/* message type */
)
{
	uint8_t packet[SSH_MAX_PACKET];		/* plain-text packet to send */
	uint8_t out_packet[SSH_MAX_PACKET];	/* encrypted packet, tmp str */
	uint8_t tmp_str[SSH_MAX_PACKET];	/* another tmp str */

	uint8_t * tmp_ptr;			/* pointer to tmp_str */

	uint32_t packet_len;	/* length of packet */
	uint32_t glom_len;	/* length of glommed type, data */

	int count;		/* count of bytes sent */

	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}



/*
	old method replaced by that below (kohno, 11 April 1998)

	if (ssh_packet_pack(ssh_info, packet, &packet_len, type, data,
		data_len, SSH_MAX_PACKET))
	{
		return(S_BAD);
	}
*/

/*
**	form the type + data block (note that here out_packet is just
**	being used as a temporary string)
*/
	if (ssh_packet_block_glom(out_packet, &glom_len, type, data,
		data_len, SSH_MAX_PACKET))
	{
		return(S_BAD);
	}

/*
**	if compression is being used, compress the packet here
**	otherwise point tmp_ptr to our glommed data
*/
	if (ssh_compression_active(ssh_info))
	{
		if (ssh_compression_compress(ssh_info, out_packet,
			tmp_str, glom_len, SSH_MAX_PACKET, &glom_len))
		{
			return(S_BAD);
		}
		
		tmp_ptr = tmp_str;
	} else
	{
		tmp_ptr = out_packet;
	}

/*
**	create the packet
*/
	if (ssh_packet_pack_block(ssh_info, packet, &packet_len, tmp_ptr,
		glom_len, SSH_MAX_PACKET))
	{
		return(S_BAD);
	}

/*
**	encipher the packet
*/
	if (ssh_cipher_encode(ssh_info, packet, out_packet))
	{
		return(S_BAD);
	}

/*
**	send the packet
*/
	if ((count = send(sockfd, (char *) out_packet, packet_len, SR_FLAGS))
		!= (int) packet_len)
	{
		ssh_errno_set(SSH_ERRNO_SEND);
		return(S_BAD);
	}

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_write_stdin
**
**     Purpose:                 send data as stdin data for program
**				running on server
**
**     Entry (pre) conditions:  sockfd opened
**				input pointer valid and NULL terminated
**
**     Parameters:              sockfd		socket to talk over
**				ssh_info	connection-specific information
**				input		input string
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**				ssh_errno set by called functions
**
**     Side effects:            input string sent to server over
**				sockfd using ssh-v1.5 protocol
**				and chosen form of encryption
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**
**     Notes:
**
*********************************************************************/

int ssh_write_stdin
(
	socket_type sockfd,		/* socket to talk over */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	const uint8_t * input		/* data to send */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data for packet */
	uint32_t data_len;		/* length of data */

	if (ssh_cmsg_stdin_data_encode(data, &data_len, input,
		strlen((const char *)input), SSH_MAX_PACKET))
	{
		return(S_BAD);
	}

	if (ssh_send(sockfd, ssh_info, data, data_len, SSH_CMSG_STDIN_DATA))
	{
		return(S_BAD);
	}

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_write_stdin_n
**
**     Purpose:                 send data as stdin data for program
**				running on server (counterpart to
**				ssh_write_stdin())
**
**     Entry (pre) conditions:  sockfd opened
**				input pointer valid "length" bytes long
**
**     Parameters:              sockfd		socket to talk over
**				ssh_info	connection-specific information
**
**				input		input string
**				length		length of input
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**				ssh_errno set by called functions
**
**     Side effects:            input string sent to server over
**				sockfd using ssh-v1.5 protocol
**				and chosen form of encryption
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**     Modified:                Tadayoshi Kohno, 5 April 1998
**					(take a length parameter)
**
**     Notes:
**
*********************************************************************/

int ssh_write_stdin_n
(
	socket_type sockfd,		/* socket to talk over */
	struct ssh_struct * ssh_info,	/* connection-specific information */

	const uint8_t * input,		/* data to send */
	int length			/* length of input */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data for packet */
	uint32_t data_len;		/* length of data */

	if (ssh_cmsg_stdin_data_encode(data, &data_len, input,
		length, SSH_MAX_PACKET))
	{
		return(S_BAD);
	}

	if (ssh_send(sockfd, ssh_info, data, data_len, SSH_CMSG_STDIN_DATA))
	{
		return(S_BAD);
	}

	return(S_GOOD);
}

/*********************************************************************
**
**     Function:                ssh_read_merge
**
**     Purpose:                 read stdout, stderr data from server
**
**     Entry (pre) conditions:  sockfd opened, *output valid pointer
**
**     Parameters:              sockfd		socket to read from
**				ssh_info	connection-specific information
**				*output		stdout/stderr data from server
**
**     Return value:            length of data read
**
**     Error codes:             SSH_NO_DATA		error
**					(possible because no is data ready)
**
**     Side effects:            packet read from server and decoded
**				packet is STDOUT/STDERR, then it's
**				data is returned to caller.
**
**				if message received is of a special
**				type, other stuff may happen.  (*)
**
**				ssh_errno set to SSH_ERRNO_DISCONNECTED
**				if we're disconnected
**
**     Author/Date:             Tadayoshi Kohno, 28 November 1997
**
**     Notes:
**	After initialization, pretty much the only contact between the
**	application and the server is through reads and writes (and
**	adjusting terminal size, ...).  So, we may need to handle other
**	types of messages here.
**
**	Note that the output is NOT NULL terminated.
**	
**	(*) Depricated in favor of ssh_read_interactive
**
*********************************************************************/

int ssh_read_merge
(
	socket_type sockfd,		/* socket to read from */
	struct ssh_struct * ssh_info,	/* connection-specific information */
	uint8_t * output		/* output (stdout,stderr)  */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data from server */
	uint32_t data_len;		/* output string */

	int type;			/* type of message received */

	/*
	**	First let's get the message
	*/
	type = ssh_recv(sockfd, ssh_info, data, &data_len);

	/*
	**	And now let's act upong what type of message it was
	*/
	switch (type)
	{
	case SSH_MSG_NOTYPE:
	case SSH_MSG_ERROR:
		/* no data available or an error occured */
		return(SSH_NO_DATA);
		break;

	case SSH_SMSG_STDOUT_DATA:
	case SSH_SMSG_STDERR_DATA:
		/* pass stdout/stderr data to application */
		if (ssh_smsg_merge_data_decode(data, data_len, output,
			SSH_MAX_PACKET))
		{
			ssh_debugger_new(&(ssh_info->debug_info),
				"error decoding", "ssh_read_merge");
			return(SSH_NO_DATA);
		}
		return(data_len - sizeof(uint32_t));
		break;

	case SSH_SMSG_EXITSTATUS:
		/* we should really handle this by sending a reply */
		ssh_errno_set(SSH_ERRNO_DISCONNECTED);
		ssh_debugger_new(&(ssh_info->debug_info),
			"Received SSH_SMSG_EXITSTATUS", "ssh_read_merge");
		break;

	case SSH_MSG_DISCONNECT:
		/* tell the caller that we've been disconnected */
		ssh_errno_set(SSH_ERRNO_DISCONNECTED);
		ssh_debugger_new(&(ssh_info->debug_info),
			"Disconnected", "ssh_read_merge");
		break;

	default:
		/*
		**	Handle other server events
		*/
		break;

	}
	return(SSH_NO_DATA);
}


/*********************************************************************
**
**     Function:                ssh_read_interactive
**
**     Purpose:                 read data from server in interactive mode
**
**     Entry (pre) conditions:  sockfd opened, ssh_connection
**				established, *output valid pointer
**
**     Parameters:              sockfd		socket to read from
**				ssh_info	connection-specific information
**
**				*output		stdout/stderr data from server
**				*output_len	length of output
**				max_output_len	maximum allowed length for
**						output
**
**     Return value:            message type received
**
**     Error codes:             SSH_MSG_ERROR	error with recv.
**
**     Side effects:            packet read from server and decoded
**				and data (if applicable) returned to
**				server as *output.
**
**				*output_len typically set to the length of
**				*output.  for the SSH_SMSG_EXITSTATUS,
**				*output_len is set to the exit status
**
**     Author/Date:             Tadayoshi Kohno, 7 March 1998
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					handle too small buffers and
**					return error
**
**     Notes:
**	After initialization, pretty much the only contact between the
**	application and the server is through reads and writes (and
**	adjusting terminal size, ...).
**
**	This function provides the facilities to read packets from
**	the server.  The packets may be of many types (SSH_MSG_NOTYPE)
**	SSH_SMSG_STD{OUT,ERR}_DATA, SSH_SMSG_EXITSTATUS, ...
**
**	This function reads the packet and determines what type it is
**	It then decodes it based upon the type and returns the
**	packet data (for example, stderr output) to the caller.
**	The type of message parsed is returned to the caller, or
**	SSH_MSG_NOTYPE if no packet is available or if an error has
**	occured.
**
**	An explanation of the various types
**		SSH_MSG_NOTYPE
**		SSH_MSG_ERROR
**			no packet available or error occured
**
**		SSH_SMSG_STDOUT_DATA
**		SSH_SMSG_STDERR_DATA
**			stderr, stdout data returned (NOT NULL terminated)
**			through *output.  the length is specified in
**			*output_length (set output[output_length] to 0
**			before printing).
**
**		SSH_SMSG_EXITSTATUS
**			shell or command has exited.  *output_len is
**			set to the exit status the server sent.  
**			caller should call ssh_disconnect_client_confirm
**
**		SSH_MSG_DISCONNECT
**			server said to disconnect.  client can just exit
**			(no need to call any ssh_disconnect_* function)
**
**	Note that the *output is NOT NULL terminated for the cases that
**	*output is used.
**
**	(4 April 1998)  There have been several requests to add internal
**	buffering to this functions.  This would definititely be a "good"
**	thing because it would allow calling applications to request
**	exactly the number of bytes they want (for stdout/stderr data).
**	
**	After putting in a few stubs for these (ssh_std_buffer_struct, ...),
**	it looks like this solution might actually not be as elegent as
**	people thought.  Namely, if the calling application uses select()
**	to determine if there is data available, select() will not show that
**	there is data available in ssh's internal buffer.  I guess we
**	could also write an ssh_select() function, but that is beyond the
**	current goals.  Also, it looks like many people who use libSSH
**	may want to write their own "wrapper" routines anyway.  Thus, for
**	libSSH to remain as general as possible, there currently is no internal
**	buffering
**
*********************************************************************/

int ssh_read_interactive
(
	socket_type sockfd,		/* socket to read from */
	struct ssh_struct * ssh_info,	/* connection-specific information */

	uint8_t * output,		/* output (stdout,stderr,...)  */
	uint32_t * output_len,		/* output length */

	uint32_t max_output_length	/* maximum output length */
)
{
	uint8_t data[SSH_MAX_PACKET];	/* data from server */
	uint32_t data_len;		/* output string */

	int type;			/* type of message received */

	*output_len = 0;

	/*
	**	First let's get the message
	*/
	type = ssh_recv(sockfd, ssh_info, data, &data_len);

	/*
	**	And now let's act upong what type of message it was
	*/
	switch (type)
	{
	case SSH_MSG_NOTYPE:
		/*
		**	no data available or an error occured
		*/
		return(type);
		break;

	case SSH_SMSG_STDOUT_DATA:
	case SSH_SMSG_STDERR_DATA:
		/*
		**	pass stdout/stderr data to application
		*/
		if (ssh_smsg_merge_data_decode(data, data_len, output,
			max_output_length))
		{
			ssh_debugger_new(&(ssh_info->debug_info),
				"error decoding std (probably because length"
				" allocated is too small)", "ssh_read_merge");
			return(SSH_MSG_ERROR);
		}
		*output_len = data_len - sizeof(uint32_t);
		return(type);
		break;

	case SSH_SMSG_EXITSTATUS:
		/*
		** received an exit status, so inform caller to call
		** ssh_disconnect_client_active
		*/
		if (ssh_smsg_exitstatus_decode(data, data_len,
			output_len) != S_GOOD)
		{
			ssh_debugger_new(&(ssh_info->debug_info),
				"error decoding exitstatus", "ssh_read_merge");
			return(SSH_MSG_ERROR);
		}

		ssh_debug_int_new(&(ssh_info->debug_info),
			"Received SSH_SMSG_EXITSTATUS %d", *output_len,
			"ssh_read_interactive");
		return(type);
		break;

	case SSH_MSG_DISCONNECT:
		/*
		**	received a disconnect.  caller just needs to
		**	exit (no need to do ssh_disconnect_client_confirm)
		*/
		ssh_debugger_new(&(ssh_info->debug_info),
			"Received SSH_MSG_DICONNECT", "ssh_read_interactive");
		return(type);
		break;

	case SSH_MSG_ERROR:
		/*
		**	some sort of error in recv
		*/
		ssh_debugger_new(&(ssh_info->debug_info),
			"Error from recv", "ssh_read_interactive");
		return(type);
		break;

	default:
		/*
		**	Handle other server events
		*/
		ssh_debug_int_new(&(ssh_info->debug_info),
			"received unknown packet type %d", type,
			"ssh_read_interactive");
		return(SSH_MSG_NOTYPE);
		break;

	}
	return(SSH_MSG_NOTYPE);
}

