/*
 *  $Id: cryptfs.h,v 1.1 1998/12/02 23:37:18 ezk Exp $
 */

#ifndef __FIST_CRYPTFS_H_
#define __FIST_CRYPTFS_H_

#include <blowfish.h>

#ifdef __KERNEL__
/* these macros are needed first */
#define FIST_HASH_SHIFT  4
#define FIST_HASH_SIZE  (1 << FIST_HASH_SHIFT)
#define FIST_HASH_MASK  (FIST_HASH_SIZE - 1)

/*
 * STRUCTURES:
 */

typedef struct fist_key {
    int uid;
    BF_KEY key;
    struct fist_key *next;
} fist_key_t;

/* cryptfs inode data in memory */
struct cryptfs_inode_info {
    inode_t *inode;
};

/* cryptfs super-block data in memory */
struct cryptfs_sb_info {
    super_block_t *s_sb;
#if 0
    /* not needed b/c linux has an s_root field in struct super_block */
    dentry_t *s_root;
#endif
    fist_key_t *s_user_key[FIST_HASH_SIZE]; /* hash table for user keys */
};


/*
 * MACROS:
 */

// File TO Hidden File
#define ftohf(file) ((file_t *)((file)->private_data))
// Inode TO Private Data
#define itopd(ino) ((struct cryptfs_inode_info *)(ino)->u.generic_ip)
// Inode TO Hidden Inode
#define itohi(ino) (itopd(ino)->inode)
// Superblock TO Private Data
#define stopd(super) ((struct cryptfs_sb_info *)(super)->u.generic_sbp)
// Superblock TO Hidden Superblock
#define stohs(super) (stopd(super)->s_sb)
// Dentry TO Hidden Dentry (and private data --- for the future)
#define dtopd(dentry) ((dentry)->d_fsdata)
#define dtohd(dentry) ((dentry_t *)dtopd(dentry))
/* VMArea TO Hidden VMArea */
#define vmatohvma(vma) ((vm_area_t *)((vma)->vm_private_data))

#define sbt(sb) ((sb)->s_type->name)

/* 13 bits -> 8192 bytes for the block size */
#define CRYPT_BLOCK_SHIFT	12
#define CRYPT_BLOCK_SIZE	(1 << CRYPT_BLOCK_SHIFT)
#define CRYPT_BLOCK_MASK	(CRYPT_BLOCK_SIZE - 1)

/*
 * Flags for crypt_{en,de}code_filename
 * DO_DOTS means the special entries . and .. should be encoded (for symlink)
 * SKIP_DOTS means they should be preserved intact
 */
#define DO_DOTS   0
#define SKIP_DOTS 1


/*
 * EXTERNALS:
 */
extern struct file_operations cryptfs_fops;
extern struct inode_operations cryptfs_iops;
extern struct super_operations cryptfs_sops;
extern struct dentry_operations cryptfs_dops;
extern struct vm_operations_struct cryptfs_shared_vmops;
extern struct vm_operations_struct cryptfs_private_vmops;

extern int cryptfs_decode_block(const char *from, char *to, int len, void *key);
extern int cryptfs_decode_filename(const char *name, int length, char **decrypted_name, void *key, int skip_dots);
extern int cryptfs_encode_block(const char *from, char *to, int len, void *key);
extern int cryptfs_encode_filename(const char *name, int length, char **uuencoded_name, void *key, int skip_dots);
extern int cryptfs_interpose(dentry_t *hidden_dentry, dentry_t *this_dentry, super_block_t *sb, int flag);
extern void *fist_get_userpass(const super_block_t *super);
extern void fist_set_userpass(const super_block_t *super, unsigned char *key);


#ifdef FIST_DEBUG
# define cryptfs_hidden_dentry(d) __cryptfs_hidden_dentry(__FILE__,__FUNCTION__,__LINE__,(d))
extern dentry_t *__cryptfs_hidden_dentry(char *file, char *func, int line, dentry_t *this_dentry);
#else /* not FIST_DEBUG */
# define cryptfs_hidden_dentry(dentry) dtohd(dentry)
#endif /* not FIST_DEBUG */

#endif /* __KERNEL__ */

/*
 * Definitions for user and kernel code
 */
/* for user programs to set the key */
#define FIST_IOCTL_SET_KEY		_IOW(0x15, 10, char [16])

#endif	/* __FIST_CRYPTFS_H_ */

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */



