/*
 * clients/kauth/kauth.c
 *
 * Copyright 1990 by the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * Export of this software from the United States of America may
 *   require a specific license from the United States Government.
 *   It is the responsibility of any person or organization contemplating
 *   export to obtain such a license before exporting.
 * 
 * WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
 * distribute this software and its documentation for any purpose and
 * without fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of M.I.T. not be used in advertising or publicity pertaining
 * to distribution of the software without specific, written prior
 * permission.  M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 * 
 *
 * Takes a principal and a password and verifies that the user is valid
 */

#include <krb5.h>
#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <signal.h>
#include <sys/un.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "getopt.h"

char service[] = "netscape";

char *path;

void socket_cleanup()
{
    unlink(path);
	exit(0);
}

/* if struct[2] == NULL, then long_getopt acts as if the short flag
   struct[3] was specified.  If struct[2] != NULL, then struct[3] is
   stored in *(struct[2]), the array index which was specified is
   stored in *index, and long_getopt() returns 0. */

struct option long_options[] = {
    { "version", 0, NULL, 0x01 },
    { "nofork", 0, NULL, 'n' },
    { NULL, 0, NULL, 0 }
};

int
main(argc, argv)
    int argc;
    char **argv;
{
    struct sockaddr_un addr;
    krb5_context kcontext;
    krb5_error_code code;
    krb5_keytab keytab = NULL;
    int errflg = 0;
    int nofork = 0;
    int i, fd, newfd, len;
    char machine_name[256];
    char * service_name = NULL;
    char * input;
    struct sigaction act;
    krb5_verify_init_creds_opt opt;

    
    gethostname(machine_name, 255);
    krb5_init_context(&kcontext);
    krb5_init_ets(kcontext);

    if (strrchr(argv[0], '/'))
	argv[0] = strrchr(argv[0], '/')+1;

    while ((i = getopt_long(argc, argv, "S:t:n", long_options, NULL)) != EOF) {
	switch (i) {
	case 1: /* Print the version */
	    printf("%s\n", krb5_version);
	    exit(0);
	case 't':
	    if (code = krb5_kt_resolve(kcontext, optarg, &keytab)) {
                com_err(argv[0], code, "resolving keytab %s", optarg);
		exit(1);
            }
	    break;
        case 'S':
	    service_name = optarg;
	    break;
	case 'n':
            nofork++;                   /* don't detach from terminal */
            break;
	default:
	    errflg++;
	    break;
	}
    }

    if ((optind != argc-1) || errflg) {
	fprintf(stderr, "Usage: %s [--version] [-S target_service] [-t keytab_file] socket\n", argv[0]);
	exit(2);
    }

    if (strlen(argv[optind]) > 108) {
    	fprintf(stderr, "Socket name is too long\n");
    	exit(2);
    }

    addr.sun_family = AF_UNIX;
    strcpy(addr.sun_path, path = argv[optind]);
    if ((fd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
	fprintf(stderr, "Cannot get a new socket\n");
	exit(1);
    }

    if (bind(fd, &addr, sizeof(addr)) < 0) {
	fprintf(stderr, "Cannot bind to address\n");
	exit(1);
    }

    listen(fd, 8);

    /* Create default service key if necessary */
    if (service_name == NULL) {
	if ((service_name = malloc(MAXHOSTNAMELEN+strlen(service)+1)) == NULL) {
	    fprintf(stderr, "Cannot create service principal\n");
	    exit(1);
	}
	sprintf(service_name, "%s/%s", service, machine_name);
    }

    /* Allocate a minimum buffer */
    if ((input = malloc(len = 1024)) == NULL) {
	fprintf(stderr, "Cannot allocate space for buffer\n");
	exit(1);
    }

    /* Set up signal handlers to cleanup socket on exit */
    act.sa_flags = 0;
    sigemptyset(&act.sa_mask);
    act.sa_handler = socket_cleanup;
    sigaction(SIGTERM, &act, NULL);
    sigaction(SIGHUP, &act, NULL);
    sigaction(SIGINT, &act, NULL);

    /* Do daemon fork here */
    if (!nofork && daemon(0, 0)) {
        com_err(argv[0], errno, "while detaching from tty");
	exit(1);
    }

    krb5_verify_init_creds_opt_init(&opt);
    krb5_verify_init_creds_opt_set_ap_req_nofail(&opt, 1);

    while (i = sizeof(addr), (newfd = accept(fd, &addr, &i)) >= 0) {
        int input_len, current_len = 0;
	char * output, * user, * pw;
	krb5_creds my_creds;
        krb5_principal me;

	if ((current_len = read(newfd, input, len)) <= 0)
	    continue;

	input_len = atoi(input);

	if (input_len > current_len) {
	    if (input_len > len) {	
		char * newbuf;
		if ((newbuf = malloc(len)) == NULL) {
		    /* Should print an error message back to Netscape */
		    close(newfd);
		    continue;
		}
		memcpy(input, newbuf, current_len);
		len = input_len;
		free(input);
		input = newbuf ;
	    }
	    if (read(newfd, input + current_len, len - current_len) <= 0)
		continue;
	}

	if (((user = strchr(input, ':')) == NULL) || 
	  ((pw = strchr(++user, ':')) == NULL))
	    continue;

	*pw++ = '\0';

	/* Use specified name */
	if ((code = krb5_parse_name (kcontext, user, &me))) {
	    output = "fail:while parsing name";
	} else {
    	    switch(code = krb5_get_init_creds_password(kcontext, &my_creds, me, 
    	      	pw, NULL, NULL, 0, service_name, NULL)) {
            case 0:
		if (code = krb5_verify_init_creds(kcontext, &my_creds, NULL,
		  				  keytab, NULL, &opt)) {
		    output = "fail: while verifying credential";
		} else {
		    output = "pass";
	 	}
    	        break;
    	    case KRB5KRB_AP_ERR_BAD_INTEGRITY:
	        output = "fail: password is incorrect";
    	        break;
            default:
		output = "fail: while getting credential";
    	        break;
            }
	}
    	write (newfd, output, strlen(output));

        krb5_free_principal(kcontext, me);
    }

    krb5_free_context(kcontext);
    unlink(argv[optind]);
    exit(0);
}
