# Support for krb5.conf and similar config files.
# These interfaces are likely to change for a future version; don't
# rely on them.
#
# Suggested revisions:
# * Use itcl classes to hide internals; one for basic format, more for
#   krb5 and kdc specific stuff.  Tags like "libdefaults" shouldn't occur
#   outside this file.
# * More regular names.  Which functions create new objects?  Which will
#   modify existing ones?  Which require exactly one exist already?  Which
#   return lists?
# * No global variables.  Make the profiles be objects themselves.
#
set default_tgs_enctypes des-cbc-crc
set default_tkt_enctypes des-cbc-crc

set profile {}

# other pathnames to check for krb.conf?
if [file exists /usr/kerberos/lib/krb.conf] {
    set krb4_compat yes
    set krb4_config /usr/kerberos/lib/krb.conf
    set krb4_realms /usr/kerberos/lib/krb.realms
} else {
    set krb4_compat no
    set krb4_config /usr/kerberos/lib/krb.conf
    set krb4_realms /usr/kerberos/lib/krb.realms
}
if [file exists /etc/srvtab]&&![file exists /etc/krb-srvtab] {
    set krb4_srvtab /etc/srvtab
} else {
    set krb4_srvtab /etc/krb-srvtab
}

proc make_pair { tag value } {
    if ![krb5tcl_is_okay_tagname $tag] {
	error [list $tag is not a valid tag name]
    }
    if ![krb5tcl_is_okay_value $value] {
	error [list $value is not a valid value for this field]
    }
    return [list value $tag $value]
}

proc make_section { tag values } {
    if ![krb5tcl_is_okay_tagname $tag] {
	error [list $tag is not a valid tag name]
    }
    return [list section $tag $values]
}

proc issection { x { name {} } } {
    if [string compare [lindex $x 0] section] { return 0 }
    if ![string compare $name ""] { return 1 }
    if ![string compare $name [lindex $x 1]] { return 1 }
    return 0
}

proc ensure_top_section { tag } {
    global profile
    foreach x $profile {
	if [issection $x $tag] {
	    return
	}
    }
    lappend profile [make_section $tag ""]
}

proc have_profile_section { args } {
    global profile
    have_section $profile $args
}
proc have_section { prof names } {
    set sname [lindex $names 0]
    set rest [lrange $names 1 end]
    foreach x $prof {
	if "[issection $x $sname]" {
	    if [llength $rest] {
		return [have_section [lindex $x 2] $rest]
	    } else { return 1 }
	}
    }
    return 0
}

proc mapcar { p vals } {
    set result ""
    foreach x $vals {
	set cmd $p
	lappend cmd $x
	lappend result [uplevel $cmd]
    }
    return $result
}

# FROB_SECTION CMD TAGLIST SEC
# Find subsections of SEC identified by remaining entries in TAGLIST,
# and apply CMD when we've narrowed them down as much as possible.
# The CMD invocation gets the {section FOO LIST-OF-DATA-ITEMS} list.
proc frob_section { cmd taglist sec } {
#    puts [list frob_section $cmd $taglist $sec]
    set this_tag [lindex $taglist 0]
    if [llength $taglist]==0 {
	error "frob_section requires non-empty tag list"
    }
    if [issection $sec $this_tag] {
#	puts "name matches"
	if [llength $taglist]>1 {
#	    puts "more tags"
	    set othertags [lrange $taglist 1 end]
	    set rcmdpfx [list frob_section $cmd $othertags]
	    set newcontents [mapcar $rcmdpfx [lindex $sec 2]]
	    return [make_section $this_tag $newcontents]
	} else {
	    lappend cmd $sec
#	    puts [list frob_section invoking $cmd]
	    return [eval $cmd]
	}
    } else {
#	puts [list frob_section not sec $this_tag : $sec]
	return $sec
    }
}

proc frob_profile { cmd taglist } {
    global profile
#    puts \n[list ... STARTING FROB_PROFILE $cmd $taglist ...]\n
    set profile [mapcar [list frob_section $cmd $taglist] $profile]
    return
}

# ADD_SECTION ID ?ID...?
# Creates a new section, using the successive IDs as the tag names
# at each level going down.  Only the lowest-level subsection is
# created; higher-level ones must already exist.  If multiple higher-level
# subsections exist, all identified by the same sequence of tag names,
# only one has the new subsection created; which this is is undefined.
proc add_section.1 { lev newsecname sec } {
    # get access to add_section local vars
    upvar #$lev done done
    # now, do some work
    if !$done {
	set done 1
	set last [lindex $sec 2]
	lappend last [make_section $newsecname {}]
	return [lreplace $sec 2 2 $last]
    } else {
	return $sec
    }
}
proc add_section { args } {
    set lastname [lindex $args end]
    set done 0

    frob_profile "add_section.1 [info level] $lastname" [lreplace $args end end]
}

# ADD_PAIR TAG VALUE ?SEC SUBSEC ...?
proc add_pair.1 { lev tag value sec } {
    # get access to add_section local vars
    upvar #$lev done done
    # now, do some work
    if !$done {
	set done 1
	set last [lindex $sec 2]
	lappend last [make_pair $tag $value]
	return [lreplace $sec 2 2 $last]
    } else { return $sec }
}
proc add_pair { tag value args } {
    set done 0
    frob_profile [list add_pair.1 [info level] $tag $value] $args
    return
}
proc remove_values { tag args } {
    frob_profile "purge_section_entries $tag" $args
}

proc create_realm { rname } {
    ensure_top_section realms
    if ![have_realm $rname] { add_section realms $rname }
}
proc have_realm { rname } {
    have_profile_section realms $rname
}
proc remove_realm_info { realm tag } {
    if ![have_realm $realm] { error [list realm $realm does not exist] }
    frob_profile "purge_section_entries $tag" [list realms $realm]
}
proc add_realm_pair { realm tag value } {
    if ![have_realm $realm] { error [list realm $realm does not exist] }
    add_pair $tag $value realms $realm
}

proc default_set_pair { tag value args } {
    if [llength [eval [concat get_value_list $args [list $tag]]]]==0 {
	eval [list add_pair $tag $value] $args
    }
}
proc default_set_realm_pair { realm tag value } {
    if [llength [get_value_list realms $realm $tag]]==0 {
	add_realm_pair $realm $tag $value
    }
}

proc purge_section_entries { name sec } {
    set newcontents {}
    foreach x [lindex $sec 2] {
	set type [lindex $x 0]
	if [string compare $type value]&&[string compare $type section] {
	    lappend newcontents $x
	} else {
	    if [string compare [lindex $x 1] $name] {
		lappend newcontents $x
	    }
	}
    }
    set result [lreplace $sec 2 2 $newcontents]
    return $result
}

proc get_value_list.1 { lev tag sec } {
    upvar #$lev result result
    foreach x [lindex $sec 2] {
	if ![string compare value [lindex $x 0]]&&![string compare $tag [lindex $x 1]] {
	    lappend result [lindex $x 2]
	}
    }
    return $sec
}
proc get_value_list_va { tags } {
    set result {}
    frob_profile [list get_value_list.1 [info level] [lindex $tags end]] \
	    [lreplace $tags end end]
    return $result
}
proc get_value_list { args } { get_value_list_va $args }
proc get_value { args } {
    set values [get_value_list_va $args]
    switch [llength $values] {
	0	{ error [list No values for $args .] }
	1	{ return [lindex $values 0] }
	default	{ error [list Multiple values defined for $args .] }
    }
}

proc set_kdcs { rname args } {
    remove_realm_info $rname kdc
    eval [concat [list add_kdcs $rname] $args]
}
proc add_kdcs { rname args } {
    foreach kdc $args { add_kdc $rname $kdc }
}
proc add_kdc { rname kdc } {
    add_realm_pair $rname kdc $kdc
}
proc get_kdcs { rname } {
    get_value_list realms $rname kdc
}

proc set_admin_server { realm asname } {
    remove_realm_info $realm admin_server
    add_realm_pair $realm admin_server $asname
}
proc get_admin_server { realm } {
    set x [get_value_list realms $realm admin_server]
    switch [llength $x] {
	0	{ error [list No admin server for realm $realm.] }
	1	{ return [lindex $x 0] }
	default	{ error [list Multiple admin servers defined for realm $realm.] }
    }
}

proc set_default_domain { realm domain } {
    remove_realm_info $realm default_domain
    add_realm_pair $realm default_domain $domain
}

proc set_libdefault { tag value } {
    ensure_top_section libdefaults
    frob_profile "purge_section_entries $tag" libdefaults
    add_pair $tag $value libdefaults
}
proc set_domain_realm { domain realm } {
    ensure_top_section domain_realm
    frob_profile "purge_section_entries $domain" domain_realm
    add_pair $domain $realm domain_realm
}
proc set_default_realm { realm } {
    if ![have_realm $realm] { error [list realm $realm does not exist] }
    set_libdefault default_realm $realm
}

proc set_v4_config { config realms srvtab } {
    set_libdefault krb4_srvtab $srvtab
    set_libdefault krb4_config $config
    set_libdefault krb4_realms $realms
}
proc set_v4_iconvert { realm v4 v5 } {
    if ![have_profile_section realms $realm v4_instance_convert] {
	add_section realms $realm v4_instance_convert
    }
    frob_profile "purge_section_entries $v4" \
	    [list realms $realm v4_instance_convert]
    add_pair $v4 $v5 realms $realm v4_instance_convert
}

proc krb5_setup_default_config { } {
    default_set_pair default_tgs_enctypes des-cbc-crc libdefaults
    default_set_pair default_tkt_enctypes des-cbc-crc libdefaults

    if ![have_realm ATHENA.MIT.EDU] {
	create_realm ATHENA.MIT.EDU
	set_kdcs ATHENA.MIT.EDU \
		kerberos.mit.edu kerberos-1.mit.edu kerberos-2.mit.edu:88
	set_admin_server ATHENA.MIT.EDU kerberos.mit.edu
	set_default_domain ATHENA.MIT.EDU mit.edu

	set_v4_iconvert ATHENA.MIT.EDU mit mit.edu
	set_v4_iconvert ATHENA.MIT.EDU lithium lithium.lcs.mit.edu

	set_domain_realm .mit.edu ATHENA.MIT.EDU
	set_domain_realm mit.edu ATHENA.MIT.EDU
    }

    if 0 {
	set_domain_realm .media.mit.edu MEDIA-LAB.MIT.EDU
	set_domain_realm media.mit.edu MEDIA-LAB.MIT.EDU
    }

    if ![have_realm CYGNUS.COM] {
	create_realm CYGNUS.COM
	set_kdcs CYGNUS.COM \
		kerberos.cygnus.com kerberos-1.cygnus.com
	set_admin_server CYGNUS.COM kerberos.cygnus.com
    }

#    set_default_realm CYGNUS.COM

    global krb4_compat krb4_config krb4_realms krb4_srvtab
    if ![string compare $krb4_compat yes] {
	set_v4_config $krb4_config $krb4_realms $krb4_srvtab
    }

    # no data on this realm at present
#    set_domain_realm .ucsc.edu CATS.UCSC.EDU
}

proc with_profile { varname script } {
    global profile
    upvar 1 $varname alt_profile
    set tmp $profile
    set profile $alt_profile
    set result [uplevel $script]
    set alt_profile $profile
    set profile $tmp
    return result
}
