# who to report bugs to
set bug_addr bugs@cygnus.com
# additional instructions, or empty
set bug_how " using the send-pr program"
# name and version of package
set pkg KerbNet
set pkgvers $krb5_version

set symlink_name /usr/cygnus/kerbnet


# Support code.

# Always returns a hostname.
proc safe_canonicalize { hostname } {
    set result $hostname
    catch {set result [krb5tcl_canonicalize_hostname $hostname]}
    return [string toupper $result]
}

proc write_profile { data file } {
    if [catch {
	krb5tcl_write_profile $file.new $data
	krb5tcl_system_only_write $file.new
	exec mv -f $file.new $file
    } err] {
	puts "Can't write profile $file:\n$err"
	exit 1
    }
}
proc save_krb5_conf {} {
    global profile
    global krb5_path
    write_profile $profile $krb5_path(profile)
}
proc save_kdc_conf {} {
    global kdc_conf
    global krb5_path
    write_profile $kdc_conf $krb5_path(kdc_profile)
}

proc symlink { target linkname } {
    exec ln -s $target $linkname
}

proc configure_symlink {} {
    global symlink_name krb5_path
    set ret {}

    set x [get_value_list kerbnet-config symlink-name]
    if [llength $x]==1 { set symlink_name [lindex $x 0] }
    if [file exists $symlink_name] {
	set targ {}
	if ![catch {set targ [file readlink $symlink_name]} err] {
	    if ![string compare $targ $krb5_path(prefix)] {
		set ret $symlink_name
	    } else {
		# Don't use elseif here, the condition gets evaluated too early!
		if [ask-yesno "Symbolic link $symlink_name currently points to $targ;\nupdate it to point to $krb5_path(prefix)?" y] {
		    file delete -force $symlink_name
		    symlink $krb5_path(prefix) $symlink_name
		    set ret $symlink_name
		}
	    }
	} else {
	    # Something's there, but readlink as root failed.  Skip it.
	}
    } else {
	if [ask-yesno "Create symlink $symlink_name pointing to $krb5_path(prefix)?" y] {
	    symlink $krb5_path(prefix) $symlink_name
	    set ret $symlink_name
	}
    }
    if [string length $ret] {
	remove_values symlink-name kerbnet-config
	add_pair symlink-name $symlink_name kerbnet-config
	save_krb5_conf
    }
    return $ret
}


if ![krb5tcl_have_system_privs] {
    puts "This script must be run as root."
    exit 1
}

if ![info exists krb5_path(kdc_profile)] {
    set krb5_path(kdc_profile) $krb5_path(prefix)/lib/krb5kdc/kdc.conf
}

puts "Welcome to the $pkg installation program.

This program will ask you several questions, and configure your $pkg
installation based on your answers.  You may interrupt it at any time with
your terminal interrupt character (usually ^C), or responding to any question
with an end-of-file (^D); either will cause the configuration process to be
aborted.

We assume you have read the installation documentation that describes what a
Kerberos realm is, what a KDC is, etc.  If you have not, please read that
documentation before proceeding.

Please report any problems to $bug_addr$bug_how.

"

set realm ""
set hostname [info hostname]
set chostname [safe_canonicalize $hostname]

puts "Configuring $pkgvers on host $hostname ...\n"

if [file exists $krb5_path(profile)] {
    if [catch { set profile [krb5tcl_read_profile $krb5_path(profile)] } x] {
	puts "Error reading $krb5_path(profile): $x"
	puts "Ignoring it...\n"
    } else {
	puts "Using old information in $krb5_path(profile) for defaults...\n"
    }
}

proc fixperms {} {
    global krb5_path
    # Use "/." in case of symlinks.
    exec chown -R root $krb5_path(prefix)/.
    exec chmod -R go-w $krb5_path(prefix)/.
    if [file exists $krb5_path(prefix)/sbin/xdm-restart] {
	exec chmod u+s $krb5_path(prefix)/sbin/xdm-restart
    }
    exec chmod u+s $krb5_path(prefix)/bin/ksu
    exec chmod u+s $krb5_path(prefix)/bin/v4rcp
}

if [catch {
    # UNIX-specific...
    puts "Checking/fixing ownership and permissions of $krb5_path(prefix)...\n"
    fixperms
} err] {
    puts "Error: $err"
    exit 1
}

set x [get_value_list libdefaults default_realm]
if [llength $x]==1 {
    set realm [lindex $x 0]
}

if [string length $realm]==0 {
    set hsplit [split $hostname .]
    if [llength $hsplit]>1 {
	if [llength $hsplit]>2 { set hsplit [lrange $hsplit 1 end] }
	set realm [string toupper [join $hsplit .]]
    }
}

puts "Configuring realm information:\n"
set realm [ask "Name of local realm?" v.realm $realm]

set kdc_list [get_kdcs $realm]
if [llength $kdc_list]>0 {
    puts "The previously defined list of KDCs for realm $realm is:"
    foreach x $kdc_list { puts "\t$x" }
    puts ""
    if ![ask-yesno "Do you want to keep this list of KDCs?" yes] {
	set kdc_list {}
    }
} else {
    set kdc_list {}
}

if [llength $kdc_list]==0 {
    puts "\nPlease enter a list of Kerberos server (KDC) hostnames.  End the"
    puts "list with a blank line."
    while 1 {
	set kdc [ask "KDC #[expr 1 + [llength $kdc_list]] name?" v.hostname-addr-or-empty]
	if ![string compare $kdc ""] {
	    if [llength $kdc_list] break
	    puts "You have entered no KDC names.  Kerberos cannot operate without"
	    puts "at least one KDC.  Please enter your KDC hostnames now, or quit"
	    puts "the installation process."
	    continue
	}
	lappend kdc_list $kdc
	# Is the local host going to be a KDC?
	if ![string compare [safe_canonicalize $kdc] $chostname] {
	    set insttype 3
	}
    }
}

set x [lindex $kdc_list 0]
if [catch {set x [get_admin_server $realm]} err] { }
puts "\nNow you must select the name of the Kerberos administrative server.
This is usually one of the KDCs, as listed above.\n"
set admin_server [ask "$realm admin server name?" v.hostname-or-addr $x]

# This is pretty gratuitous, but by doing this here and now we
# maintain the order of entries in krb5.conf that we've always
# used before.
ensure_top_section libdefaults

create_realm $realm
set_default_realm $realm
eval "set_kdcs $realm $kdc_list"
set_admin_server $realm $admin_server

krb5_setup_default_config

ensure_top_section kerbnet-config
remove_values version kerbnet-config
add_pair version $krb5_version kerbnet-config

puts -nonewline "Writing realm info to $krb5_path(profile)..."
flush stdout
save_krb5_conf
puts "done.\n\nLocal system configuration for $chostname:"

proc v.install-type { x } {
    switch -- $x {
	"1"	-
	"2"	-
	"3"	-
	"4"	{ return 1 }
    }
    puts "\n\"$x\" is not a valid choice."
    return 0
}

set prompt "\nWill this machine be a:
  1) Kerberos application (rsh, rlogin, telnet) client system only
  2) Kerberos application server (and client) system
  3) slave KDC (and application client/server)
  4) master KDC with admin server (and application client/server)

Please enter the corresponding number for the configuration you
wish to install:"

set insttype 2
if ![string compare [safe_canonicalize $admin_server] $chostname] {
    set insttype 4
} else {
    foreach kdc $kdc_list {
	if ![string compare [safe_canonicalize $kdc] $chostname] {
	    set insttype 3
	}
    }
}

set insttype [ask $prompt v.install-type $insttype]
set insttype [lindex {nil appclient appserver slave-kdc master-kdc} $insttype]
puts ""

# Configure application clients.
set generic_prefix [configure_symlink]
if ![string length $generic_prefix] { set generic_prefix $krb5_path(prefix) }
puts ""

update_services
puts ""

proc fini {} {
    global pkg

    puts "\n$pkg basic configuration is now complete.  For advanced configuration,"
    puts "including /bin/login and xdm, see the documentation provided with $pkg."
    puts "\nHave a nice day."
    exit 0
}

# IF this will be an application client only, no other updates are
# needed.
if ![string compare appclient $insttype] { fini }

# Configure application servers.

puts "Which application services do you wish to run on this host?"
foreach x {ftp telnet {klogin "Kerberos rlogin"} {eklogin "encrypted Kerberos login"} {kshell "Kerberos remote shell"}} {
    set name [lindex $x 0]
    if [llength $x]==1 {
	set prompt $name
    } else {
	set prompt "$name ([lindex $x 1])"
    }
    if [ask-yesno "  $prompt?" y] { lappend krb5_daemons $name }
}

update_inetd

# Configure KDCs.

proc v.localstatedir { d } {
    if [string compare [file pathtype $d] absolute] {
	puts "Directory name must be absolute."
	return 0
    }
    if [file exists $d]&&![file isdirectory $d] {
	puts "$d is not a directory."
	return 0
    }
    return 1
}

proc setup_localstatedir {} {
    global conf_dir
    global profile

    ensure_top_section kerbnet-config

    set x [get_value_list kerbnet-config conf-dir]
    if [llength $x] {
	set conf_dir [lindex $x 0]
    } else {
	puts "You need to select a directory to store config files that will"
	puts "be preserved across releases.  (At the moment, only the KDC"
	puts "config file and database are stored there.)\n"
	puts "This directory should NOT be stored under the installation"
	puts "directory for any version of this product.  It should be on"
	puts "local disk storage for this machine, not accessed over the net."
	if [file isdirectory /var] {
	    set lsd /var/kerbnet
	} else {
	    set lsd /etc/kerbnet
	}
	set conf_dir [ask "Config file directory?" v.localstatedir $lsd]

	add_pair conf-dir $conf_dir kerbnet-config
    }
    if ![file exists $conf_dir] {
	if [catch {file mkdir $conf_dir} err] {
	    puts "** Error: $err"
	    puts "Please fix this error and try again."
	    exit 1
	}
    }
    if [catch {krb5tcl_system_only_write $conf_dir} err] {
	puts "** Error: $err"
    }
    save_krb5_conf
}

set have_old_db 0
if [string match *kdc $insttype] {
    if [catch {
	# do KDC stuff

	setup_localstatedir
	if ![file exists $conf_dir/kdc] { file mkdir $conf_dir/kdc }
	if ![file exists $krb5_path(prefix)/lib/krb5kdc] {
	    symlink $conf_dir/kdc $krb5_path(prefix)/lib/krb5kdc
	}

	global kdc_conf
	set kdc_conf {}
	if [file exists $krb5_path(kdc_profile)] {
	    if [catch {set kdc_conf [krb5tcl_read_profile $krb5_path(kdc_profile)]} err] {
		puts "$err\nError loading $krb5_path(kdc_profile), ignoring."
	    } else {
		puts "Using $krb5_path(kdc_profile) for KDC defaults..."
	    }
	}
	with_profile kdc_conf {
	    ensure_top_section kdcdefaults
	    ensure_top_section realms
	    create_realm $realm
	    default_set_pair kdc_ports 750,88 kdcdefaults
	    default_set_realm_pair $realm kdc_ports 750,88
	    default_set_realm_pair $realm kadmind_port 749
	    default_set_realm_pair $realm max_life "10h 0m 0s"
	    default_set_realm_pair $realm max_renewable_life "7d 0h 0m 0s"
	    default_set_realm_pair $realm master_key_type des-cbc-crc
	    default_set_realm_pair $realm supported_enctypes "des-cbc-crc:normal des-cbc-crc:v4"
	}
	save_kdc_conf

	setup_database $realm $conf_dir/kdc
    } err] {
	# error happened
	puts $err
	exit 1
    }
}

set admin_principal {}
proc v.local_principal {n} {
    if ![regexp -nocase {^[A-Za-z0-9./-]+$} $n] {
	puts "Principal name \"$n\" contains invalid characters.  Please"
	puts "enter a valid principal name."
	return 0
    }
    return 1
}
proc get_admin_principal {} {
    ask "Admin principal name (without realm)?" v.local_principal admin/admin
}


set kdcdir $krb5_path(prefix)/lib/krb5kdc
if ![string compare master-kdc $insttype]&&!$have_old_db {
    if [catch {
	if [string length $admin_principal]==0 {
	    puts "For administrative purposes, a principal with privileged"
	    puts "access to the database will be created now.  The default"
	    puts "name is \"admin/admin\", but you may choose another name."
	    puts ""
	    set admin_principal [get_admin_principal]
	}
	set apassword [askpw "Enter the admin password:" \
		"For verification, enter the admin password again:"]
	create_admin_inst $apassword
	set f [open "$kdcdir/kadm5.acl" w]
	puts $f "$admin_principal@$realm *"
	close $f

	# Don't use built in fns, we're running with a remote client that
	# needs kadmind already running.
	file delete -force $kdcdir/kadm5.keytab
	exec $kadmin_local_prog -q "ktadd -k $kdcdir/kadm5.keytab \
		kadmin/admin kadmin/changepw" </dev/null
    } err] {
	# error
	puts $err
	exit 1
    }
} else {
    if [string length $admin_principal]==0 {
	puts "The name of a principal with administrative access to the"
	puts "database is needed now.  If this principal does not exist,"
	puts "or does not have access to register, change, and retrieve"
	puts "keys, the rest of the installation may fail."
	puts ""
	set admin_principal [get_admin_principal]
    }
    set apassword [askpw "Please enter the password for \"$admin_principal\":" \
	    "For verification, enter the password again:"]
}
if ![string compare master-kdc $insttype] {
    if [catch {
	puts -nonewline "Starting new master KDC daemons: kdc..."
	flush stdout
	exec "$generic_prefix/sbin/krb5kdc"
	puts -nonewline " kadmind..."
	flush stdout
	exec "$generic_prefix/sbin/kadmind"
	puts "done."
    } err] {
	puts "Error starting Kerberos daemons: $err"
	exit 1
    }
}
if ![string compare slave-kdc $insttype] {
    if [catch {
	set f [open $kdcdir/kpropd.acl a]
	puts $f "host/[string tolower [safe_canonicalize $admin_server]]@$realm"
	close $f
    } err] {
	puts "Error writing to $kdcdir/kpropd.acl:\n  $err"
	exit 1
    }
    lappend krb5_daemons krb5_prop
}

# Okay.  Now we have the admin password, and if we're the master KDC, the
# daemons have been started.  If we're not the master, assume they've been
# started.

set kadmin_connect_done 0
proc kadmin_connect {} {
    global kadmin_connect_done realm apassword admin_principal
    if $kadmin_connect_done return
    puts "Connecting to admin server..."
    if [catch {kadm5_init -realm $realm -princ $admin_principal -password $apassword} err] {
	puts "Error initializing connection to kadmin server:\n  $err"
	puts "Please fix the problem and try again."
	exit 1
    }
    set kadmin_connect_done 1
}

set my_inst [string tolower $chostname]
set my_princ "host/$my_inst@$realm"
set keytab /etc/v5srvtab

set need_host_key 1
if [file exists $keytab] {
    if [catch [list kadm5_ktlist $keytab] keys] {
	puts "Error checking for $my_princ in keytab $keytab:\n$keys"
	puts "Please fix the problem and try again."
	exit 1
    }
    foreach x $keys {
	set princ [lindex $x 2]
	if ![string compare $princ $my_princ] {
	    set need_host_key 0
	    break
	}
    }
}
puts ""
if $need_host_key {
    kadmin_connect
    puts -nonewline "\nCreating host key ($my_princ) for\napplication services..."
    flush stdout
    if [catch {kadm5_getprinc $my_princ} err] {
	# assume error meant it doesn't exist
	if [catch {kadm5_addprinc -randkey $my_princ} err] {
	    puts "Error creating host key in database:\n  $err"
	    exit 1
	}
    }
    kadm5_ktadd $my_princ
    puts "done.\n"
} else {
    puts "Found a host key ($my_princ) already in"
    puts "$keytab; not creating a new one.\n"
}

if ![string compare master-kdc $insttype]&&[llength $kdc_list]>1 {
    set slavelist {}
    set as [safe_canonicalize $admin_server]
    foreach kdc $kdc_list {
	if [string compare $as [safe_canonicalize $kdc]] {
	    lappend slavelist $kdc
	}
    }
    if [llength $slavelist]<[llength $kdc_list]&&[llength $slavelist]>0 {
	if [catch {
	    if [file exists $krb5_path(prefix)/install/do-kprop] {
		file delete -force $krb5_path(prefix)/install/do-kprop
	    }
	    exec sed -e "s,@realm@,$realm,g" -e "s,@kdclist@,$slavelist,g" \
		    -e "s,@prefix@,$krb5_path(prefix),g" \
		    <$krb5_path(prefix)/install/do-kprop.in \
		    >$krb5_path(prefix)/install/do-kprop
	    exec chmod a+x $krb5_path(prefix)/install/do-kprop
	    puts "\nPlease arrange for $generic_prefix/install/do-kprop to be"
	    puts "run often (perhaps hourly or more often) on this machine.\n"
	} err] {
	    puts "\nError writing slave propagation script:\n  $err"
	    puts "You'll have to fix this up by hand; see the documentation.\n"
	}
    } else {
	# Reality check bounced?
	puts "Error working out the list of slaves.  You'll have to write the slave"
	puts "propagation script yourself.  See"
	puts "      $krb5_path(prefix)/install/do-kprop.in"
	puts "as an example.\n"
    }
}

switch -- $insttype {
    master-kdc	{
	puts "Please arrange for $generic_prefix/install/rc.master to"
	puts "be run at boot time."
    }
    slave-kdc	{
	puts "Please arrange for $generic_prefix/install/rc.slave to"
	puts "be run at boot time, and remember to set up slave propagation"
	puts "from the master KDC.  Do NOT start the KDC on this machine"
	puts "until you have done a database propagation."
    }
    appserver	{
    }
}
fini
