#if defined(_WIN32) && defined(__CYGWIN32__)
#include <windows.h>
#include <exceptions.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include "win32service.h"

#include "krb5.h"
#include "osconf.h"

/* Local #defines. */
#define KRB5_CONFIG_ENV "KRB5_CONFIG"
#define KRB5_CONFIG_ENV_VALUE "etc\\krb5.conf"
#define KRB5_KDC_PROFILE_ENV "KRB5_KDC_PROFILE"
#define KRB5_KDC_PROFILE_ENV_VALUE "etc\\kdc.conf"
#define KERBNET_KADMIN_REG_NAME "SOFTWARE\\Cygnus Solutions\\Kerbnet\\1\\KAdmin"
#define KADMIN_ARGS_REG_NAME "KAdmin args"
#define NOFORK_ARG "-nofork"

/*
 * Included from kdc main code.
 */

extern krb5_sigtype request_exit(int);
extern int original_main(int argc, char **argv);

/*
 * Stop mutex - this will go away when select can be interrupted
 * by a signal in cygwin32.
 */

static HANDLE can_stop = INVALID_HANDLE_VALUE;

/*
 * Calls to lock/unlock mutex.
 */

DWORD lock_exit(DWORD timeout)
{
  DWORD ret;
  if(can_stop == INVALID_HANDLE_VALUE)
    return WAIT_OBJECT_0;
  ret = WaitForSingleObject(can_stop, timeout);
  if(ret == WAIT_ABANDONED || ret == WAIT_FAILED) {
    log_message(LOG_ERR, "Failed to wait for stop mutex. Error was %s",
                    str_oserr(GetLastError()));
  }
  return ret;
}

void unlock_exit()
{
  if(can_stop != INVALID_HANDLE_VALUE)
    ReleaseMutex(can_stop);
}

/*
 * Setup the environment variables KRB5_CONFIG and
 * KRB5_KDC_PROFILE from the registry. 
 */

int setup_environment_variables()
{
  char *kerbnet_home;
  HKEY hkey;
  DWORD type;
  DWORD datasize;
  DWORD home_length;
  DWORD err;

  if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, KERBNET_BASE_REG_NAME, 0,
		KEY_READ, &hkey)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to open Registry key %s. Error was %s",
                KERBNET_BASE_REG_NAME, str_oserr(err));
    return -1;
  }
  
  /* Try and get the KERBNET_HOME value size. */
  datasize = 0;
  if((err = RegQueryValueEx(hkey, KERBNET_HOME_REG_NAME, 0, &type, 0, &datasize)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to find Registry value %s. Error was %s", 
                KERBNET_HOME_REG_NAME, str_oserr(err));
    RegCloseKey(hkey);
    return -1;
  }

  /* Add one char for a possibly missing '\' character */
  datasize += sizeof(char);
  /* Add in the size we will use to append to it. */
  datasize += 20;
 
  /* Allocate the memory for the value */
  if((kerbnet_home = (char *)LocalAlloc(LMEM_FIXED, datasize)) == 0) {
    log_message(LOG_ERR, "Failed to allocate space for value %s. Error was %s", 
                KERBNET_HOME_REG_NAME, str_oserr(GetLastError()));
    RegCloseKey(hkey);
    return -1;
  }

  /* Now get the value */
  if((err = RegQueryValueEx(hkey, KERBNET_HOME_REG_NAME, 0, &type, (LPBYTE)kerbnet_home, 
							&datasize)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to retrieve Registry value %s. Error was %s.", 
                KERBNET_HOME_REG_NAME, str_oserr(err));
    RegCloseKey(hkey);
    LocalFree((HLOCAL)kerbnet_home);
    return -1;
  }

  /* Ensure it ends in a '\' */
   if(kerbnet_home[strlen(kerbnet_home)-1] != '\\')
    strcat(kerbnet_home, "\\");

  home_length = strlen(kerbnet_home);

  /*
   * Set KERBNET_HOME as an environment variable, then set KRB5_CONFIG and
   * KRB5_KDC_PROFILE.
   */
  setenv(KERBNET_HOME_REG_NAME, kerbnet_home, 1);
  /* Add on the extra for KRB5_CONFIG */
  strcat( kerbnet_home, KRB5_CONFIG_ENV_VALUE);
  setenv(KRB5_CONFIG_ENV, kerbnet_home,1);
  /* Truncate back to KERBNET_HOME */
  kerbnet_home[home_length] = '\0';
  /* Add on the extra for KRB5_KDC_PROFILE */
  strcat(kerbnet_home, KRB5_KDC_PROFILE_ENV_VALUE);
  setenv(KRB5_KDC_PROFILE_ENV, kerbnet_home,1);

  LocalFree((HLOCAL)kerbnet_home);
  RegCloseKey(hkey);
  return 0;   
}

/*
 * Add any needed args to the argc, argv array.
 * Fish them from the registry from a REG_MULTI_SZ datatype.
 * By default there is always one extra arg, NOFORK_ARG.
 */
 
int add_extra_args( struct argc_argv *args)
{
  int extra_args;
  char *kadmin_args = 0;
  char *p;
  char **new_argv;
  int i;
  DWORD datasize;
  DWORD type;
  HKEY hkey;
  DWORD err;

  if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, KERBNET_KADMIN_REG_NAME, 0,
		KEY_READ, &hkey)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to open Registry key %s. Error was %s", KERBNET_KADMIN_REG_NAME,
		    str_oserr(err));
    return -1;
  }
  
  /* Try and get the Kdc args value size. */
  datasize = 0;
  if((err = RegQueryValueEx(hkey, KADMIN_ARGS_REG_NAME, 0, &type, 0, &datasize)) == ERROR_SUCCESS) {
    /* Allocate the memory for the value */
    if((kadmin_args = (char *)LocalAlloc(LMEM_FIXED, datasize)) == 0) {
      log_message(LOG_ERR, "Failed to allocate space for value %s. Error was %s.", 
                  KADMIN_ARGS_REG_NAME, str_oserr(GetLastError()));
      RegCloseKey(hkey);
      return -1;
    }

    /* Now get the value */
    if((err = RegQueryValueEx(hkey, KADMIN_ARGS_REG_NAME, 0, &type, (LPBYTE)kadmin_args, 
  							&datasize)) != ERROR_SUCCESS) {
      log_message(LOG_ERR, "Failed to retrieve Registry value %s. Error was %s", 
                  KADMIN_ARGS_REG_NAME, str_oserr(err));
      RegCloseKey(hkey);
      LocalFree((HLOCAL)kadmin_args);
      return -1;
    }
  }

  RegCloseKey(hkey);

  /* Now go through the REG_MULTI_SZ kadmin_args and add them into a re-allocated
     argv array. extra_args starts at 1 due to the NOFORK_ARG arg. */
  extra_args = 1;
  if(kadmin_args != 0) {
    for( p = kadmin_args; *p; p += strlen(p) + 1, ++extra_args)
      ;
  }

  if((new_argv = (char **)LocalAlloc(LMEM_FIXED,
                          (args->argc + extra_args + 1)*sizeof(char *)))==0) {
      log_message(LOG_ERR,
                      "Failed to allocate space for new kadmin arg array. Error was %s", 
                      str_oserr(GetLastError()));
      LocalFree((HLOCAL)kadmin_args);
      return -1;
  }

  /* Setup nofork as the first extra arg. */
  new_argv[0] = args->argv[0];
  new_argv[1] = NOFORK_ARG;

  /* Add in the remaining args from the command line. */
  for( i = 1; i < args->argc; ++i)
    new_argv[i+1] = args->argv[i];

  /* Increment by one for the extra NOFORK_ARG. */
  ++i;
  
  /* Now add in the args from the registry */
  if(kadmin_args != 0) {
    for( p = kadmin_args; *p; p += strlen(p) + 1) {
      if(strcmp(p, NOFORK_ARG) == 0)
        continue; /* Don't need two NOFORK_ARG args. */
      new_argv[i++] = p;
    }
  }

  /* Null terminate argv array */
  new_argv[i] == 0;
  args->argv = new_argv;
  args->argc = i;
  return 0;
}

/*
 * Code called by boilerplate service code.
 */
 
int do_service_install(int *argc_ret, char **argv, int *cont)
{
  int err;
  int i, saved_i;
  int argc = *argc_ret;
  SECURITY_DESCRIPTOR sd;
  char username[512];
  DWORD usize = sizeof(username);

  *cont = TRUE;

  /*
   * Check we are running as Administrator - exit otherwise.
   */

  if(GetUserName(username, &usize) == FALSE) {
    *cont = FALSE;
    log_message(LOG_ERR, "do_service_install: GetUserName failed. Error was %s.",
                str_oserr(GetLastError()));
    return -1;
  }

  if(strcasecmp(username, "Administrator") != 0) {
    log_message(LOG_ERR, "You must be running as user Administrator to use this program. You are currently user %s.", username);
    *cont = FALSE;
    free(username);
    return -1;
  }

  free(username);

  /* Get the required Security Descriptor. */
  if(create_sd_from_list( &sd, 3, "Administrators", GENERIC_ALL,
                                  "SYSTEM", GENERIC_ALL,
                                  "Users", GENERIC_READ) == FALSE) {
    *cont = FALSE;
    return err;
  }

  /* Ensure that the registry heirarchy exists */
  if(err = create_registry_tree( HKEY_LOCAL_MACHINE, KERBNET_KADMIN_REG_NAME, &sd)) {
    *cont = FALSE;
    return err;
  }

  /*
   * If we are invoked as /install /args then we are being invoked
   * to change the args in the registry path
   * SOFTWARE\Cygnus Solutions\Kerbnet\1\KAdmin\KAdmin args
   * - thus don't check the the values under
   * SOFTWARE\Cygnus Solutions\Kerbnet\1
   *  are correct.
   */
  if((argc >= 3) && (*argv[2] == '/' || *argv[2] == '-') && (strcasecmp( &argv[2][1], "args") == 0)) {
    err = setup_service_args( HKEY_LOCAL_MACHINE, KERBNET_KADMIN_REG_NAME,
                              KADMIN_ARGS_REG_NAME, argc - 3, &argv[3]);
    *cont = FALSE;
    return err;
  }

  /*
   * We are really being installed. Add the args
   * given on the command line.
   */
  if((err = check_kerbnet_base_args()) != 0) {
    *cont = FALSE;
    return err;
  }
  for( i = 2; i < argc; i++)
    if(((argv[i][0] != '/') && (argv[i][0] != '-')) ||
       ((strncasecmp( &argv[i][1], "username", 9) != 0) &&
        (strncasecmp( &argv[i][1], "password", 9) != 0)))
      break;

  if(i == argc) /* No args */
    return 0;

  saved_i = i;

  if((argv[i] != 0) && (*argv[i] == '/' || *argv[i] == '-') && (strcasecmp( &argv[i][1], "args") == 0))
    i++;

  err = setup_service_args( HKEY_LOCAL_MACHINE, KERBNET_KADMIN_REG_NAME,
                            KADMIN_ARGS_REG_NAME, argc - i, &argv[i]);
  /* Truncate the args here */
  argv[saved_i] = 0;
  *argc_ret = saved_i;
  return err;
}


int do_service_delete(int *argc, char **argv, int *cont)
{
  DWORD err;
  *cont = TRUE;

  if((*argc == 3) && (*argv[2] == '/' || *argv[2] == '-') && (strcasecmp( &argv[2][1], "registry") == 0)) {
    *cont = FALSE;
    if(err = delete_registry_tree( HKEY_LOCAL_MACHINE, KERBNET_KADMIN_REG_NAME)) 
      return err;
  }
  return 0;
}

/*
 * Exception handler.
 */

void except_handler(EXCEPTION_RECORD *a, void *b, CONTEXT *c, void *d)
{
  log_message(LOG_ERR, "Exception trapped. Service terminating. Code = %x, at ip = %x",
            a->ExceptionCode, c->Eip);
  exit(0);
}

/*
 * This is x86 specific. We need to make this generic for
 * ALPHA also (as currently that's the only other processor
 * NT runs on, with PowerPC being cancelled).
 */
 
extern exception_list *_except_list asm ("%fs:0");

/*
 * This is called in a separate thread.
 */

int do_service_main(struct argc_argv *args)
{
  int ret;
  
  /*
   * Catch any exceptions.
   */
  exception_list el;
  el.handler = except_handler;
  el.prev = _except_list;
  _except_list = &el;

  /* Open the syslog. */
  openlog( global_service_name, LOG_PID, LOG_DAEMON);
  
  /*
   * Create the can_stop mutex.
   */
  if((can_stop = CreateMutex( 0, FALSE, 0)) == 0) {
    log_message(LOG_ERR, "Failed to create stop mutex. Error was %s", 
                str_oserr(GetLastError()));
    return -1;
  }

  if(global_debug_flag == TRUE) {
    /*
     * Running in foreground - just call
     * original_main() after removing -debug arg.
     */
    int i;
    for( i = 0; i < args->argc; ++i) {
      if(strcasecmp(&args->argv[i][1], "debug")==0) {
        --(args->argc);
      	for(; i < args->argc; ++i) {
      	  args->argv[i] = args->argv[i+1];
      	}
        args->argv[args->argc] = 0;
      	break;
      }
    }
    ret = original_main(args->argc, args->argv);
    closelog();
    return ret;
  }

  /*
   * Running as a service.
   * Set up the neccessary environment variables from
   * the registry. The need to be KRB5_CONFIG and KRB5_KDC_PROFILE.
   */
  if(setup_environment_variables() != 0)
    return -1;

  /*
   * Add in the neccessary
   * args then call the original_main().
   */
  if(add_extra_args(args) != 0)
    return -1;

  /* Tell the service manager we have started. */
  TellServiceManager(SERVICE_RUNNING, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
  
  /* Inform the event log we have been started. */
  syslog( LOG_INFO, "Service %s has been started", global_service_name);
  ret = original_main(args->argc, args->argv);

  /* Tell the service manager we have stopped. */
  TellServiceManager(SERVICE_STOPPED, 0, 0, 0);
  closelog();
  return ret;
}

/*
 * Call to ask service to stop. Currently, as cygwin32 does not
 * interrupt a select when a signal fires, we have to use the
 * can_stop mutex and TerminateThread(). When this is fixed we
 * will just use kill().
 */
 
int request_service_stop(SERVICE_STATUS *ssh)
{
  openlog( global_service_name, LOG_PID, LOG_DAEMON);
  lock_exit(5000);
  request_exit(SIGINT);
  TerminateThread(global_service_thread, 0);
  unlock_exit();
  /* Log that we exited. */
  syslog( LOG_INFO, "Service %s has been stopped", global_service_name);
  closelog();
  return 0;
}

/*
 * Request the service pause. Get the mutex
 * and then suspend the thread.
 */
 
int request_service_pause(SERVICE_STATUS *ssh)
{
  openlog( global_service_name, LOG_PID, LOG_DAEMON);
  if(lock_exit(4500) != WAIT_OBJECT_0) {
    /* Failed to get mutex */
    TellServiceManager(SERVICE_RUNNING, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
    closelog();
    return 0;
  }
  SuspendThread(global_service_thread);
  TellServiceManager(SERVICE_PAUSED, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
  closelog();
  return 0;
}

/*
 * Request the service continue. Release the thread
 * then release the mutex.
 */

int request_service_continue(SERVICE_STATUS *ssh)
{
  openlog( global_service_name, LOG_PID, LOG_DAEMON);
  unlock_exit();
  ResumeThread(global_service_thread);
  TellServiceManager(SERVICE_RUNNING, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
  closelog();
  return 0;
}

int do_service_usage(int argc, char **argv)
{
  extern void param_usage(FILE *);

  printf("usage: kadmin.exe\n");
  printf("\t/install          -  installs service & creates registry entries.\n");
  printf("\t/delete           -  deletes service & removes registry entries.\n");
  printf("\t/install /args arg1 [arg2] ... [argn] - adds arguments into registry key. Args can be :\n");
  param_usage(stdout);
  printf("\t/delete /args     - removes arguments from registry key.\n");
  return 0;
}

#endif /* _WIN32 && __CYGWIN32__ */

