/*
 * Copyright, OpenVision Technologies, Inc., 1996, All Rights Reserved
 *
 * WARNING:  Retrieving the OpenVision Kerberos Administration system
 * source code, as described below, indicates your acceptance of the
 * following terms.  If you do not agree to the following terms, do not
 * retrieve the OpenVision Kerberos administration system.
 *
 * You may freely use and distribute the Source Code and Object Code
 * compiled from it, but this Source Code is provided to you "AS IS"
 * EXCLUSIVE OF ANY WARRANTY, INCLUDING, WITHOUT LIMITATION, ANY
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR
 * ANY OTHER WARRANTY, WHETHER EXPRESS OR IMPLIED.  IN NO EVENT WILL
 * OPENVISION HAVE ANY LIABILITY FOR ANY LOST PROFITS, LOSS OF DATA OR
 * COSTS OF PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, OR FOR ANY
 * SPECIAL, INDIRECT, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THIS
 * AGREEMENT, INCLUDING, WITHOUT LIMITATION, THOSE RESULTING FROM THE
 * USE OF THE SOURCE CODE, OR THE FAILURE OF THE SOURCE CODE TO PERFORM,
 * OR FOR ANY OTHER REASON.
 *
 * OpenVision retains all rights, title, and interest in the donated
 * Source Code.  With respect to OpenVision's copyrights in the donated
 * Source Code, OpenVision also retains rights to derivative works of
 * the Source Code whether created by OpenVision or a third party.
 *
 * OpenVision Technologies, Inc. has donated this Kerberos
 * Administration system to MIT for inclusion in the standard Kerberos 5
 * distribution. This donation underscores our commitment to continuing
 * Kerberos technology development and our gratitude for the valuable
 * work which has been performed by MIT and the Kerberos community.
 */

/*
 * $Header: /afs/athena.mit.edu/astaff/project/krbdev/.cvsroot/src/kadmin/server/ovsec_kadmd.c,v 1.55 1996/08/12 15:50:19 bjaspan Exp $
 */

#if !defined(lint) && !defined(__CODECENTER__)
static char *rcsid = "$Header: /afs/athena.mit.edu/astaff/project/krbdev/.cvsroot/src/kadmin/server/ovsec_kadmd.c,v 1.55 1996/08/12 15:50:19 bjaspan Exp $";
#endif

#include    <stdio.h>
#include    <signal.h>
#include    <syslog.h>
#include    <sys/types.h>
#ifdef _AIX
#include    <sys/select.h>
#endif
#include    <sys/time.h>
#include    <sys/socket.h>
#include    <unistd.h>
#include    <netinet/in.h>
#include    <arpa/inet.h>  /* inet_ntoa */
#include    <netdb.h>
#include    <rpc/rpc.h>
#include    <gssapi/gssapi_krb5.h>
#include    <rpc/auth_gssapi.h>
#include    <kadm5/admin.h>
#include    <kadm5/kadm_rpc.h>
#include    <string.h>
#include    <errno.h>
#include    "getopt.h"

#ifdef PURIFY
#include    "purify.h"

int	signal_pure_report = 0;
int	signal_pure_clear = 0;
void	request_pure_report(int);
void	request_pure_clear(int);
#endif /* PURIFY */

int	signal_request_exit = 0;
int	signal_request_reset = 0;
void	request_exit(int);
void	request_reset_db(int);
void	reset_db(void);
void	sig_pipe(int);
void	kadm_svc_run(void);

#define	TIMEOUT	15

gss_name_t gss_changepw_name = NULL, gss_oldchangepw_name = NULL;
void *global_server_handle;

/*
 * This is a kludge, but the server needs these constants to be
 * compatible with old clients.  They are defined in <kadm5/admin.h>,
 * but only if USE_KADM5_API_VERSION == 1.
 */
#define OVSEC_KADM_ADMIN_SERVICE	"ovsec_adm/admin"
#define OVSEC_KADM_CHANGEPW_SERVICE	"ovsec_adm/changepw"

/*
 * This enables us to set the keytab that gss_acquire_cred uses, but
 * it also restricts us to linking against the Kv5 GSS-API library.
 * Since this is *k*admind, that shouldn't be a problem.
 */
extern 	char *krb5_defkeyname;

char *build_princ_name(char *name, char *realm);
void log_badauth(OM_uint32 major, OM_uint32 minor,
		 struct sockaddr_in *addr, char *data);
void log_badverf(gss_name_t client_name, gss_name_t server_name,
		 struct svc_req *rqst, struct rpc_msg *msg,
		 char *data);
void log_miscerr(struct svc_req *rqst, struct rpc_msg *msg, char
		 *error, char *data);
void log_badauth_display_status(OM_uint32 code, int type, int *n, 
				char **outbuf);
void log_badauth_free_buffers(int nbuf, char **buffers);
	
int schpw;
void do_schpw(int s, kadm5_config_params *params);
kadm5_config_params params;
krb5_error_code process_chpw_request(krb5_context context, void *server_handle,
				     char *realm, int s, krb5_keytab keytab,
				     struct sockaddr_in *sin,
				     krb5_data *req, krb5_data *rep);

#ifdef __CYGWIN32__
/*
 * These two rather gross functions are neccessary for a kadmin
 * service as cygwin32 currently doesn't allow signals to
 * interrupt a select(). So we ensure our thread cannot be
 * terminated when processing a users command.
 */
extern int lock_exit(unsigned int);
extern int unlock_exit();
#endif /* __CYGWIN32__ */


/*
 * Function: param_usage
 * 
 * Purpose: print out the server usage message parameters.
 *
 * Arguments: FILE * to print out to.
 * Requires:
 * Effects:
 * Modifies:
 */

void param_usage(FILE *fp)
{
  fprintf(fp, "\t[-r realm] [-m] [-n | --nofork] "
             "[-p | --port port-number]\n");
}

/*
 * Function: usage
 * 
 * Purpose: print out the server usage message
 *
 * Arguments:
 * Requires:
 * Effects:
 * Modifies:
 */

void usage()
{
     fprintf(stderr, "Usage: kadmind ");
     param_usage(stderr);
     exit(1);
}

/* Variables for getopt() */
int off_option;
struct option long_options[] = {
    { "version", 0, NULL, 0x01 },
    { "nofork", 0, NULL, 'n' },
    { "port", 0, NULL, 'p' },
    { NULL, 0, NULL, 0 }
};

/* XXX yuck.  the signal handlers need this */
static krb5_context context;

#ifdef __CYGWIN32__
int original_main(int argc, char *argv[])
#else /* __CYGWIN32__ */
int main(int argc, char *argv[])
#endif /* __CYGWIN32__ */
{
     void	kadm_1(struct svc_req *, SVCXPRT *);
     register	SVCXPRT *transp;
     extern	char *optarg;
     extern	int optind, opterr;
     int ret, rlen, nofork, oldnames = 0;
     OM_uint32 OMret;
     char *whoami;
     FILE *acl_file;
     gss_buffer_desc in_buf;
     struct servent *srv;
     struct sockaddr_in addr;
     int s,i, index;
     short port = 0;
     auth_gssapi_name names[4];

     names[0].name = names[1].name = names[2].name = names[3].name = NULL;
     names[0].type = names[1].type = names[2].type = names[3].type =
	  gss_nt_krb5_name;

#ifdef PURIFY
     purify_start_batch();
#endif /* PURIFY */
     whoami = (strrchr(argv[0], '/') ? strrchr(argv[0], '/')+1 : argv[0]);

     nofork = 0;

     memset((char *) &params, 0, sizeof(params));
     
    while ((i = getopt_long(argc, argv, "r:mno:p:",
      long_options, &index)) != EOF) {
	switch(i) {
	case 'r':
	    params.mask |= KADM5_CONFIG_REALM;
	    params.realm = optarg;
	    break;
	case 'm':
	    params.mask |= KADM5_CONFIG_MKEY_FROM_KBD;
	    params.mkey_from_kbd = 1;
	    break;
	case 'n':
	    nofork = 1;
	    break;
	case 'o':
	    /* Kludge case for -nofork */
	    break;
	case 'p':
	    if (*optarg == 'o') /* Kludge case for -port */
		optarg = argv[++optind];
	    params.kadmind_port = atoi(optarg);
	    params.mask |= KADM5_CONFIG_KADMIND_PORT;
	case 1: /* Print the version */
            printf("%s\n", krb5_version);
            exit(0);
	case '?':
	default:
	    usage();
	    break;
	}
     }
     
     if (argc - optind != 0)
	  usage();

     if (ret = krb5_init_context(&context)) {
	  fprintf(stderr, "%s: %s while initializing context, aborting\n",
		  whoami, error_message(ret));
	  exit(1);
     }

     krb5_klog_init(context, "admin_server", whoami, 1);

     if((ret = kadm5_init("kadmind", NULL,
			  NULL, &params,
			  KADM5_STRUCT_VERSION,
			  KADM5_API_VERSION_2,
			  &global_server_handle)) != 
	KADM5_OK) {
	 krb5_klog_generic(context, LOG_ERR, "main", 
			   "initialization failed", NULL, 
			   "msg", error_message(ret), NULL);
	  fprintf(stderr, "%s: %s while initializing, aborting\n",
		  whoami, error_message(ret));
	  krb5_klog_close();
	  exit(1);
     }
     
     if (ret = kadm5_get_config_params(context, NULL, NULL, &params,
				       &params)) {
	 
	 krb5_klog_generic(context, LOG_ERR, "main", 
			   "initialization failed", NULL, 
			   "msg", error_message(ret), NULL);
	  fprintf(stderr, "%s: %s while initializing, aborting\n",
		  whoami, error_message(ret));
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();
	  exit(1);
     }

#define REQUIRED_PARAMS (KADM5_CONFIG_REALM | KADM5_CONFIG_ACL_FILE | \
			 KADM5_CONFIG_ADMIN_KEYTAB)

     if ((params.mask & REQUIRED_PARAMS) != REQUIRED_PARAMS) {
	 char flagbuf[32];
	 sprintf(flagbuf, "%x", 
		 (params.mask & REQUIRED_PARAMS) ^ REQUIRED_PARAMS);
	 krb5_klog_generic(context, LOG_ERR, "main", 
			   "initialization failed", NULL, 
			   "msg", "missing configuration values", 
			   "needs", flagbuf , NULL);
	  fprintf(stderr, "%s: Missing required configuration values "
		  "(%x) while initializing, aborting\n", whoami,
		  (params.mask & REQUIRED_PARAMS) ^ REQUIRED_PARAMS);
	  krb5_klog_close();
	  kadm5_destroy(global_server_handle);
	  exit(1);
     }

     if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
	 krb5_klog_generic(context, LOG_ERR, "main", 
			   "cannot create kadmind socket", NULL, 
			   "msg", error_message(errno), NULL);
	 fprintf(stderr, "Cannot create kadmind socket: %s",
		 error_message(errno));
	 kadm5_destroy(global_server_handle);
	 krb5_klog_close();	  
	 exit(1);
     }

     if ((schpw = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
 	  krb5_klog_generic(context, LOG_ERR, "main", 
			    "cannot create simple chpw socket", NULL, 
			    "msg", error_message(errno), NULL);
	  fprintf(stderr, "Cannot create simple chpw socket: %s",
		  error_message(errno));
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();	  
	  exit(1);
     }

#ifdef SO_REUSEADDR
     /* the old admin server turned on SO_REUSEADDR for non-default
	port numbers.  this was necessary, on solaris, for the tests
	to work.  jhawk argues that the debug and production modes
	should be the same.  I think I agree, so I'm always going to set
	SO_REUSEADDR.  The other option is to have the unit tests wait
	until the port is useable, or use a different port each time.  
	--marc */

     {
	 int	allowed;

	 allowed = 1;
	 if (setsockopt(s,
			SOL_SOCKET,
			SO_REUSEADDR,
			(char *) &allowed,
			sizeof(allowed)) < 0) {
	     krb5_klog_generic(context, LOG_ERR, "main", 
			       "cannot set SO_REUSEADDR on kadmind socket", 
			       NULL, "msg", error_message(errno), NULL);
	     fprintf(stderr, "Cannot set SO_REUSEADDR: %s", 
		     error_message(errno));
	     kadm5_destroy(global_server_handle);
	     krb5_klog_close();	  
	     exit(1);
	 }
	 if (setsockopt(schpw,
			SOL_SOCKET,
			SO_REUSEADDR,
			(char *) &allowed,
			sizeof(allowed)) < 0) {
	     krb5_klog_generic(context, LOG_ERR, "main", 
			       "cannot set SO_REUSEADDR on simple chpw socket", 
			       NULL, "msg", error_message(errno), NULL);
	     fprintf(stderr,
		     "Cannot set SO_REUSEADDR on simple chpw socket: %s",
		     error_message(errno));
	     kadm5_destroy(global_server_handle);
	     krb5_klog_close();	  
	     exit(1);
	 }
     }
#endif /* SO_REUSEADDR */
     memset(&addr, 0, sizeof(addr));
     addr.sin_family = AF_INET;
     addr.sin_addr.s_addr = INADDR_ANY;
     addr.sin_port = htons(params.kadmind_port);

     if (bind(s, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
	  int oerrno = errno;
	  char portbuf[32];
	  fprintf(stderr, "%s: Cannot bind socket.\n", whoami);
	  fprintf(stderr, "bind: %s\n", error_message(oerrno));
	  sprintf(portbuf, "%d", ntohs(addr.sin_port));
	  krb5_klog_generic(context, LOG_ERR, "main", 
			    "cannot bind kadmind socket", NULL, 
			    "msg", error_message(oerrno), 
			    "port", portbuf, NULL);
	  if(oerrno == EADDRINUSE) {
	       char *w = strrchr(whoami, '/');
	       if (w) {
		    w++;
	       }
	       else {
		    w = whoami;
	       }
	       fprintf(stderr,
"This probably means that another %s process is already\n"
"running, or that another program is using the server port (number %d)\n"
"after being assigned it by the RPC portmap deamon.  If another\n"
"%s is already running, you should kill it before\n"
"restarting the server.  If, on the other hand, another program is\n"
"using the server port, you should kill it before running\n"
"%s, and ensure that the conflict does not occur in the\n"
"future by making sure that %s is started on reboot\n"
		       "before portmap.\n", w, ntohs(addr.sin_port), w, w, w);
	  }
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();	  
	  exit(1);
     }
     
     memset(&addr, 0, sizeof(addr));
     addr.sin_family = AF_INET;
     addr.sin_addr.s_addr = INADDR_ANY;
     /* XXX */
     addr.sin_port = htons(params.kpasswd_port);

     if (bind(schpw, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
	  char portbuf[32];
	  int oerrno = errno;
	  fprintf(stderr, "%s: Cannot bind socket.\n", whoami);
	  fprintf(stderr, "bind: %s\n", error_message(oerrno));
	  errno = oerrno;
	  sprintf(portbuf, "%d", ntohs(addr.sin_port));
	  krb5_klog_generic(context, LOG_ERR, "main", 
			    "cannot bind simple chpw socket", NULL, 
			    "msg", error_message(oerrno), 
			    "port", portbuf, NULL);
	  if(oerrno == EADDRINUSE) {
	       char *w = strrchr(whoami, '/');
	       if (w) {
		    w++;
	       }
	       else {
		    w = whoami;
	       }
	       fprintf(stderr,
"This probably means that another %s process is already\n"
"running, or that another program is using the server port (number %d).\n"
"If another %s is already running, you should kill it before\n"
"restarting the server.\n",
		       w, ntohs(addr.sin_port), w, w, w);
	  }
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();	  
	  exit(1);
     }
     
     transp = svctcp_create(s, 0, 0);
     if(transp == NULL) {
	  fprintf(stderr, "%s: Cannot create RPC service.\n", whoami);
	  krb5_klog_generic(context, LOG_ERR, "main", 
			    "cannot create RPC service", NULL, 
			    "msg", error_message(errno), NULL);
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();	  
	  exit(1);
     }
     if(!svc_register(transp, KADM, KADMVERS, kadm_1, 0)) {
	  fprintf(stderr, "%s: Cannot register RPC service.\n", whoami);
	  krb5_klog_generic(context, LOG_ERR, "main", 
			    "cannot register RPC service", NULL, NULL);
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();	  
	  exit(1);
     }

     names[0].name = build_princ_name(KADM5_ADMIN_SERVICE, params.realm);
     names[1].name = build_princ_name(KADM5_CHANGEPW_SERVICE, params.realm);
     names[2].name = build_princ_name(OVSEC_KADM_ADMIN_SERVICE, params.realm);
     names[3].name = build_princ_name(OVSEC_KADM_CHANGEPW_SERVICE,
				      params.realm); 
     if (names[0].name == NULL || names[1].name == NULL ||
	 names[2].name == NULL || names[3].name == NULL) {
	 krb5_klog_generic(context, LOG_ERR, "main", 
			   "cannot initialize GSS-API authentication", NULL, 
			   "notes", "couldn't get principal names", NULL);
	  fprintf(stderr, "%s: Cannot initialize GSS-API authentication (1).\n",
		  whoami);
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();	  
	  exit(1);
     }

     krb5_defkeyname = params.admin_keytab;

     /*
      * Try to acquire creds for the old OV services as well as the
      * new names, but if that fails just fall back on the new names.
      */
     if (_svcauth_gssapi_set_names(names, 4) == TRUE)
	  oldnames++;
     if (!oldnames && _svcauth_gssapi_set_names(names, 2) == FALSE) {
	 krb5_klog_generic(context, LOG_ERR, "main", 
			   "cannot initialize GSS-API authentication", NULL, 
			   "admin_service", names[0].name ? names[0].name : "",
			   "changepw_service", 
			   names[1].name ? names[1].name : "", NULL);
	  fprintf(stderr, "%s: Cannot initialize GSS-API authentication (2).\n",
		  whoami);
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();	  
	  exit(1);
     }

     /* if set_names succeeded, this will too */
     in_buf.value = names[1].name;
     in_buf.length = strlen(names[1].name) + 1;
     (void) gss_import_name(&OMret, &in_buf, gss_nt_krb5_name,
			    &gss_changepw_name);
     if (oldnames) {
	  in_buf.value = names[3].name;
	  in_buf.length = strlen(names[3].name) + 1;
	  (void) gss_import_name(&OMret, &in_buf, gss_nt_krb5_name,
				 &gss_oldchangepw_name);
     }

     _svcauth_gssapi_set_log_badauth_func(log_badauth, NULL);
     _svcauth_gssapi_set_log_badverf_func(log_badverf, NULL);
     _svcauth_gssapi_set_log_miscerr_func(log_miscerr, NULL);
     
     if (ret = acl_init(context, 0, params.acl_file)) {
	 krb5_klog_generic(context, LOG_ERR, "main", "couldn't init acl file", 
			   NULL, "msg", error_message(ret), 
			   "acl_file", params.acl_file, NULL);
	  fprintf(stderr, "%s: Cannot initialize acl file: %s\n",
		  whoami, error_message(ret));
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();
	  exit(1);
     }

     if (!nofork && (ret = daemon(0, 0))) {
	  ret = errno;
	  krb5_klog_generic(context, LOG_ERR, "main", "couldn't detach tty", 
			    NULL, "msg", error_message(ret), NULL);
	  fprintf(stderr, "%s: Cannot detach from tty: %s\n",
		  whoami, error_message(ret));
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();
	  exit(1);
     }

/* Currently Cygwin32 doesn't handle signals correctly. */     
#ifndef __CYGWIN32__
     signal(SIGINT, request_exit);
     signal(SIGTERM, request_exit);
     signal(SIGQUIT, request_exit);
     signal(SIGHUP, request_reset_db);
     signal(SIGPIPE, sig_pipe);
#endif /* __CYGWIN32__ */
#ifdef PURIFY
     signal(SIGUSR1, request_pure_report);
     signal(SIGUSR2, request_pure_clear);
#endif /* PURIFY */
     krb5_klog_generic(context, LOG_INFO, "main", "starting", NULL, NULL);

     kadm_svc_run();
     krb5_klog_generic(context, LOG_INFO, "main", "exiting", NULL, NULL);
     kadm5_destroy(global_server_handle);
     close(s);
     krb5_klog_close();
     exit(2);
}

/*
 * Function: kadm_svc_run
 * 
 * Purpose: modified version of sunrpc svc_run.
 *	    which closes the database every TIMEOUT seconds.
 *
 * Arguments:
 * Requires:
 * Effects:
 * Modifies:
 */

void kadm_svc_run(void)
{
     fd_set	rfd;
     int	sz = _rpc_dtablesize();
     struct	timeval	    timeout;
     
     while(signal_request_exit == 0) {
	  if (signal_request_reset)
	    reset_db();
#ifdef PURIFY
	  if (signal_pure_report)	/* check to see if a report */
					/* should be dumped... */
	    {
	      purify_new_reports();
	      signal_pure_report = 0;
	    }
	  if (signal_pure_clear)	/* ...before checking whether */
					/* the info should be cleared. */
	    {
	      purify_clear_new_reports();
	      signal_pure_clear = 0;
	    }
#endif /* PURIFY */
	  timeout.tv_sec = TIMEOUT;
	  timeout.tv_usec = 0;
	  rfd = svc_fdset;
	  FD_SET(schpw, &rfd);
	  switch(select(sz, (fd_set *) &rfd, NULL, NULL, &timeout)) {
	  case -1:
	       if(errno == EINTR)
		    continue;
	       perror("select");
	       return;
	  case 0:
#ifdef __CYGWIN32__
	       lock_exit(0xFFFFFFFF);
#endif /* __CYGWIN32__ */          
	       reset_db();
#ifdef __CYGWIN32__
	       unlock_exit();
#endif /* __CYGWIN32__ */          
	       break;
	  default:
#ifdef __CYGWIN32__
               lock_exit(0xFFFFFFFF);
#endif /* __CYGWIN32__ */          
	       if (FD_ISSET(schpw, &rfd))
		    do_schpw(schpw, &params);
	       else
		    svc_getreqset(&rfd);
#ifdef __CYGWIN32__
	       unlock_exit();
#endif /* __CYGWIN32__ */          
	  }
     }
}

#ifdef PURIFY
/*
 * Function: request_pure_report
 * 
 * Purpose: sets flag saying the server got a signal and that it should
 *		dump a purify report when convenient.
 *
 * Arguments:
 * Requires:
 * Effects:
 * Modifies:
 *	sets signal_pure_report to one
 */

void request_pure_report(int signum)
{
    krb5_klog_generic(context, LOG_DEBUG, "purify", 
		      "request for report received", NULL, NULL);
     signal_pure_report = 1;
     return;
}

/*
 * Function: request_pure_clear
 * 
 * Purpose: sets flag saying the server got a signal and that it should
 *		dump a purify report when convenient, then clear the
 *		purify tables.
 *
 * Arguments:
 * Requires:
 * Effects:
 * Modifies:
 *	sets signal_pure_report to one
 *	sets signal_pure_clear to one
 */

void request_pure_clear(int signum)
{
    krb5_klog_generic(context, LOG_DEBUG, "purify", 
		      "request for report and clear info received", 
		      NULL, NULL);
     signal_pure_report = 1;
     signal_pure_clear = 1;
     return;
}
#endif /* PURIFY */

/*
 * Function: request_reset_db
 * 
 * Purpose: sets flag saying the server got a signal and that it should
 *		reset the database files when convenient.
 *
 * Arguments:
 * Requires:
 * Effects:
 * Modifies:
 *	sets signal_request_reset to one
 */

void request_reset_db(int signum)
{
    krb5_klog_generic(context, LOG_DEBUG, "main", 
		      "request for database reset received", 
		      NULL, NULL);
     signal_request_reset = 1;
     return;
}

/*
 * Function: reset-db
 * 
 * Purpose: flushes the currently opened database files to disk.
 *
 * Arguments:
 * Requires:
 * Effects:
 * 
 * Currently, just sets signal_request_reset to 0.  The kdb and adb
 * libraries used to be sufficiently broken that it was prudent to
 * close and reopen the databases periodically.  They are no longer
 * that broken, so this function is not necessary.
 */
void reset_db(void)
{
#ifdef notdef
     kadm5_ret_t ret;
     
     if (ret = kadm5_flush(global_server_handle)) {
	 krb5_klog_generic(context, LOG_ERR, "main", 
			   "FATAL ERROR when flushing database.  Check database for corruption.", NULL, 
			   "msg", error_message(ret), 
			   NULL);
	  krb5_klog_close();
	  exit(3);
     }
#endif

     signal_request_reset = 0;
     return;
}

/*
 * Function: request-exit
 * 
 * Purpose: sets flags saying the server got a signal and that it
 *	    should exit when convient.
 *
 * Arguments:
 * Requires:
 * Effects:
 *	modifies signal_request_exit which ideally makes the server exit
 *	at some point.
 *
 * Modifies:
 *	signal_request_exit
 */

void request_exit(int signum)
{
    krb5_klog_generic(context, LOG_DEBUG, "main", "exit request received", 
		      NULL, NULL);
     signal_request_exit = 1;
     return;
}

/*
 * Function: sig_pipe
 *
 * Purpose: SIGPIPE handler
 *
 * Effects: krb5_klog_syslogs a message that a SIGPIPE occurred and returns,
 * thus causing the read() or write() to fail and, presumable, the RPC
 * to recover.  Otherwise, the process aborts.
 */
void sig_pipe(int unused)
{
    krb5_klog_generic(context, LOG_NOTICE, "main", 
		      "SIGPIPE received; probably an aborted client",
		      NULL, NULL);
     return;
}

/*
 * Function: build_princ_name
 * 
 * Purpose: takes a name and a realm and builds a string that can be
 *	    consumed by krb5_parse_name.
 *
 * Arguments:
 *	name		    (input) name to be part of principal
 *	realm		    (input) realm part of principal
 * 	<return value>	    char * pointing to "name@realm"
 *
 * Requires:
 *	name be non-null.
 * 
 * Effects:
 * Modifies:
 */

char *build_princ_name(char *name, char *realm)
{
     char *fullname;

     fullname = (char *) malloc(strlen(name) + 1 +
				(realm ? strlen(realm) + 1 : 0));
     if (fullname == NULL)
	  return NULL;
     if (realm)
	  sprintf(fullname, "%s@%s", name, realm);
     else
	  strcpy(fullname, name);
     return fullname;
}

/*
 * Function: log_badverf
 *
 * Purpose: Call from GSS-API Sun RPC for garbled/forged/replayed/etc
 * messages.
 *
 * Argiments:
 * 	client_name	(r) GSS-API client name
 * 	server_name	(r) GSS-API server name
 * 	rqst		(r) RPC service request
 * 	msg		(r) RPC message
 * 	data		(r) arbitrary data (NULL), not used
 *
 * Effects:
 *
 * Logs the invalid request via krb5_klog_syslog(); see functional spec for
 * format.
 */
void log_badverf(gss_name_t client_name, gss_name_t server_name,
		 struct svc_req *rqst, struct rpc_msg *msg, char
		 *data)
{
     static const char *const proc_names[] = {
	  "kadm5_create_principal",
	  "kadm5_delete_principal",
	  "kadm5_modify_principal",
	  "kadm5_rename_principal",
	  "kadm5_get_principal",
	  "kadm5_chpass_principal",
	  "kadm5_randkey_principal",
	  "kadm5_create_policy",
	  "kadm5_delete_policy",
	  "kadm5_modify_policy",
	  "kadm5_get_policy",
	  "kadm5_get_privs",
     };
     OM_uint32 minor;
     gss_buffer_desc client, server;
     gss_OID gss_type;
     char *a;

     (void) gss_display_name(&minor, client_name, &client, &gss_type);
     (void) gss_display_name(&minor, server_name, &server, &gss_type);
     a = inet_ntoa(rqst->rq_xprt->xp_raddr.sin_addr);

     krb5_klog_generic(context, LOG_NOTICE, "gssapi", 
		       "Forged or garbled request", a, 
		       "request", proc_names[msg->rm_call.cb_proc],
		       "client", client.value,
		       "server", server.value, NULL);

     (void) gss_release_buffer(&minor, &client);
     (void) gss_release_buffer(&minor, &server);
}

/*
 * Function: log_miscerr
 *
 * Purpose: Callback from GSS-API Sun RPC for miscellaneous errors
 *
 * Arguments:
 * 	rqst		(r) RPC service request
 * 	msg		(r) RPC message
 *	error		(r) error message from RPC
 * 	data		(r) arbitrary data (NULL), not used
 *
 * Effects:
 *
 * Logs the error via krb5_klog_syslog(); see functional spec for
 * format.
 */
void log_miscerr(struct svc_req *rqst, struct rpc_msg *msg,
		 char *error, char *data)
{
     char *a;
     
     a = inet_ntoa(rqst->rq_xprt->xp_raddr.sin_addr);
     krb5_klog_generic(context, LOG_NOTICE, "gssapi", 
		       "Miscellaneous RPC error", a, 
		       "msg", error, NULL);
}



/*
 * Function: log_badauth
 *
 * Purpose: Callback from GSS-API Sun RPC for authentication
 * failures/errors.
 *
 * Arguments:
 * 	major 		(r) GSS-API major status
 * 	minor		(r) GSS-API minor status
 * 	addr		(r) originating address
 * 	data		(r) arbitrary data (NULL), not used
 *
 * Effects:
 *
 * Logs the GSS-API error via krb5_klog_syslog(); see functional spec for
 * format.
 */
void log_badauth(OM_uint32 major, OM_uint32 minor,
		 struct sockaddr_in *addr, char *data)
{
     char *a;
     char *buffers[256];
     int nbuf = 0;
     
     /* Authentication attempt failed: <IP address>, <GSS-API error */
     /* strings> */

     a = inet_ntoa(addr->sin_addr);

     log_badauth_display_status(major, GSS_C_GSS_CODE, &nbuf, buffers);
     log_badauth_display_status(minor, GSS_C_MECH_CODE, &nbuf, buffers);
     krb5_klog_generic_array(context, LOG_NOTICE, "gssapi", 
			     "failed authentication", a, 
			     nbuf, buffers);
     log_badauth_free_buffers(nbuf, buffers);
}

void log_badauth_display_status(OM_uint32 code, int type, int *n, 
				char **outbuf)
{
    OM_uint32 gssstat, minor_stat;
    gss_buffer_desc msg;
    int msg_ctx;
    int level = 0;
    
    msg_ctx = 0;
    while (1) {
	/* allocate nine bytes for `major_nn', followed by null */
	outbuf[*n] = (char *)malloc(9);
	sprintf(outbuf[*n], "%s_%d", (type == GSS_C_GSS_CODE ?
				      "major" : "minor"), level);
	*n += 1;
	
	gssstat = gss_display_status(&minor_stat, code,
				     type, GSS_C_NULL_OID,
				     &msg_ctx, &msg);
      
	outbuf[*n] = (char *)malloc(msg.length);
	sprintf(outbuf[*n], msg.value);
	*n += 1;

	(void) gss_release_buffer(&minor_stat, &msg);

	if (gssstat != GSS_S_COMPLETE) {
	    /* An error occurred with gss_display_status.  We should
               handle this some nicer way */
	    break;
	}
	    
	if (!msg_ctx)
	    break;
    }
}

void log_badauth_free_buffers(int nbuf, char **buffers)
{
    int i;
    for(i = 0; i < nbuf; i++)
	free(buffers[i]);
}

void do_schpw(int s1, kadm5_config_params *params)
{
     krb5_error_code ret;
     /* XXX buffer = ethernet mtu */
     char req[1500];
     int len;
     struct sockaddr_in from;
     int fromlen;
     krb5_keytab kt;
     krb5_data reqdata, repdata;
     int s2;

     fromlen = sizeof(from);
     if ((len = recvfrom(s1, req, sizeof(req), 0, (struct sockaddr *)&from,
			 &fromlen)) < 0) {
	  krb5_klog_generic(context, LOG_ERR, "chpw", 
			   "Couldn't receive request", 
			    inet_ntoa(from.sin_addr),
			    "msg", error_message(errno), NULL);
	  return;
     }

     if (ret = krb5_kt_resolve(context, params->admin_keytab, &kt)) {
	  krb5_klog_generic(context, LOG_ERR, "chpw", 
			   "Couldn't open admin keytab", 
			    inet_ntoa(from.sin_addr),
			    "msg", error_message(ret), NULL);
	  return;
     }

     reqdata.length = len;
     reqdata.data = req;

    /* this is really obscure.  s1 is used for all communications.  it
       is left unconnected in case the server is multihomed and routes
       are asymmetric.  s2 is connected to resolve routes and get
       addresses.  this is the *only* way to get proper addresses for
       multihomed hosts if routing is asymmetric.  

       A related problem in the server, but not the client, is that
       many os's have no way to disconnect a connected udp socket, so
       the s2 socket needs to be closed and recreated for each
       request.  The s1 socket must not be closed, or else queued
       requests will be lost.

       A "naive" client implementation (one socket, no connect,
       hostname resolution to get the local ip addr) will work and
       interoperate if the client is single-homed. */

     if ((s2 = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
 	  krb5_klog_generic(context, LOG_ERR, "main", 
			    "cannot create connecting socket", NULL, 
			    "msg", error_message(errno), NULL);
	  fprintf(stderr, "Cannot create connecting socket: %s",
		  error_message(errno));
	  kadm5_destroy(global_server_handle);
	  krb5_klog_close();	  
	  exit(1);
     }

     if (connect(s2, (struct sockaddr *) &from, sizeof(from)) < 0) {
	  krb5_klog_generic(context, LOG_ERR, "chpw", 
			    "Couldn't connect to client", 
			    inet_ntoa(from.sin_addr),
			    "msg", error_message(errno), NULL);
	  return;
     }

     if (ret = process_chpw_request(context, global_server_handle,
				    params->realm, s2, kt, &from,
				    &reqdata, &repdata)) {
	  krb5_klog_generic(context, LOG_ERR, "chpw", 
			    "Error processing request", 
			    inet_ntoa(from.sin_addr),
			    "msg", error_message(ret), NULL);
     }

     close(s2);

     if (repdata.length == 0)
	  /* just qreturn.  This means something really bad happened */
	  return;

     len = sendto(s1, repdata.data, repdata.length, 0,
		  (struct sockaddr *) &from, sizeof(from));

     if (len < repdata.length) {
	  krb5_xfree(repdata.data);

	  krb5_klog_generic(context, LOG_ERR, "chpw", 
			   "Error sending reply", 
			    inet_ntoa(from.sin_addr),
			    "msg", error_message(errno), NULL);
	  return;
     }

     krb5_xfree(repdata.data);

     return;
}
