/*
 * Copyright, OpenVision Technologies, Inc., 1996, All Rights Reserved
 *
 * WARNING:  Retrieving the OpenVision Kerberos Administration system
 * source code, as described below, indicates your acceptance of the
 * following terms.  If you do not agree to the following terms, do not
 * retrieve the OpenVision Kerberos administration system.
 *
 * You may freely use and distribute the Source Code and Object Code
 * compiled from it, but this Source Code is provided to you "AS IS"
 * EXCLUSIVE OF ANY WARRANTY, INCLUDING, WITHOUT LIMITATION, ANY
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR
 * ANY OTHER WARRANTY, WHETHER EXPRESS OR IMPLIED.  IN NO EVENT WILL
 * OPENVISION HAVE ANY LIABILITY FOR ANY LOST PROFITS, LOSS OF DATA OR
 * COSTS OF PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, OR FOR ANY
 * SPECIAL, INDIRECT, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THIS
 * AGREEMENT, INCLUDING, WITHOUT LIMITATION, THOSE RESULTING FROM THE
 * USE OF THE SOURCE CODE, OR THE FAILURE OF THE SOURCE CODE TO PERFORM,
 * OR FOR ANY OTHER REASON.
 *
 * OpenVision retains all rights, title, and interest in the donated
 * Source Code.  With respect to OpenVision's copyrights in the donated
 * Source Code, OpenVision also retains rights to derivative works of
 * the Source Code whether created by OpenVision or a third party.
 *
 * OpenVision Technologies, Inc. has donated this Kerberos
 * Administration system to MIT for inclusion in the standard Kerberos 5
 * distribution. This donation underscores our commitment to continuing
 * Kerberos technology development and our gratitude for the valuable
 * work which has been performed by MIT and the Kerberos community.
 */

/*
 * $Header: /afs/athena.mit.edu/astaff/project/krbdev/.cvsroot/src/lib/kadm5/srv/server_dict.c,v 1.1 1996/07/24 22:23:18 tlyu Exp $
 */

#if !defined(lint) && !defined(__CODECENTER__)
static char *rcsid = "$Header: /afs/athena.mit.edu/astaff/project/krbdev/.cvsroot/src/lib/kadm5/srv/server_dict.c,v 1.1 1996/07/24 22:23:18 tlyu Exp $";
#endif

#include    <sys/types.h>
#include    <sys/file.h>
#include    <fcntl.h>
#include    <sys/stat.h>
#include    <unistd.h>
#include    <kadm5/admin.h>
#include    <stdlib.h>
#include    <stdio.h>
#include    <string.h>
#ifdef HAVE_MALLOC_H
#include    <malloc.h>
#endif /* HAVE_MALLOC_H */
#include    <memory.h>
#include    <syslog.h>
#include    "server_internal.h"

static char	    **word_list = NULL;	    /* list of word pointers */
static char	    *word_block = NULL;	    /* actual word data */
static int	    word_count = 0;	    /* number of words */
extern int	    errno;

/*
 * Function: word_compare
 * 
 * Purpose: compare two words in the dictionary.
 *
 * Arguments:
 *	w1		(input)	pointer to first word
 *	w2		(input) pointer to second word
 *	<return value>	result of strcmp
 *
 * Requires:
 *	w1 and w2 to point to valid memory
 * 
 */

static int
word_compare(const void *s1, const void *s2)
{
    return (strcasecmp(*(char **)s1, *(char **)s2));
}

/*
 * Function: init-dict
 * 
 * Purpose: Initialize in memory word dictionary
 *
 * Arguments:
 *	    none
 *	    <return value> KADM5_OK on sucsess errno on failure;
 * 			   (but success on ENOENT)
 *
 * Requires:
 *	If WORDFILE exists, it must contain a list of words,
 *	one word per-line.
 * 
 * Effects:
 *	If WORDFILE exists, it is read into memory sorted for future
 * use.  If it does not exist, it syslogs an error message and returns
 * success.
 *
 * Modifies:
 *	word_list to point to a chunck of allocated memory containing
 *	pointers to words
 *	word_block to contain the dictionary.
 * 
 */

int init_dict(kadm5_config_params *params)
{
    int		    fd,
		    len,
		    i;
    char	    *p,
		    *t;
    struct  stat    sb;
    
    if(word_list != NULL && word_block != NULL)
	return KADM5_OK;
    if (! (params->mask & KADM5_CONFIG_DICT_FILE)) {
	 syslog(LOG_INFO, "No dictionary file specified, continuing "
		"without one.");
	 return KADM5_OK;
    }
    if ((fd = open(params->dict_file, O_RDONLY, 0)) == -1) {
	 if (errno == ENOENT) {
	      syslog(LOG_ERR, "WARNING!  Cannot find dictionary file %s, "
		     "continuing without one.", params->dict_file);
	      return KADM5_OK;
	 } else
	      return errno;
    }
    if (fstat(fd, &sb) == -1) 
	return errno;
    if ((word_block = (char *) malloc(sb.st_size + 1)) == NULL)
	return errno;
    if (read(fd, word_block, sb.st_size) != sb.st_size)
	return errno;
    (void) close(fd);
    word_block[sb.st_size] = '\0';

    p = word_block;
    len = sb.st_size;
    while(len > 0 && (t = memchr(p, '\n', len)) != NULL) {
	*t = '\0';
	len -= t - p + 1;	
	p = t + 1;
	word_count++;
    }
    if ((word_list = (char **) malloc(word_count * sizeof(char *))) == NULL)
	return errno;
    p = word_block;
    for (i = 0; i < word_count; i++) {
	word_list[i] = p;
	p += strlen(p) + 1;
    }
    qsort(word_list, word_count, sizeof(char *), word_compare);
    return KADM5_OK;
}

/*
 * Function: find_word
 * 
 * Purpose: See if the specified word exists in the in-core dictionary
 *
 * Arguments:
 *	word		(input) word to search for.
 * 	<return value>	WORD_NOT_FOUND if not in dictionary,
 *			KADM5_OK if if found word
 *			errno if init needs to be called and returns an
 *			error
 *
 * Requires:
 *	word to be a null terminated string.
 *	That word_list and word_block besetup
 * 
 * Effects:
 *	finds word in dictionary.
 * Modifies:
 *	nothing.
 * 
 */

int
find_word(const char *word)
{
    char    **value;

    if(word_list == NULL || word_block == NULL) 
	    return WORD_NOT_FOUND;
    if ((value = (char **) bsearch(&word, word_list, word_count, sizeof(char *),
				   word_compare)) == NULL)
	return WORD_NOT_FOUND;
    else
	return KADM5_OK;
}

/*
 * Function: destroy_dict
 * 
 * Purpose: destroy in-core copy of dictionary.
 *
 * Arguments:
 *	    none
 *	    <return value>  none
 * Requires:
 *	    nothing
 * Effects:
 *	frees up memory occupied by word_list and word_block
 *	sets count back to 0, and resets the pointers to NULL
 *
 * Modifies:
 *	word_list, word_block, and word_count.
 * 
 */

void
destroy_dict(void)
{
    if(word_list) {
	free(word_list);
	word_list = NULL;
    }
    if(word_block) {
	free(word_block);
	word_block = NULL;
    }
    if(word_count)
	word_count = 0;
    return;
}
