/*
 * lib/kdb/kdb_oci.c
 */

/* 
 * This is a re-implementation of the libkdb.spec using the Oracle
 * Call Interface for the backend.
 */

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

/* Obtain dispatch table definitions from kdb.h */
#include "k5-int.h"
#include "kdb_dbc.h"
#include <stdio.h>
#include <errno.h>
#include <utime.h>
#include <syslog.h>

/* Include oracle */
#include <oracle.h>

#include "kdb_oci.h"

#define KEY_MAXSIZE 255
#define ENTRY_MAXSIZE 1024

/* Store data encrypted in a long raw */
#define KRB5_OCI_RAWDB
#define DEFAULT_KDB_DB "kerberos/ocidb"
#define DEFAULT_TABLE_NAME "principal"
#define AUX_TABLE_NAME "principal_bak"

static char default_db_name[] = DEFAULT_KDB_DB;

static krb5_error_code krb5_oci_db_start_update 
	PROTOTYPE((krb5_context));
static krb5_error_code krb5_oci_db_end_update 
	PROTOTYPE((krb5_context));

/*
 * Routines to deal with context.
 */
#define	k5oci_inited(c)	(c && c->db_context &&	\
			 ((krb5_db_context *) c->db_context)->db_inited)

/*
 * Restore the default context.
 */
static void
k5oci_clear_context(dbctx)
    krb5_db_context *	dbctx;
{
    /*
     * Free any dynamically allocated memory.
     */
    if (dbctx->db_name && (dbctx->db_name != default_db_name))
	free(dbctx->db_name);
    /*
     * Clear the structure and reset the defaults.
     */
    memset((char *) dbctx, 0, sizeof(krb5_db_context));
    dbctx->db_name = default_db_name;
}

static krb5_error_code
k5oci_init_context(context)
    krb5_context	context;
{
    krb5_db_context *	db_ctx;

    if (context->db_context == NULL) {
	if ((db_ctx = (krb5_db_context *) malloc(sizeof(krb5_db_context)))) {
	    memset((char *) db_ctx, 0, sizeof(krb5_db_context));
	    k5oci_clear_context((krb5_db_context *)db_ctx);
	    context->db_context = (void *) db_ctx;
	} else 
	    return(ENOMEM);
    }
    return(0);
}

/* Analyze a database name and populate the table_name field */
static char *
krb5_oci_table_name(krb5_db_context *dbctx)
{
    int l = strlen(dbctx->db_name)-1;
    if (dbctx->db_name[l] == '~') {
	/* specify as an auxiliary database */
	dbctx->db_name[l] = (char) 0;
	dbctx->table_name = AUX_TABLE_NAME;
    } else {
	dbctx->table_name = DEFAULT_TABLE_NAME;
    }
}

/* Routines and definitions for debugging OCI calls */
static void 
krb5_oci_err_report(Lda_Def *lda, Cda_Def *cursor)
{
  text msg[512];
  Cda_Def *c = (cursor ? cursor : (Cda_Def *)lda);
  sword n = (sword)oerhms(lda, c->rc, msg, (sword) sizeof msg);

#ifdef DEBUG
  printf("Oracle error (%s): %s",
	 (c->fc>0 ? oci_func_tab[c->fc] : "?"), msg);
#endif
  fprintf(stderr, "Oracle error (%s): %s",
	  (c->fc>0 ? oci_func_tab[c->fc] : "?"), msg);
  /*
  krb5_klog_generic(NULL, LOG_ERR, "kdb_oci", "Oracle error", NULL, 
		    "oci_call", c->fc>0 ? oci_func_tab[c->fc] : "?",
		    "msg", msg, NULL);
		    */
}

#define OCI_ERROR(l,c) krb5_oci_err_report(l,c)

/*
 * initialization for data base routines.  For the OCI backend, this
 * boils down to logging into the database.
 */

krb5_error_code
krb5_oci_db_init(context)
    krb5_context 	  context;
{
    char 		* filename = NULL;
    krb5_db_context	* db_ctx;
    krb5_error_code	  retval;

    if (k5oci_inited(context))
	return 0;

    /* Check for presence of our context, if not present, allocate one. */
    if ((retval = k5oci_init_context(context)))
	return(retval);

    db_ctx = context->db_context;
    
    /* Perform the login */
    if (olog(&db_ctx->lda, db_ctx->hda, db_ctx->db_name, NULL_TERM, 
	     (text *)0, NULL_TERM, (text *)0, NULL_TERM, OCI_LM_DEF)) {
	OCI_ERROR(&db_ctx->lda, NULL);
	return KRB5_KDB_UK_RERROR;
    }
    /* Start the cursor */
    if (oopen(&db_ctx->cda, &db_ctx->lda,(text *)0, NULL_TERM, NULL_TERM, 
	      (text *)0, NULL_TERM)) {
	OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
	return KRB5_KDB_UK_RERROR;
    }
    /* Enable autocommit */
    if (ocon(&db_ctx->lda)) {
	OCI_ERROR(&db_ctx->lda, NULL);
	return KRB5_KDB_UK_RERROR;
    }
    
    db_ctx->db_inited++;

    return 0;
}

/*
 * gracefully shut down database--must be called by ANY program that does
 * a krb5_oci_db_init
 */
krb5_error_code
krb5_oci_db_fini(context)
    krb5_context context;
{
    krb5_error_code retval = 0;
    krb5_db_context	*db_ctx;

    db_ctx = (krb5_db_context *) context->db_context;

    if (k5oci_inited(context)) {
	/* close the cursor. */
	if (oclose(&db_ctx->cda)) {
	    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
	    return KRB5_KDB_UK_SERROR;
	}
	/* logoff the database. */
	if (ologof(&db_ctx->lda))
	    OCI_ERROR(&db_ctx->lda, NULL);
    }
    if (db_ctx) {
	k5oci_clear_context(db_ctx);
	free (context->db_context);
	context->db_context = NULL;
    }
    return retval;
}

krb5_error_code
krb5_oci_db_open_database(context)
    krb5_context context;
{
    if (!k5oci_inited(context))
    	return KRB5_KDB_DBNOTINITED;
    return 0;
}

krb5_error_code
krb5_oci_db_close_database(context)
    krb5_context context;
{
    if (!k5oci_inited(context))
    	return KRB5_KDB_DBNOTINITED;
    return 0;
}

/*
 * Set/Get the master key associated with the database
 *
 * These only exist because the db_context is part of the kcontext
 * The should really reference the db_context
 */
krb5_error_code
krb5_oci_db_set_mkey(context, db_context, eblock)
    krb5_context 	  context;
    krb5_db_context 	* db_context;
    krb5_encrypt_block  * eblock;
{
    krb5_db_context *db_ctx;

    if (!k5oci_inited(context))
	return(KRB5_KDB_DBNOTINITED);

    db_ctx = context->db_context;
    db_ctx->db_master_key = eblock;
    return 0;
}

krb5_error_code
krb5_oci_db_get_mkey(context, db_context, eblock)
    krb5_context 	  context;
    krb5_db_context 	* db_context;
    krb5_encrypt_block  **eblock;
{
    krb5_db_context *db_ctx;

    if (!k5oci_inited(context))
	return(KRB5_KDB_DBNOTINITED);

    db_ctx = context->db_context;
    *eblock = db_ctx->db_master_key;
    return 0;

}

/*
 * Set the "name" of the current database to some alternate value.
 *
 * Passing a null pointer as "name" will set back to the default.
 * 
 * Kerberos uses a trailing '~' to set up a backup file.  Because
 * Oracle database names won't like trailing tildes (since the Oracle
 * names are usually login specs, and thus the password will be
 * wrong), this handles a trailing tilde as a special case, and
 * specifies it as a backup file.
 */

krb5_error_code
krb5_oci_db_set_name(context, name)
    krb5_context context;
    char *name;
{
    krb5_error_code kret;
    krb5_db_context *db_ctx;

    if (k5oci_inited(context))
	/* This used to return a KRB5_KDB_DBINITED, but this shouldn't
           be a problem with the Oracle database (which needs to
           initialize when creating) */
	return KRB5_KDB_DBINITED;

    if ((kret = k5oci_init_context(context)))
	return(kret);

    if (name == NULL)
	name = default_db_name;

    db_ctx = context->db_context;
    db_ctx->db_name = strdup(name);
    krb5_oci_table_name(db_ctx);

    /* Perform a test login */
    if (olog(&db_ctx->lda, db_ctx->hda, db_ctx->db_name, NULL_TERM, 
	     (text *)0, NULL_TERM, (text *)0, NULL_TERM, OCI_LM_DEF)) {
	OCI_ERROR(&db_ctx->lda, NULL);
	return KRB5_KDB_UK_RERROR;
    }
    ologof(&db_ctx->lda);
    return 0;
}

/*
 * Return the last modification time of the database.
 *
 * We may be able to get this from Oracle if we start auditing the
 * principal table.
 */

krb5_error_code
krb5_oci_db_get_age(context, db_name, age)
    krb5_context context;
    char *db_name;
    time_t *age;
{
    return 0;
}

/*
 * Remove the semaphore file; indicates that database is currently
 * under renovation.
 */

static krb5_error_code
krb5_oci_db_start_update(context)
    krb5_context context;
{
    return 0;
}

static krb5_error_code
krb5_oci_db_end_update(context)
    krb5_context context;
{
    return 0;
}

/* 
 * Locking currently not implemented for OCI backends.
 */
krb5_error_code
krb5_oci_db_lock(context, mode)
    krb5_context 	  context;
    int 	 	  mode;
{
    return 0;
}

krb5_error_code
krb5_oci_db_unlock(context)
    krb5_context context;
{
    return 0;
}

/*
 * Create the database, assuming it's not there.
 */
krb5_error_code
krb5_oci_db_create(context, db_name)
    krb5_context context;
    char *db_name;
{
    krb5_db_context *db_ctx;
    register krb5_error_code retval = 0;
    int close_on_exit = 0;
    text sql1[255];
    text sql2[255];

    if ((retval = k5oci_init_context(context)))
	return(retval);

    /* log in and create tables, then log out. */
    db_ctx = context->db_context;
    db_ctx->db_name = strdup(db_name);
    krb5_oci_table_name(db_ctx);

    sprintf(sql1, "DROP TABLE %s", db_ctx->table_name);
#ifdef KRB5_OCI_RAWDB
    sprintf(sql2, 
	    "CREATE TABLE %s ( princ VARCHAR2(255) PRIMARY KEY, data LONG RAW )", 
	    db_ctx->table_name);
#else
    /* Something else appropriate.*/
#endif

    if (!k5oci_inited(context))
	krb5_oci_db_init(context);
    
    if (oparse(&db_ctx->cda, sql1, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7) ||
	oexec(&db_ctx->cda)) {
	/* 942 means that the table being dropped did not exist. */
	if (db_ctx->cda.rc != 942)
	    goto oracle_error;
    }
    if (oparse(&db_ctx->cda, sql2, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7) ||
	oexec(&db_ctx->cda))
	goto oracle_error;

    krb5_oci_db_fini(context);
    return 0;

oracle_error:
    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
    return KRB5_KDB_UK_SERROR;
}

/*
 * Since the destroy operation happens outside the init/fini bracket, we
 * have some tomfoolery to undergo here.  If we're operating under no
 * database context, then we initialize with the default.
 */
krb5_error_code
krb5_oci_db_destroy(context, dbname)
     krb5_context context;
     char	*dbname;
{
    krb5_db_context *db_ctx;
    krb5_error_code retval;
    krb5_boolean tmpcontext;
    text sql1[100];

    tmpcontext = 0;
    if (!context->db_context) {
	tmpcontext = 1;
	if ((retval = k5oci_init_context(context)))
	    return(retval);
    }

    db_ctx = context->db_context;
    db_ctx->db_name = strdup(dbname);
    krb5_oci_table_name(db_ctx);
    sprintf(sql1,"DROP TABLE %s", db_ctx->table_name);

    /* We need to log into the database in order to destroy the table */
    if ((retval = krb5_oci_db_init(context)))
	return retval;

    if (oparse(&db_ctx->cda, sql1, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7) ||
	oexec(&db_ctx->cda)) {
	/* 942 == table did not exist */
	if (db_ctx->cda.rc != 942) {
	    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
	    return KRB5_KDB_UK_SERROR;
	}
    }

    if (tmpcontext) {
	k5oci_clear_context((krb5_db_context *) context->db_context);
	free(context->db_context);
	context->db_context = (void *) NULL;
    }
    return(0);
}

/*
 * "Atomically" rename the database in a way that locks out read
 * access in the middle of the rename.
 *
 * Since this is a relational DB backend, rather than a DBM-style DB,
 * we have to log in.  Right now, we only support going from and to an
 * auxiliary database of the same name (which means that we can
 * accomplish it through a rename command).
 */
krb5_error_code
krb5_oci_db_rename(context, from, to)
    krb5_context context;
    char *from;
    char *to;
{
    krb5_db_context *db_ctx, tmp_ctx;
    krb5_error_code retval;
    text sql1[100];
    text sql2[200];

    if ((retval = k5oci_init_context(context)))
	return(retval);

    db_ctx = context->db_context;
    db_ctx->db_name = strdup(from);
    krb5_oci_table_name(db_ctx);

    tmp_ctx.db_name = to;
    krb5_oci_table_name(&tmp_ctx);

    if(strcmp(tmp_ctx.db_name,db_ctx->db_name))
	/* currently unable to move between Oracle logins */
	return KRB5_KDB_UK_RERROR;

    sprintf(sql1,"DROP TABLE %s", tmp_ctx.table_name);
    sprintf(sql2,"RENAME %s TO %s", db_ctx->table_name, tmp_ctx.table_name);

    /* We need to log into the database in order to destroy the table */
    if ((retval = krb5_oci_db_init(context)))
	return retval;

    if (oparse(&db_ctx->cda, sql1, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7) ||
	oexec(&db_ctx->cda)) {
	/* 942 == table did not exist */
	if (db_ctx->cda.rc != 942) {
	    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
	    return KRB5_KDB_UK_SERROR;
	}
    }
    if (oparse(&db_ctx->cda, sql2, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7) ||
	oexec(&db_ctx->cda)) {
	OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
	return KRB5_KDB_UK_SERROR;
    }
    
    krb5_oci_db_fini(context);
    
    return 0;
}

/*
 * look up a principal in the data base.
 * returns number of entries found, and whether there were
 * more than requested. 
 */

krb5_error_code
krb5_oci_db_get_principal(context, searchfor, entries, nentries, more)
    krb5_context context;
krb5_principal searchfor;
krb5_db_entry *entries;		/* filled in */
int *nentries;				/* how much room/how many found */
krb5_boolean *more;			/* are there more? */
{
    krb5_error_code retval;
    char query[1024];
    datum key, contents;
    krb5_db_context *db_ctx;

    sb2 data_ind;
    ub2 data_rlen;
    ub2 rcode;
    sb4 offset = 0;

    contents.dptr = malloc(ENTRY_MAXSIZE);
    *more = FALSE;
    *nentries = 0;

    if (!k5oci_inited(context))
	return KRB5_KDB_DBNOTINITED;

    /* set up the principal name */
    if ((retval = krb5_encode_princ_dbmkey(context, &key, searchfor)))
	goto cleanup;
    
    db_ctx = context->db_context;
	
    /* set up the SQL query */
    sprintf(query,"SELECT data from principal WHERE princ = '%s'",key.dptr);
    if (oparse(&db_ctx->cda, query, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7))
	goto oracle_error;
    if (odefin(&db_ctx->cda, 1, (ub1 *) contents.dptr, 
	       (sword) ENTRY_MAXSIZE, (sword) SQLT_LBI,
	       (sword) -1, (sb2 *)&data_ind, (text *)0, 0, -1,
	       (ub2 *)&data_rlen, (ub2 *)&rcode)) 
	goto oracle_error;
    
    /* execute and position one row */
    if (oexfet(&db_ctx->cda, (ub4) 1, 0, 0))
	if (db_ctx->cda.rc == 1403)
	    /* no data found; just exit */
	    goto cleanup;
	else 
	    goto oracle_error;

    /* decode princ data */
    contents.dsize = data_rlen;
    if (!(retval = 
	  krb5_decode_princ_contents(context, &contents, entries)))
      *nentries = 1;

cleanup:
    return retval;

oracle_error:
    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
    return KRB5_KDB_UK_RERROR;
}

/*
  Free stuff returned by krb5_oci_db_get_principal.
 */
void
krb5_oci_db_free_principal(context, entries, nentries)
    krb5_context context;
    krb5_db_entry *entries;
    int nentries;
{
    register int i;
    for (i = 0; i < nentries; i++)
	krb5_dbe_free_contents(context, &entries[i]);
    return;
}

/*
  Stores the *"nentries" entry structures pointed to by "entries" in the
  database.

  *"nentries" is updated upon return to reflect the number of records
  acutally stored; the first *"nstored" records will have been stored in the
  database (even if an error occurs).
 */

krb5_error_code
krb5_oci_db_put_principal(context, entries, nentries)
    krb5_context context;
    krb5_db_entry *entries;
    register int *nentries;		/* number of entry structs to update */
{
    int i, n;
    datum   key, contents;
    krb5_error_code retval;
    char query[] = "INSERT INTO principal (princ,data) VALUES (:1, :2)";
    krb5_db_context *db_ctx;

    n = *nentries;
    *nentries = 0;

    if (!k5oci_inited(context))
	return KRB5_KDB_DBNOTINITED;

    db_ctx = context->db_context;
    
    /* for each one, stuff temps, and do replace/append */
    for (i = 0; i < n; i++) {
	if ((retval = krb5_encode_princ_contents(context, &contents,
						 entries)))
	    break;

	if ((retval = krb5_encode_princ_dbmkey(context, &key,
					       entries->princ))) {
	    krb5_free_princ_contents(context, &contents);
	    break;
	}

	/* when obndrn'ing the 1st (princ) column, ignore the null
           at the end. */
	if (oparse(&db_ctx->cda, query, NULL_TERM, DEFER_PARSE, 
		   (ub4) ORACLE_V7) ||
	    obndrn(&db_ctx->cda, 1, (ub1 *)key.dptr, key.dsize-1, SQLT_CHR,
		   -1, (sb2 *) 0, (text *)0, 0, -1) ||
	    obndrn(&db_ctx->cda, 2, (ub1 *)contents.dptr, contents.dsize, 
		   SQLT_LBI, -1, (sb2 *) 0, (text *)0, 0, -1))
	    goto oracle_error;
	if (oexec(&db_ctx->cda))
	    if (db_ctx->cda.rc != 1)
		goto oracle_error;
	    else {
		/* If the rc is 1, it indicates a constraint
                   violation.  This means that we need to update the
                   current value. */
		char q2[] = "UPDATE principal SET data = :2 WHERE princ = :1";

		if (oparse(&db_ctx->cda, q2, NULL_TERM, DEFER_PARSE, 
			   (ub4) ORACLE_V7) ||
		    obndrn(&db_ctx->cda, 1, (ub1 *)key.dptr, key.dsize-1, 
			   SQLT_CHR, -1, (sb2 *) 0, (text *)0, 0, -1) ||
		    obndrn(&db_ctx->cda, 2, (ub1 *)contents.dptr, 
			   contents.dsize, SQLT_LBI, -1, (sb2 *) 0, 
			   (text *)0, 0, -1))
		    goto oracle_error;
		if (oexec(&db_ctx->cda))
		    goto oracle_error;
	    }

	krb5_free_princ_contents(context, &contents);
	krb5_free_princ_dbmkey(context, &key);
	if (retval)
	    break;
	entries++;			/* bump to next struct */
    }

    *nentries = i;
    return(retval);

oracle_error:
    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
    return KRB5_KDB_UK_SERROR;
}

/*
 * delete a principal from the data base.
 * returns number of entries removed
 */

krb5_error_code
krb5_oci_db_delete_principal(context, searchfor, nentries)
    krb5_context 	  context;
    krb5_principal 	  searchfor;
    int 		* nentries;	/* how many found & deleted */
{
    krb5_error_code 	  retval;
    krb5_db_entry 	  entry;
    datum   		  key;
    int			  i;
    char query[] = "DELETE FROM principal WHERE princ = :1";
    krb5_db_context *db_ctx;

    if (!k5oci_inited(context))
	return KRB5_KDB_DBNOTINITED;

    if ((retval = krb5_encode_princ_dbmkey(context, &key, searchfor)))
	goto cleanup;

    db_ctx = context->db_context;

    if (oparse(&db_ctx->cda, query, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7) ||
	obndrn(&db_ctx->cda, 1, (ub1 *)key.dptr, key.dsize-1, SQLT_CHR,
	       -1, (sb2 *) 0, (text *)0, 0, -1) ||
	oexec(&db_ctx->cda))
	goto oracle_error;

    /* db_ctx->cda.rpc is the number of rows processed, i.e. deleted. */
    *nentries = db_ctx->cda.rpc;
    return 0;
cleanup:
    return retval;
oracle_error:
    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
    return KRB5_KDB_UK_SERROR;
}

krb5_error_code
krb5_oci_db_iterate (context, func, func_arg)
    krb5_context context;
    krb5_error_code (*func) PROTOTYPE((krb5_pointer, krb5_db_entry *));
    krb5_pointer func_arg;
{
    datum key, contents;
    krb5_db_entry entries;
    krb5_error_code retval;
    char query[] = "SELECT data FROM principal";
    sb2 data_ind;
    ub2 data_rlen;
    krb5_db_context *db_ctx = context->db_context;

    if (!k5oci_inited(context))
	return KRB5_KDB_DBNOTINITED;

    contents.dptr = malloc(ENTRY_MAXSIZE);
    
    if (oparse(&db_ctx->cda, query, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7))
	goto oracle_error;
    if (odefin(&db_ctx->cda, 1, (ub1 *) contents.dptr, 
	       (sword) ENTRY_MAXSIZE, (sword) SQLT_LBI,
	       (sword) -1, (sb2 *)&data_ind, (text *)0, -1, -1,
	       (ub2 *)&data_rlen, (ub2 *)0))
	goto oracle_error;
    
    /* execute and fetch rows */
    if(oexec(&db_ctx->cda))
	goto oracle_error;
	
    while(ofetch(&db_ctx->cda) == 0) {
	/* We break out of the loop when the fetch returns an error. */

	/* decode princ data */
	contents.dsize = data_rlen;
	if ((retval = krb5_decode_princ_contents(context, &contents, 
						 &entries)))
	    break;

	retval = (*func)(func_arg, &entries);
	krb5_dbe_free_contents(context, &entries);
	if (retval)
	    break;
    }

    return retval;

oracle_error:
    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
    return KRB5_KDB_UK_SERROR;
}

krb5_error_code
krb5_oci_db_bounded_iter (context, func, func_arg, start_princ, n)
    krb5_context context;
    krb5_error_code (*func) PROTOTYPE((krb5_pointer, krb5_db_entry *));
    krb5_pointer func_arg;
    char *start_princ;
    int n;
{
    datum key, contents;
    krb5_db_entry entries;
    krb5_error_code retval;
    char query[] = "SELECT data FROM principal ORDER by princ %s";
    char query2[] = "SELECT data FROM principal WHERE princ %c '%s' ORDER by princ %s";
    char buf[512];
    sb2 data_ind;
    ub2 data_rlen;
    int i = 0;
    krb5_db_context *db_ctx = context->db_context;

    if (!k5oci_inited(context))
	return KRB5_KDB_DBNOTINITED;

    if (start_princ) {
	sprintf(buf, query2, (n < 0 ? '<' : '>'), start_princ,
		(n > 0 ? "" : "DESC"));
    } else {
	sprintf(buf, query, (n > 0 ? "" : "DESC"));
    }

    contents.dptr = malloc(ENTRY_MAXSIZE);
    
    if (oparse(&db_ctx->cda, buf, NULL_TERM, DEFER_PARSE, (ub4) ORACLE_V7))
	goto oracle_error;
    if (odefin(&db_ctx->cda, 1, (ub1 *) contents.dptr, 
	       (sword) ENTRY_MAXSIZE, (sword) SQLT_LBI,
	       (sword) -1, (sb2 *)&data_ind, (text *)0, -1, -1,
	       (ub2 *)&data_rlen, (ub2 *)0))
	goto oracle_error;
    
    /* execute and fetch rows */
    if(oexec(&db_ctx->cda))
	goto oracle_error;
	
    if (n < 0) n = -n;
    while(ofetch(&db_ctx->cda) == 0) {
	/* We break out of the loop when the fetch returns an error,
           or when we reach the designated number of entries. */

	/* decode princ data */
	contents.dsize = data_rlen;
	if ((retval = krb5_decode_princ_contents(context, &contents, 
						 &entries)))
	    break;

	retval = (*func)(func_arg, &entries);
	krb5_dbe_free_contents(context, &entries);
	if (retval)
	    break;

	if (++i >= n)
	    break;
    }

    return retval;

oracle_error:
    OCI_ERROR(&db_ctx->lda, &db_ctx->cda);
    return KRB5_KDB_UK_SERROR;
}
    
krb5_boolean
krb5_oci_db_set_lockmode(context, mode)
    krb5_context context;
    krb5_boolean mode;
{
    return 0;
}

/*
 * Set dispatch table.  I'm not convinced that this function is being
 * used right now...
 */
krb5_error_code
kdb5_db_set_dbops(context, new)
    krb5_context	context;
    kdb5_dispatch_table	*new;
{
    return(KRB5_KDB_DBINITED);
}

#if 0
/*
 * Context serialization operations.
 */

/*
 * kdb5_context_size()	- Determine size required to serialize.
 */
static krb5_error_code
kdb5_context_size(kcontext, arg, sizep)
    krb5_context	kcontext;
    krb5_pointer	arg;
    size_t		*sizep;
{
    krb5_error_code	kret;
    size_t		required;
    krb5_db_context	*dbctx;

    /*
     * The database context requires at minimum:
     *	krb5_int32	for KV5M_DB_CONTEXT
     *	krb5_int32	for db_inited
     *	krb5_int32	for database lockfile non-blocking flag
     *	krb5_int32	for database lockfile lock count
     *	krb5_int32	for database lockfile lock mode
     *	krb5_int32	for length of database name.
     *	krb5_int32	for KV5M_DB_CONTEXT
     */
    kret = EINVAL;
    if ((dbctx = (krb5_db_context *) arg)) {
	required = (sizeof(krb5_int32) * 7);
	if (dbctx->db_inited && dbctx->db_dispatch && dbctx->db_name)
	    required += strlen(dbctx->db_name);
	kret = 0;
	*sizep += required;
    }
    return(kret);
}

/*
 * kdb5_context_externalize()	- Externalize the database context.
 */
static krb5_error_code
kdb5_context_externalize(kcontext, arg, buffer, lenremain)
    krb5_context	kcontext;
    krb5_pointer	arg;
    krb5_octet		**buffer;
    size_t		*lenremain;
{
    krb5_error_code	kret;
    krb5_db_context	*dbctx;
    size_t		required;
    krb5_octet		*bp;
    size_t		remain;

    required = 0;
    bp = *buffer;
    remain = *lenremain;
    kret = EINVAL;
    if ((dbctx = (krb5_db_context *) arg)) {
	kret = ENOMEM;
	if (!kdb5_context_size(kcontext, arg, &required) &&
	    (required <= remain)) {
	    /* Write magic number */
	    (void) krb5_ser_pack_int32(KV5M_DB_CONTEXT, &bp, &remain);

	    /* Write inited flag */
	    (void) krb5_ser_pack_int32((krb5_int32) dbctx->db_inited,
				       &bp, &remain);

	    /* Write blocking lock lockmode */
	    (void) krb5_ser_pack_int32((krb5_int32) dbctx->db_nb_locks,
				       &bp, &remain);

	    /* Write lock count */
	    (void) krb5_ser_pack_int32((krb5_int32)
				       (dbctx->db_inited) ?
				       dbctx->db_locks_held : 0,
				       &bp, &remain);

	    /* Write lock mode */
	    (void) krb5_ser_pack_int32((krb5_int32)
				       (dbctx->db_inited) ?
				       dbctx->db_lock_mode : 0,
				       &bp, &remain);

	    /* Write length of database name */
	    (void) krb5_ser_pack_int32((dbctx->db_inited && dbctx->db_name) ?
				       (krb5_int32) strlen(dbctx->db_name) : 0,
				       &bp, &remain);
	    if (dbctx->db_inited && dbctx->db_name)
		(void) krb5_ser_pack_bytes((krb5_octet *) dbctx->db_name,
					   strlen(dbctx->db_name),
					   &bp, &remain);

	    /* Write trailer */
	    (void) krb5_ser_pack_int32(KV5M_DB_CONTEXT, &bp, &remain);
	    kret = 0;
	    *buffer = bp;
	    *lenremain = remain;
	}
    }
    return(kret);
}

/*
 * kdb5_context_internalize()	- Internalize the database context.
 */
static krb5_error_code
kdb5_context_internalize(kcontext, argp, buffer, lenremain)
    krb5_context	kcontext;
    krb5_pointer	*argp;
    krb5_octet		**buffer;
    size_t		*lenremain;
{
    krb5_error_code	kret;
    krb5_context	tmpctx;
    krb5_db_context	*dbctx;
    krb5_int32		ibuf;
    krb5_octet		*bp;
    size_t		remain;
    krb5_int32		iflag;
    krb5_int32		nb_lockmode;
    krb5_int32		lockcount;
    krb5_int32		lockmode;
    krb5_int32		dbnamelen;
    char		*dbname;

    bp = *buffer;
    remain = *lenremain;
    kret = EINVAL;
    dbctx = (krb5_db_context *) NULL;
    /* Read our magic number */
    if (krb5_ser_unpack_int32(&ibuf, &bp, &remain))
	ibuf = 0;
    if (ibuf == KV5M_DB_CONTEXT) {
	kret = ENOMEM;

	if (!(kret = krb5_ser_unpack_int32(&iflag, &bp, &remain)) &&
	    !(kret = krb5_ser_unpack_int32(&nb_lockmode, &bp, &remain)) &&
	    !(kret = krb5_ser_unpack_int32(&lockcount, &bp, &remain)) &&
	    !(kret = krb5_ser_unpack_int32(&lockmode, &bp, &remain)) &&
	    !(kret = krb5_ser_unpack_int32(&dbnamelen, &bp, &remain)) &&
	    !(kret = krb5_init_context(&tmpctx))) {
	    if (iflag) {
		dbname = (char *) NULL;
		if (dbnamelen &&
		    (dbname = (char *) malloc((size_t) (dbnamelen+1)))) {
		    kret = krb5_ser_unpack_bytes((krb5_octet *) dbname,
						 (size_t) dbnamelen,
						 &bp, &remain);
		    if (!kret)
			dbname[dbnamelen] = '\0';
		}
		if (!kret &&
		    (!dbname || !(kret = krb5_db_set_name(tmpctx, dbname))) &&
		    !(kret = krb5_db_init(tmpctx))) {
		    dbctx = (krb5_db_context *) tmpctx->db_context;
		    (void) krb5_oci_db_set_lockmode(tmpctx, 0);
		    if (lockmode)
			kret = krb5_db_lock(tmpctx, lockmode);
		    if (!kret && lockmode)
			dbctx->db_locks_held = lockcount;
		    (void) krb5_oci_db_set_lockmode(tmpctx, nb_lockmode);
		}
		if (dbname)
		    krb5_xfree(dbname);
	    }
	    if (!kret)
		kret = krb5_ser_unpack_int32(&ibuf, &bp, &remain);
	    if (kret || (ibuf != KV5M_DB_CONTEXT))
		kret = EINVAL;

	    if (kret) {
		if (dbctx)
		    krb5_db_fini(tmpctx);
	    }
	    else
		tmpctx->db_context = (void *) NULL;
	    krb5_free_context(tmpctx);
	}
    }
    if (!kret) {
	*buffer = bp;
	*lenremain = remain;
	*argp = (krb5_pointer) dbctx;
    }
    return(kret);
}

/* Dispatch entry */
static const krb5_ser_entry kdb5_context_ser_entry = {
    KV5M_DB_CONTEXT,			/* Type			*/
    kdb5_context_size,			/* Sizer routine	*/
    kdb5_context_externalize,		/* Externalize routine	*/
    kdb5_context_internalize		/* Externalize routine	*/
};
#endif /* 0 */

/*
 * Register serializer.
 */
krb5_error_code
krb5_ser_db_context_init(kcontext)
    krb5_context	kcontext;
{
    /* return(krb5_register_serializer(kcontext, &kdb5_context_ser_entry)); */
    return 0;
}
