/*********************************************************************
**
**     File name:               ssh_crypt.c
**                                  
**                              Copyright 1997 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 Handle Encryption and Decryption
**				of SSH Packets
**
**     Author/Date:             Tadayoshi Kohno, 25 November 1997
**
**     References:                  
**	I am *REALLY* indebted to Eric Young's SSLeay libraries.
**	All the cryptographic related functionality of my program
**	interfaces with his libraries (SSLeay-0.8.1).
**
**		ftp://ftp.psy.uq.oz.au/pub/Crypto/SSL
**
**     Todo:
**	Our encryption keys, ... are currently static to this file.
**	This was based upon the idea that I would only have one instance
**	of an ssh connection at a time.  My WinSiCP (Windows Secure
**	interactive Copy) will open two connections at once.  For
**	this reason, I need to gut a majorr part of this.
**
**     Notes:
**	This file handles encryption and decryption of the SSH's packets
**	based upon the CipherType chosen (which is initializaed through
**	a ssh_set_cipher() call).
**
**	The routines in this file work the following way:
**		ssh_set_cipher	handles initialization of the cipher
**	Subsequently, ssh_cipher_{encode,decode} encodes and decodes
**		a packet based upon the chosen CipherType.
**
**	As with all parts of this program, things can be improved.  For
**	example, I do a lot of my_bcopy's.  I don't think this is necessary,
**	and will think about improving this as soon as possible.
**
**	This file also contains static struct variables like IdeaValues.
**	These structs store information about the current encryption
**	process.  For example, IdeaValues store the initialization vectors
**	for the encryption and decryption process as well as the
**	encryption key.  The reason they're static to the file (as opposed
**	to being passed around) is that they're used locally and their
**	uses are pretty obvious.
**
**	Another possible future direction is moving each encryption to
**	another file (like ssh_idea.c).  I, personally, don't think that
**	would be a good idea because, although it might make it a little
**	clearer, is kind of overkill
**
**     Functions:
**	ssh_set_cipher		initialize the cipher
**
**	ssh_cipher_decode	decode a packet
**	ssh_cipher_encode	encode a packet
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_crypt.c,v 3.5 1998/05/21 17:35:48 kohno Exp $";
#endif

#include <assert.h>

#include "ssh.h"
#include "ssh_crypt.h"

#include "ssh_types.h"

#include "ssh_util.h"

/*********************************************************************
**
**     Function:                ssh_set_cipher
**
**     Purpose:                 set CipherType and initialize cipher
**
**     Entry (pre) conditions:  mode is a valid CipherType
**
**     Parameters:              ssh_info	connection-specific information
**				mode		chosen cipher type
**				session_key	session key for cipher init
**
**     Return value:            S_GOOD		success
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set to SSH_ERRNO_UNKNOWN_CIPHER
**				if the cipher type is unknown or not
**				implemented
**
**     Side effects:            CipherType set to desired mode
**				Struct for cipher initialized
**					(IdeaValues,...)
**
**     Author/Date:             Tadayoshi Kohno, 25 November 1997
**     Modified:                Tadayoshi Kohno, 4 December 1997
**					Added DES, 3DES
**     Modified:		Tadayoshi Kohno, 21 May 1998
**					Added Blowfish
**
**     Notes:
**	This *must* be called before any encryption/decryption occurs for
**	each session (because of initialization vectors).
**
*********************************************************************/

int ssh_set_cipher
(
	struct ssh_struct * ssh_info,	/* connection-specific information */
	uint8_t mode,			/* chosen cipher method */
	uint8_t * session_key		/* session key */
)
{
	int ret_val;		/* return value */

	ssh_info->cipher_type = mode;
	switch(ssh_info->cipher_type)
	{
	case SSH_CIPHER_NONE:
		/* do nothing special if we don't want encryption */
		ssh_debugger_new(&(ssh_info->debug_info),
			"using none", "ssh_cipher_set");
		break;

	case SSH_CIPHER_IDEA:
		/* setup keys/initialization vectors for IDEA encryption */
		idea_set_encrypt_key(session_key,
			&(ssh_info->cipher_data.idea_values.key));
		my_bzero(ssh_info->cipher_data.idea_values.encode_iv,
			IDEA_IV_LEN);
		my_bzero(ssh_info->cipher_data.idea_values.decode_iv,
			IDEA_IV_LEN);
		ssh_info->cipher_data.idea_values.en_num = 0;
		ssh_info->cipher_data.idea_values.de_num = 0;
		ssh_debugger_new(&(ssh_info->debug_info),
			"using idea", "ssh_cipher_set");
		break;

	case SSH_CIPHER_DES:
		/* setup keys/initialization vectors for DES encryption */
		if ((ret_val = des_set_key((C_Block *) session_key,
			ssh_info->cipher_data.des_values.key)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}

		my_bzero(ssh_info->cipher_data.des_values.iv_encrypt,
			DES_IV_LEN);
		my_bzero(ssh_info->cipher_data.des_values.iv_decrypt,
			DES_IV_LEN);
		ssh_debugger_new(&(ssh_info->debug_info),
			"using des", "ssh_cipher_set");

		break;

	case SSH_CIPHER_3DES:
		/* setup keys/initialization vectors for 3DES encryption */
		if ((ret_val = des_set_key((C_Block *) session_key,
			ssh_info->cipher_data.des3_values.key_1)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}
		if ((ret_val = des_set_key((C_Block *) (session_key
			+ DES_KEY_LEN),
			ssh_info->cipher_data.des3_values.key_2)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}
		if ((ret_val = des_set_key((C_Block *) (session_key
			+ 2 * DES_KEY_LEN),
			ssh_info->cipher_data.des3_values.key_3)) != 0)
		{
			ssh_errno_set(SSH_ERRNO_INIT_CIPHER);
			return(S_BAD);
		}

		my_bzero(ssh_info->cipher_data.des3_values.iv_encrypt_1,
			DES_IV_LEN);
		my_bzero(ssh_info->cipher_data.des3_values.iv_encrypt_2,
			DES_IV_LEN);
		my_bzero(ssh_info->cipher_data.des3_values.iv_encrypt_3,
			DES_IV_LEN);
		my_bzero(ssh_info->cipher_data.des3_values.iv_decrypt_1,
			DES_IV_LEN);
		my_bzero(ssh_info->cipher_data.des3_values.iv_decrypt_2,
			DES_IV_LEN);
		my_bzero(ssh_info->cipher_data.des3_values.iv_decrypt_3,
			DES_IV_LEN);
		ssh_debugger_new(&(ssh_info->debug_info),
			"using 3des", "ssh_cipher_set");

		break;

	case SSH_CIPHER_TSS:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;

	case SSH_CIPHER_RC4:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;

	case SSH_CIPHER_BLOWFISH:
		BF_set_key(&(ssh_info->cipher_data.blowfish_values.key),
			SSH_SESSION_KEY_SIZE, session_key);

		my_bzero(ssh_info->cipher_data.blowfish_values.encode_iv,
			BF_IV_LEN);
		my_bzero(ssh_info->cipher_data.blowfish_values.decode_iv,
			BF_IV_LEN);

		ssh_debugger_new(&(ssh_info->debug_info),
			"using blowfish", "ssh_cipher_set");
		
		break;

	default:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;
	}
	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                ssh_cipher_decode
**
**     Purpose:                 decrypt a packet
**
**     Entry (pre) conditions:  packet encrypted
**				memory allocated and enough for
**				packet and out_packet
**
**     Parameters:              ssh_info	connection-specific info
**				packet		packet to decode
**				out_packet	decoded packet
**
**     Return value:            S_GOOD		success
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set to SSH_ERRNO_UNKNOWN_CIPHER
**				if the cipher type is unknown or not
**				implemented
**
**     Side effects:            out_packet is decoded version of
**				packet.
**
**				encryption struct (IdeaValues, ...)
**				sometimes updated (initialization vectors,
**				other parameters).
**
**     Author/Date:             Tadayoshi Kohno, 25 November 1997
**     Modified:                Tadayoshi Kohno, 4 December 1997
**					Added DES, 3DES
**     Modified:		Tadayoshi Kohno, 21 May 1998
**					Added Blowfish
**
**     Notes:
**	This function should be used to decrypt an encrypted packet.
**	Use only after calling ssh_set_cipher().
**
**	Note that for all encryption, the entire ssh-packet (except for
**	length field) is encrypted.
**
**	About the weird shifting of bits for Blowfish encryption,
**	it looks like the freeware SSH (SSH Communications, Inc.) version
**	and SSLeay do blowfish in a slightly different way.  A quick fix
**	that adds a bit of overhead to blowfish is to reorder every four
**	bytes.  Namely, tin before an encrypt in SSLeay's bf_cbc.c function
**	(SSLeay-0.8.1) may be aabbccdd whereas the freeware ssh client would
**	have it as ddccbbaa...
**
*********************************************************************/

int ssh_cipher_decode
(
	struct ssh_struct * ssh_info,	/* connection-specific information */
	uint8_t * packet,		/* packet to decode */
	uint8_t * out_packet		/* decoded packet */
)
{
	static char tmp_packet1[SSH_MAX_PACKET];	/* temporary packet */
	static char tmp_packet2[SSH_MAX_PACKET];	/* temporary packet */

	uint32_t length;	/* length of packet (minus len field) */
	uint32_t padding;	/* padding in packet */

	uint32_t packet_start;	/* start of packet (after len field) */

	uint32_t index;		/* index into packet (for blowfish) */
	uint8_t tmp_byte;	/* a temporary byte (for blowfish) */

	/*
	**	First lets compute the length of the packet to decode
	**	This is based upon the length of the data + padding
	*/
	my_bcopy((void *) packet, (void *) out_packet, sizeof(length));
	my_bcopy((void *) packet, (void *) &(length), sizeof(length));
	length = ntohl(length);		/* length of packet minus padding */

	padding = 8 - (length % 8);
	length = length + padding;

	assert(length % 8 == 0);

	packet_start = sizeof(length);

	/*
	**	Decode the packet based upon the CipherType
	*/
	switch (ssh_info->cipher_type)
	{
	case SSH_CIPHER_NONE:
		/* no encryption, so just copy it over */
		my_bcopy((void *) packet, (void *) out_packet,
			length + packet_start);
		break;

	case SSH_CIPHER_IDEA:
		/* run IDEA encryption on it in cfb mode */
		idea_cfb64_encrypt(packet + packet_start,
			out_packet + packet_start, length,
			&(ssh_info->cipher_data.idea_values.key),
			ssh_info->cipher_data.idea_values.decode_iv,
			&(ssh_info->cipher_data.idea_values.de_num),
			IDEA_DECRYPT);
		break;

	case SSH_CIPHER_DES:
		/* use DES decryption in CBC mode */
		des_ncbc_encrypt((C_Block *)(packet + packet_start),
			(C_Block *)(out_packet + packet_start), (long) length,
			ssh_info->cipher_data.des_values.key,
			(C_Block *)ssh_info->cipher_data.des_values.iv_decrypt,
			DES_DECRYPT);
		break;

	case SSH_CIPHER_3DES:
		/* use 3 independed DES encryptions/decryptions in CBC mode */
		des_ncbc_encrypt((C_Block *)(packet + packet_start),
			(C_Block *)(tmp_packet1 + packet_start), (long) length,
			ssh_info->cipher_data.des3_values.key_3,
			(C_Block *)(ssh_info->cipher_data.des3_values.iv_decrypt_1),
			DES_DECRYPT);
		des_ncbc_encrypt((C_Block *)(tmp_packet1 + packet_start),
			(C_Block *)(tmp_packet2 + packet_start), (long) length,
			ssh_info->cipher_data.des3_values.key_2,
			(C_Block *)(ssh_info->cipher_data.des3_values.iv_decrypt_2),
			DES_ENCRYPT);
		des_ncbc_encrypt((C_Block *)(tmp_packet2 + packet_start),
			(C_Block *)(out_packet + packet_start), (long) length,
			ssh_info->cipher_data.des3_values.key_1,
			(C_Block *)(ssh_info->cipher_data.des3_values.iv_decrypt_3),
			DES_DECRYPT);

		break;

	case SSH_CIPHER_TSS:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;

	case SSH_CIPHER_RC4:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;

	case SSH_CIPHER_BLOWFISH:
		/* see notes about weird shifting of bytes for blowfish */
		for (index = 0; index < length; index += BF_REORDER_LEN)
		{
			tmp_byte = packet[packet_start + index];
			packet[packet_start + index]
				= packet[packet_start + index + 3];
			packet[packet_start + index + 3] = tmp_byte;
			tmp_byte = packet[packet_start + index + 1];
			packet[packet_start + index + 1]
				=packet[packet_start + index +2];
			packet[packet_start + index + 2] = tmp_byte;
		}

		BF_cbc_encrypt(packet + packet_start,
			out_packet + packet_start, (long) length, 
			&(ssh_info->cipher_data.blowfish_values.key),
			ssh_info->cipher_data.blowfish_values.decode_iv,
			BF_DECRYPT);

		/* see notes about weird shifting of bytes for blowfish */
		for (index = 0; index < length; index += BF_REORDER_LEN)
		{
			tmp_byte = out_packet[packet_start + index];
			out_packet[packet_start + index]
				= out_packet[packet_start + index + 3];
			out_packet[packet_start + index + 3] = tmp_byte;
			tmp_byte = out_packet[packet_start + index + 1];
			out_packet[packet_start + index + 1]
				=out_packet[packet_start + index +2];
			out_packet[packet_start + index + 2] = tmp_byte;
		}
		break;

	default:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;
	}
	return(S_GOOD);
}


/*********************************************************************
**
**     Function:                ssh_cipher_encode
**
**     Purpose:                 encrypt packet
**
**     Entry (pre) conditions:  packet and out_packet memory exist
**
**     Parameters:              ssh_info	connection-specific information
**				packet		packet to encrypt
**				out_packet	encrypted packet
**
**     Return value:            S_GOOD
**
**     Error codes:             S_BAD		error
**
**				ssh_errno set to SSH_ERRNO_UNKNOWN_CIPHER
**				if the cipher type is unknown or not
**				implemented
**
**     Side effects:            out_packet contains encrypted packet
**
**				structs for encryption (IdeaValues, ...)
**				updated (initialization vectors, ...)
**
**     Author/Date:             Tadayoshi Kohno, 25 November 1997
**     Modified:                Tadayoshi Kohno, 4 December 1997
**					Added DES, 3DES
**     Modified:		Tadayoshi Kohno, 21 May 1998
**					Added Blowfish
**
**     Notes:
**	Another potential way to make this function faster would be to
**	pass the length to this function (so I don't have to recompute it)
**	This is possible because this function is (almost) always called
**	after a pack().
**
**	For information about the weird shifting of bytes for blowfish,
**	see the notes for ssh_cipher_decode().
**
*********************************************************************/

int ssh_cipher_encode
(
	struct ssh_struct * ssh_info,	/* connection-specific information */
	uint8_t * packet,		/* packet to encrypt */
	uint8_t * out_packet		/* encrypted packet */
)
{
	static char tmp_packet1[SSH_MAX_PACKET];	/* temporary packet */
	static char tmp_packet2[SSH_MAX_PACKET];	/* temporary packet */

	uint32_t length;	/* length of packet (minus len field) */
	uint32_t padding;	/* padding in packet */

	uint32_t packet_start;	/* start of packet (after length field) */

	uint32_t index;		/* index into packet (for blowfish) */
	uint8_t tmp_byte;	/* a temporary byte (for blowfish) */

	/*
	**	First lets compute the length of the packet to decode
	**	This is based upon the length of the data + padding
	*/
	my_bcopy((void *) packet, (void *) out_packet, sizeof(length));
	my_bcopy((void *) packet, (void *) &(length), sizeof(length));
	length = ntohl(length);		/* length of packet minus padding */

	padding = 8 - (length % 8);
	length = length + padding;

	assert(length % 8 == 0);

	packet_start = sizeof(length);


	/*
	**	Now encrypt packet
	*/
	switch (ssh_info->cipher_type)
	{
	case SSH_CIPHER_NONE:
		/* no encryption, so just copy it over */
		my_bcopy((void *) packet, (void *) out_packet,
			length + packet_start);
		break;

	case SSH_CIPHER_IDEA:
		/* encrypt using IDEA encryption in cfb mode */
		idea_cfb64_encrypt(packet + packet_start,
			out_packet + packet_start, length,
			&(ssh_info->cipher_data.idea_values.key),
			ssh_info->cipher_data.idea_values.encode_iv,
			&(ssh_info->cipher_data.idea_values.en_num),
			IDEA_ENCRYPT);
		break;

	case SSH_CIPHER_DES:
		/* encrypt using DES encryption in CBC mode */
		des_ncbc_encrypt((C_Block *)(packet + packet_start),
			(C_Block *)(out_packet + packet_start), (long) length,
			ssh_info->cipher_data.des_values.key,
			(C_Block *)ssh_info->cipher_data.des_values.iv_encrypt,
			DES_ENCRYPT);
		break;

	case SSH_CIPHER_3DES:
		/* encrypt using 3DES encryption (all in CBC mode) */
		des_ncbc_encrypt((C_Block *)(packet + packet_start),
			(C_Block *)(tmp_packet1 + packet_start), (long) length,
			ssh_info->cipher_data.des3_values.key_1,
			(C_Block *)ssh_info->cipher_data.des3_values.iv_encrypt_1,
			DES_ENCRYPT);		
		des_ncbc_encrypt((C_Block *)(tmp_packet1 + packet_start),
			(C_Block *)(tmp_packet2 + packet_start), (long) length,
			ssh_info->cipher_data.des3_values.key_2,
			(C_Block *)ssh_info->cipher_data.des3_values.iv_encrypt_2,
			DES_DECRYPT);		
		des_ncbc_encrypt((C_Block *)(tmp_packet2 + packet_start),
			(C_Block *)(out_packet + packet_start), (long) length,
			ssh_info->cipher_data.des3_values.key_3,
			(C_Block *)ssh_info->cipher_data.des3_values.iv_encrypt_3,
			DES_ENCRYPT);
		break;

	case SSH_CIPHER_TSS:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;

	case SSH_CIPHER_RC4:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;

	case SSH_CIPHER_BLOWFISH:
		/* see notes about weird shifting of bytes for blowfish */
		for (index = 0; index < length; index += BF_REORDER_LEN)
		{
			tmp_byte = packet[packet_start + index];
			packet[packet_start + index]
				= packet[packet_start + index + 3];
			packet[packet_start + index + 3] = tmp_byte;
			tmp_byte = packet[packet_start + index + 1];
			packet[packet_start + index + 1]
				=packet[packet_start + index +2];
			packet[packet_start + index + 2] = tmp_byte;
		}

		BF_cbc_encrypt(packet + packet_start,
			out_packet + packet_start, length, 
			&(ssh_info->cipher_data.blowfish_values.key),
			ssh_info->cipher_data.blowfish_values.encode_iv,
			BF_ENCRYPT);

		/* see notes about weird shifting of bytes for blowfish */
		for (index = 0; index < length; index += BF_REORDER_LEN)
		{
			tmp_byte = out_packet[packet_start + index];
			out_packet[packet_start + index]
				= out_packet[packet_start + index + 3];
			out_packet[packet_start + index + 3] = tmp_byte;
			tmp_byte = out_packet[packet_start + index + 1];
			out_packet[packet_start + index + 1]
				=out_packet[packet_start + index +2];
			out_packet[packet_start + index + 2] = tmp_byte;
		}
		break;

	default:
		ssh_errno_set(SSH_ERRNO_UNKNOWN_CIPHER);

		return(S_BAD);
		break;
	}

	return(S_GOOD);
}

