// bytes.java

/**
 * Functions for transforming to and from byte arrays.
 * Includes transformations from and to arrays with elements of other sizes, and
 * strings.
 * All counts are in bytes, offsets are in units of the array.
 * Arrays must be pre-allocated with sufficient size before the call.
 * Return strings are allocated here.
 *
 * @author Hal Finney
 * @version December 27, 1995
 */
class bytes {


/* Scalars to byte arrays */

public static int BArrToIntLittle(byte bArr[], int off)
{
    int ret;
    ret = ((bArr[off+0]&0xff)<<0)  | ((bArr[off+1]&0xff)<<8) |
      ((bArr[off+2]&0xff)<<16) | ((bArr[off+3]&0xff)<<24);
    return ret;
}

public static int BArrToIntBig(byte bArr[], int off)
{
    int ret;
    ret = ((bArr[off+0]&0xff)<<24) | ((bArr[off+1]&0xff)<<16) |
      ((bArr[off+2]&0xff)<<8)  | ((bArr[off+3]&0xff)<<0);
    return ret;
}

public static short BArrToShortLittle(byte bArr[], int off)
{
    short ret;
    ret = (short)(((bArr[off+0]&0xff)<<0)  | ((bArr[off+1]&0xff)<<8));
    return ret;
}

public static short BArrToShortBig(byte bArr[], int off)
{
    short ret;
    ret = (short)(((bArr[off+0]&0xff)<<8)  | ((bArr[off+1]&0xff)<<0));
    return ret;
}


/* Byte arrays to scalars */

public static void IntToBArrLittle(int in, byte bArr[], int off)
{
    bArr[off+0] = (byte)(in>>0);     bArr[off+1] = (byte)(in>>8);
    bArr[off+2] = (byte)(in>>16);    bArr[off+3] = (byte)(in>>24);
}

public static void IntToBArrBig(int in, byte bArr[], int off)
{
    bArr[off+0] = (byte)(in>>24);    bArr[off+1] = (byte)(in>>16);
    bArr[off+2] = (byte)(in>>8);     bArr[off+3] = (byte)(in>>0);
}

public static void ShortToBArrLittle(short in, byte bArr[], int off)
{
    bArr[off+0] = (byte)(in>>0);     bArr[off+1] = (byte)(in>>8);
}

public static void ShortToBArrBig(short in, byte bArr[], int off)
{
    bArr[off+0] = (byte)(in>>8);     bArr[off+1] = (byte)(in>>0);
}


/* Arrays to byte arrays */

public static void ArrToBArrLittle (int iArr[], int inOff, byte bArr[], int outOff, int len)
{
    int i;
    int j = outOff;
    len >>= 2;
    for (i=0; i<len; ++i) {
        int in = iArr[inOff+i];
        bArr[j++] = (byte)(in>>0);    bArr[j++] = (byte)(in>>8);
        bArr[j++] = (byte)(in>>16);   bArr[j++] = (byte)(in>>24);
    }
}

public static void ArrToBArrBig (int iArr[], int inOff, byte bArr[], int outOff, int len)
{
    int i;
    int j = outOff;
    len >>= 2;
    for (i=0; i<len; ++i) {
        int in = iArr[inOff+i];
        bArr[j++] = (byte)(in>>24);   bArr[j++] = (byte)(in>>16);
        bArr[j++] = (byte)(in>>8);    bArr[j++] = (byte)(in>>0);
    }
}

public static void ArrToBArrLittle (short sArr[], int inOff, byte bArr[], int outOff, int len)
{
    int i;
    int j = outOff;
    len >>= 1;
    for (i=0; i<len; ++i) {
        short in = sArr[inOff+i];
        bArr[j++] = (byte)(in>>0);    bArr[j++] = (byte)(in>>8);
    }
}

public static void ArrToBArrBig (short sArr[], int inOff, byte bArr[], int outOff, int len)
{
    int i;
    int j = outOff;
    len >>= 1;
    for (i=0; i<len; ++i) {
        short in = sArr[inOff+i];
        bArr[j++] = (byte)(in>>8);    bArr[j++] = (byte)(in>>0);
    }
}


/* Byte arrays to arrays */

public static void BArrToArrLittle (byte bArr[], int inOff, int iArr[], int outOff, int len)
{
    int i;
    int j = outOff;
    for (i=inOff; i<inOff+len; i+=4)
        iArr[j++] = ((bArr[i+0]&0xff)<<0)  | ((bArr[i+1]&0xff)<<8) |
                    ((bArr[i+2]&0xff)<<16) | ((bArr[i+3]&0xff)<<24);
}


public static void BArrToArrBig (byte bArr[], int inOff, int iArr[], int outOff, int len)
{
    int i;
    int j = outOff;
    for (i=inOff; i<inOff+len; i+=4)
        iArr[j++] = ((bArr[i+0]&0xff)<<24) | ((bArr[i+1]&0xff)<<16) |
                    ((bArr[i+2]&0xff)<<8)  | ((bArr[i+3]&0xff)<<0);
}

public static void BArrToArrLittle (byte bArr[], int inOff, short sArr[], int outOff, int len)
{
    int i;
    int j = outOff;
    for (i=inOff; i<inOff+len; i+=2)
        sArr[j++] = (short)(((bArr[i+0]&0xff)<<0)  | ((bArr[i+1]&0xff)<<8));
}

public static void BArrToArrBig (byte bArr[], int inOff, short sArr[], int outOff, int len)
{
    int i;
    int j = outOff;
    for (i=inOff; i<inOff+len; i+=2)
        sArr[j++] = (short)(((bArr[i+0]&0xff)<<8)  | ((bArr[i+1]&0xff)<<0));
}


/* To and from hex strings */

public static String BArrToStringBig (byte bArr[], int off, int len)
{
    StringBuffer sb = new StringBuffer(bArr.length*2);
    for (int i=0; i<len; ++i) {
        byte b = bArr[i+off];
        sb.append("0123456789abcdef".charAt((b>>4)&0xf));
        sb.append("0123456789abcdef".charAt(b&0xf));
    }
    return sb.toString();
}

public static String BArrToStringLittle (byte bArr[], int off, int len)
{
    StringBuffer sb = new StringBuffer(bArr.length*2);
    for (int i=len-1; i>=0; --i) {
        byte b = bArr[i+off];
        sb.append("0123456789abcdef".charAt((b>>4)&0xf));
        sb.append("0123456789abcdef".charAt(b&0xf));
    }
    return sb.toString();
}

public static void StringToBArrBig (String s, byte bArr[], int off)
{
    int len = (s.length()+1)>>>1;
    int ioff = s.length()&1;
    if (ioff!=0) {
        char c1 = s.charAt(0);
        byte b1 = (byte)((c1>='Z')?(c1-'a'+10):(c1>'9')?(c1-'A'+10):(c1-'0'));
        bArr[off++] = b1;
        len--;
    }
    for (int i=0; i<len; ++i) {
        char c1 = s.charAt(ioff++);
        byte b1 = (byte)((c1>='Z')?(c1-'a'+10):(c1>'9')?(c1-'A'+10):(c1-'0'));
        char c2 = s.charAt(ioff++);
        byte b2 = (byte)((c2>='Z')?(c2-'a'+10):(c2>'9')?(c2-'A'+10):(c2-'0'));
        bArr[off++] = (byte)((b1<<4) | b2);
    }
}

public static void StringToBArrLittle (String s, byte bArr[], int off)
{
    int len = (s.length()+1)>>>1;
    int ioff = s.length()&1;
    off += len;
    if (ioff!=0) {
        char c1 = s.charAt(0);
        byte b1 = (byte)((c1>='Z')?(c1-'a'+10):(c1>'9')?(c1-'A'+10):(c1-'0'));
        bArr[--off] = b1;
        len--;
    }
    for (int i=0; i<len; ++i) {
        char c1 = s.charAt(ioff++);
        byte b1 = (byte)((c1>='Z')?(c1-'a'+10):(c1>'9')?(c1-'A'+10):(c1-'0'));
        char c2 = s.charAt(ioff++);
        byte b2 = (byte)((c2>='Z')?(c2-'a'+10):(c2>'9')?(c2-'A'+10):(c2-'0'));
        bArr[--off] = (byte)((b1<<4) | b2);
    }
}


} // class bytes
