/* pair objects */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#include "fools.h"
#include "utils.h"

#ifndef lint
static char SccsId[] = "@(#)Pair.c	1.10 1/20/90";
#endif

/* list printing parameters (unlimited if negative) */
int ListDepth = -1, ListBreadth = -1;

void objSetCar(pair, val)
     Obj pair, val;
{
    Obj car;

    ASSERT(CLASS(pair) == Pair || CLASS(pair) == Binding);
    objLink(val);
    if (car = objCar(pair))
	objUnlink(car);
    DATA(pair, car, pairInst) = val;
}

void objSetCdr(pair, val)
     Obj pair, val;
{
    Obj cdr;

    ASSERT(CLASS(pair) == Pair || CLASS(pair) == Binding);
    objLink(val);
    if (cdr = (objCdr(pair)))
	objUnlink(cdr);
    DATA(pair, cdr, pairInst) = val;
}

/* destructor */
static void pairDestroy(pair)
     Obj pair;
{
    Obj dat;

    if (dat = objCar(pair)) objUnlink(dat);
    if (pair = objCdr(pair)) {
	Obj car, cdr;

	/* lighten the number of recursive calls necessary
	 * to unlink large lists */
	while (CLASS(pair) == Pair) {
	    car = objCar(pair);
	    cdr = objCdr(pair);
	    if (--DATA(pair, rc, basicInst) == 0) {
		gcFree(pair);
		if (car) objUnlink(car);
		if (cdr == (Obj)NULL) return ;
		pair = cdr;
	    }
	    else return ;
	}
	objUnlink(pair);
    }
}

static char *specialPrints(sym)
     Obj sym;
{
    if (sym == QuoteSymb) return "'";
    else if (sym == QuasiSymb) return "`";
    else if (sym == UnquoteSymb) return ",";
    else if (sym == SpliceSymb) return ",@";

    else return (char *)NULL;
}

static int printPair(pair, file, depth)
     Obj pair;
     FILE *file;
     int depth;
{
    int breadth = 0;
    Obj temp;
    char *abbrev;
    Callback_t cb;

    cb.arg = (Ptr)pair;
    cb.func = clearMarks;
    errorPushCB(&cb);

    if (checkCond(pair, MARK) || (ListDepth >= 0 && ++depth > ListDepth)) {
	fputs("(...)", file);
	goto cleanup;
    }
    temp = objCar(pair);

    /* look for forms, like quote, with abbreviations */
    if (abbrev = specialPrints(temp)) {
	Obj arg;

	setCond(pair, MARK);
	arg = objCdr(pair);
	/* check if there's only one arg */
	if (CLASS(arg) == Pair && objCdr(arg) == NilSymb) {
	    fputs(abbrev, file);
	    if (checkCond(arg, MARK)) {
		fputs("...", file);
		goto cleanup;
	    }
	    setCond(arg, MARK);

	    arg = objCar(arg);
	    if (CLASS(arg) == Pair) printPair(arg, file, depth);
	    else objPrint(arg, file);

	    goto cleanup;
	}
	/* else invalid combination */
	else clearCond(pair, MARK);
    }

    putc('(', file);
    for (; CLASS(pair) == Pair; pair = temp) {
	intCheck();
	if (checkCond(pair, MARK) ||
	    (ListBreadth >= 0 && ++breadth > ListBreadth)) {
	    fputs("...)", file);
	    goto cleanup;
	}
	setCond(pair, MARK);

	if (CLASS(temp = objCar(pair)) == Pair) printPair(temp, file, depth);
	else if (temp == UnquoteSymb && CLASS(objCdr(pair)) == Pair
		 && objCdr(objCdr(pair)) == NilSymb) {
	    clearCond(pair, MARK);
	    break ;
	}
	else objPrint(temp, file);

	if ((temp = objCdr(pair)) != NilSymb) putc(' ', file);
	ASSERT(temp != (Obj)NULL);
    }
    if (pair != NilSymb) objfPrintf(file, ". %O", pair);
    putc(')', file);

 cleanup:
    clearMarks((Obj)cb.arg);
    errorPopCB();
}

/* printer */
static void pairPrint(pair, file)
     Obj pair;
     FILE *file;
{
    printPair(pair, file, 0);
}

/* constructor */
Obj newPair(alloc, car, cdr)
     F_OBJ alloc;
     Obj car, cdr;
{
    Obj new;

    new = (*alloc)(Pair);
    DATA(new, car, pairInst) = car;
    DATA(new, cdr, pairInst) = cdr;
    objLink(car);
    objLink(cdr);

    return new;
}

basicClass_t protoPair =
    DEFBASIC(Basic, pairInst_t, pairPrint, pairDestroy, "pair");

/* bindings */

/* bindings are stored in the binary tree of a frame */

static void bindingPrint(binding, file)
     Obj binding;
     FILE *file;
{
    char *sym;

    sym = objString(objCar(binding));
    while (binding && CLASS(binding) == Binding)
	binding = objCdr(binding);
    objfPrintf(file, "#<%s=%O>", sym, binding);
}

/* Set the value of the binding. */
void objSetBinding(binding, val, frame, bot)
     Obj binding, val, frame;
     Boolean bot;
{
    Obj prev;

    ASSERT(objIsClass(binding, Binding));
    if (bot) {
	while (CLASS(binding) == Binding) {
	    prev = binding;
	    if ((binding = objCdr(binding)) == (Obj)NULL)
		break ;
	}
    }
    else prev = binding;

    if (binding != (Obj)NULL && CLASS(binding) == FBinding) {
	Obj *fixed;
	int links;

	links = DATA(binding, links, fbindInst);
	while (--links >= 0) frame = DATA(frame, parent, frameInst);
	fixed = DATA(frame, fixed, frameInst)
	    + DATA(binding, offset, fbindInst);
	if (val) objLink(val);
	if (*fixed) objUnlink(*fixed);
	*fixed = val;
    }
    else objSetCdr(prev, val);
}
	
/* Return the value of binding or NULL if undefined. */
Obj objUnbind(binding, frame)
     Obj binding, frame;
{
    ASSERT(objIsClass(binding, Binding));
    while (CLASS(binding) == Binding) {
	if ((binding = objCdr(binding)) == (Obj)NULL)
	    return (Obj)NULL;
    }
    if (CLASS(binding) == FBinding) {
	int links;

	links = DATA(binding, links, fbindInst);
	while (--links >= 0) frame = DATA(frame, parent, frameInst);
	return DATA(frame, fixed, frameInst)[DATA(binding, offset, fbindInst)];
    }
    return binding;
}

Obj newBinding(alloc, key, val)
     F_OBJ alloc;
     Obj key, val;
{
    Obj new;

    new = (*alloc)(Binding);
    DATA(new, car, pairInst) = key;
    objLink(key);
    if (val) objLink(DATA(new, cdr, pairInst) = val);

    return new;
}

static void bindingDestroy(obj)
     Obj obj;
{
    Obj temp;

    if (temp = objCar(obj)) objUnlink(temp);
    if (CLASS(obj) == Binding) {
	if (temp = objCdr(obj)) objUnlink(temp);
    }
}

basicClass_t protoBinding =
    DEFBASIC(Basic, pairInst_t, bindingPrint, bindingDestroy, "binding");

/* fixed bindings refer to fixed binding vector of frames */

static void fbindPrint(fb, file)
     Obj fb;
     FILE *file;
{
    (void)fprintf(file, "#<fixed[%d][%d]>",
		  DATA(fb, links, fbindInst), DATA(fb, offset, fbindInst));
}

Obj newFBinding(alloc, key, links, offset)
     F_OBJ alloc;
     Obj key;
     short links, offset;
{
    Obj new;

    new = (*alloc)(FBinding);
    if (key) {
	objLink(key);
	DATA(new, car, fbindInst) = key;
    }
    DATA(new, links, fbindInst) = links;
    DATA(new, offset, fbindInst) = offset;

    return new;
}

basicClass_t protoFBinding =
    DEFBASIC(Binding, fbindInst_t, fbindPrint, (F_VOID)NULL, "fixed-binding");
