/* strings and symbols */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#include "fools.h"
#include <ctype.h>

#ifndef lint
static char SccsId[] = "@(#)String.c	1.11 2/23/90";
#endif

static void stringPrint(str, file)
     Obj str;
     FILE *file;
{
    char *val;
    int c;
    
    if (val = DATA(str, string, stringInst)) {
	putc('\"', file);
	while (c = *val++) {
	    switch (c) {
	    case '\"':
		putc('\\', file);
		putc('\"', file);
		break ;
	    case '\n':
		putc('\\', file);
		putc('n', file);
		break ;
	    case '\t':
		putc('\\', file);
		putc('t', file);
		break ;
	    case '\\':
		putc('\\', file);
		putc('\\', file);
		break ;
	    default:
		if (!isascii(c) || !isprint(c))
		    (void)fprintf(file, "\\x%.2x", c & 0xff);
		else putc(c, file);
		break ;
	    }
	}
	putc('\"', file);
    }
    else fputs("#<empty string>", file);
}

static void stringDestroy(str)
     Obj str;
{
    if (objString(str) && !checkCond(str, STATIC))
	free(objString(str));
}

static stringInst_t emptystr = { { String, 0, 1 }, "", 0 };

Obj newString(alloc, str)
     F_OBJ alloc;
     char *str;
{
    Obj new;
    int len;

    if (str == (char *)NULL || (len = strlen(str)) == 0)
	return (Obj)&emptystr;

    new = (*alloc)(String);
    DATA(new, string, stringInst) = strcopy(str);
    DATA(new, maxsize, stringInst) = len;

    return new;
}

Obj newString1(alloc, size)
     F_OBJ alloc;
     int size;
{
    Obj new;

    if (size == 0) return (Obj)&emptystr;
    new = (*alloc)(String);
    DATA(new, string, stringInst) = NEWVEC(char, size + 1);
    DATA(new, maxsize, stringInst) = size;
    return new;
}

basicClass_t protoString =
    DEFBASIC(Basic, stringInst_t, stringPrint, stringDestroy, "string");

/* symbol class */

#define HASHSIZE 887

static list_t symbolTable[ HASHSIZE ]; /* hash table for symbols */
	
/* Generate a hash value in the range 0 .. HASHSIZE-1 from s.
 * Algorithm due to P. J. Weinberger. */
static int hash(s)
     char *s;
{
    unsigned int r;

    for (r = 0; *s != NULL; s += 1) {
        r = (r << 4) + *s;
        if (r > 0x0fffffff) {
            r ^= (r >> 24) & 0xf0;
            r &= 0x0fffffff;
        }
    }
    return r % HASHSIZE;
}

static Boolean streq(obj, str)
     Obj obj;
     char *str;
{
    return strcmp(DATA(obj, string, stringInst), str) == 0;
}

/* Return the unique symbol corresponding to str. */
Obj objIntern(str, status)
     char *str;
     int status;
{
    Obj entry;
    List chain;
    
    chain = &symbolTable[ hash(str) ];
    if (entry = (Obj)listFind(chain, streq, (Ptr)str))
	return entry;

    entry = gcNew(Symbol);
    setCond(entry, status);
    DATA(entry, maxsize, stringInst) = strlen(str);
    DATA(entry, string, stringInst) =
	checkCond(entry, STATIC) ? str : strcopy(str);
    objLink(entry);
    listPush((Ptr)entry, chain);
    return entry;
}

	/* symbol properties */

/* root nodes for trees to hold macros and special forms indexed by symbol */
static TreeNode macroTree, sformTree;

/* property bindings */
struct sym_prop_s {
    Obj sym;
    Ptr data;
};

/* compare the symbols of two struct sym_prop_ses */
static int propCompare(a, b)
     struct sym_prop_s *a, *b;
{
    return (long)a->sym - (long)b->sym;
}

/* compare a symbol to a struct sym_prop_s */
static int propCompare1(a, b)
     Obj a;
     struct sym_prop_s *b;
{
    return (long)a - (long)b->sym;
}
    
/* insert new and unlink the old */
static Ptr objInsPropDest(new, old)
     struct sym_prop_s *new, *old;
{
    Obj dat;

    if (old) {
	if (old->data)
	    objUnlink((Obj)old->data);
    }
    else {
	old = (struct sym_prop_s *)memBlock(listAlloc);
	old->sym = new->sym;
    }

    if (dat = (Obj)new->data) objLink(dat);
    old->data = (Ptr)dat;

    return (Ptr)old;
}

/* insert new only if no previous value */
static Ptr objInsProp(new, old)
     struct sym_prop_s *new, *old;
{
    if (old) {
	if (old->data) return (Ptr)old;
    }
    else old = (struct sym_prop_s *)memBlock(listAlloc);

    if (new->data) objLink((Obj)new->data);
    *old = *new;

    return (Ptr)old;
}
    
/* get property for sym from tree */
Obj objGetProp(sym, tree)
     Obj sym;
     Tree tree;
{
    Tree root;

    ASSERT(sym && objIsClass(sym, Symbol));
    root = treeFind((Ptr)sym, tree, propCompare1);
    return root ? (Obj)((struct sym_prop_s *)treeRoot(root))->data : (Obj)NULL;
}

/* set property for sym to data */
void objSetProp(sym, data, tree, dest)
     Obj sym, data;
     Tree tree;
     Boolean dest; /* destroy previous contents if TRUE */
{
    struct sym_prop_s prop;

    ASSERT(sym && objIsClass(sym, Symbol));

    prop.sym = sym;
    prop.data = (Ptr)data;
    (void)treeInsert((Ptr)&prop, tree, propCompare,
		     (dest ? objInsPropDest : objInsProp));
}

Obj objMacro(sym)
     Obj sym;
{
    return objGetProp(sym, &macroTree);
}

Obj objSForm(sym)
     Obj sym;
{
    return objGetProp(sym, &sformTree);
}

void objSetMacro(sym, macro)
     Obj sym, macro;
{
    if (macro) setCond(sym, MACRO);
    objSetProp(sym, macro, &macroTree, TRUE);
}

void objSetSForm(sym, sform)
     Obj sym, sform;
{
    if (sform) setCond(sym, SFORM);
    objSetProp(sym, sform, &sformTree, TRUE);
}

static void symbolPrint(sym, file)
     Obj sym;
     FILE *file;
{
    char *val;

    val = DATA(sym, string, stringInst);
    ASSERT(val);
    (void)fputs(val, file);
}

struct basicClass_s protoSymbol =
    DEFBASIC(String, symbolInst_t, symbolPrint, (F_VOID)NULL, "symbol");
