/* compile to byte codes */

/*
 * Copyright 1989 Jonathan Lee.  All rights reserved.
 *
 * Permission to use, copy, and/or distribute for any purpose and
 * without fee is hereby granted, provided that both the above copyright
 * notice and this permission notice appear in all copies and derived works.
 * Fees for distribution or use of this software or derived works may only
 * be charged with express written permission of the copyright holder.
 * This software is provided ``as is'' without express or implied warranty.
 */

#ifndef lint
static char SccsId[] = "@(#)codegen.c	1.13 2/23/90";
#endif

#include "fools.h"
#include "codegen.h"
#include "cont.h"
#include "sforms.h"

Obj ccVec; /* constant codevec for call/cc */
static Mem codegenAlloc, labelAlloc, labeldefAlloc, bcodeAlloc;

#define makeAlloc(type, deflt, size)\
((sizeof (type) <= (deflt)->blksize) ?\
 (deflt) : memNew(ALIGNSIZE(sizeof (type)), size))

/* initialize codegen/opcode module */
void codeInit()
{
    static bcode vec[5];
    static codevecInst_t cvec = {
	{ Basic, 0, 1 }, (Obj)NULL, (Obj)NULL, vec, vec + 4, 
    };

    /* ccVec applies the procedure on the top of the stack to the
     * next object on the stack.  This is a constant codevector
     * used by the call/cc code. */
    vec[0].op = code_call;
    vec[1].num = 1;
    vec[2].op = code_halt;
    vec[3].op = code_term;
    vec[4].obj = (Obj)NULL;

    ccVec = (Obj)&cvec;

    /* set up allocators for codegen data structures */
    codegenAlloc = makeAlloc(CodeGen_t, treeAlloc, 4);
    labelAlloc = makeAlloc(Label_t, listAlloc, 256);
    labeldefAlloc = makeAlloc(LabelDef_t, listAlloc, 256);
    bcodeAlloc = makeAlloc(bcode, listAlloc, 256);
    stateAlloc = memNew(ALIGNSIZE(sizeof (struct contState_s)), 4);

    opcodeInit();
}

static void labelDefFree(ldef)
     LabelDef_t *ldef;
{
    memFreeBlock((Ptr)ldef, labeldefAlloc);
}

static void labelFree(lab)
     Label_t *lab;
{
    listFree(lab->refs, labelDefFree);
    memFreeBlock((Ptr)lab, labelAlloc);
}

#define bcodeNew() ((bcode *)memBlock(bcodeAlloc))

static void bcodeFree(bc)
     bcode *bc;
{
    memFreeBlock((Ptr)bc, bcodeAlloc);
}

/* callback to free code generator */
static void codeCallback(gen)
     CodeGen gen;
{
    listFree(gen->insts, bcodeFree);
    listFree(gen->refs, _objUnlink);
    listFree(gen->labels, labelFree);
    memFreeBlock((Ptr)gen, codegenAlloc);
}

/* create a code generator */
CodeGen codeNew(cb)
     Callback_t *cb;
{
    CodeGen new;

    new = (CodeGen)memBlock(codegenAlloc);
    new->insts = listNew();
    new->refs = listNew();
    new->labels = listNew();
    new->len = 0;

    cb->arg = (Ptr)new;
    cb->func = codeCallback;
    errorPushCB(cb);

    return new;
}

/* free code gen */
void codeFree(gen)
     CodeGen gen;
{
    codeCallback(gen);
    errorPopCB();
}

#define codeEmit(op, type, gen)\
{\
    bcode *_bc = bcodeNew();\
    _bc->type = (op);\
    listPush((Ptr)_bc, (gen)->insts);\
    (gen)->len++;\
}

/* output inst to code generator */
void codeInst(inst, gen)
     opcode inst;
     CodeGen gen;
{
    codeEmit(inst, op, gen);
}

/* output integer C operand to code generator */
void codeOp(op, gen)
     int op;
     CodeGen gen;
{
    codeEmit(op, num, gen);
}

/* output C function pointer operand to code generator */
void codeFcnOp(op, gen)
     F_OBJ op;
     CodeGen gen;
{
    codeEmit(op, fcn, gen);
}

/* output Obj operand to code generator */
void codeGCOp(op, gen)
     Obj op;
     CodeGen gen;
{
    codeEmit(op, obj, gen);
    if (op) {
	objLink(op);
	listPush((Ptr)op, gen->refs);
    }
}

/* output code to push obj onto the stack to code generator */
void codePush(obj, gen)
     Obj obj;
     CodeGen gen;
{
    codeInst(code_push, gen);
    codeGCOp(obj, gen);
}

/* output code to pop obj from the stack */
void codePop(gen)
     CodeGen gen;
{
    codeInst(code_pop, gen);
}
	 
/* create a new label for code generator */
Label_t *labelNew(gen)
     CodeGen gen;
{
    Label_t *new;

    new = (Label_t *)memBlock(labelAlloc);
    new->refs = listNew();
    new->def = -1;
    listPush((Ptr)new, gen->labels);

    return new;
}

/* output a branch instruction for label to the generator */
void codeBranch(branch, label, gen)
     opcode branch;
     Label_t *label;
     CodeGen gen;
{
    LabelDef_t *entry;

    entry = (LabelDef_t *)memBlock(labeldefAlloc);
    codeInst(branch, gen);
    codeOp(0, gen); /* dummy value */
    entry->pos = gen->len;
    entry->ref = (bcode *)listPeek(gen->insts);
    listPush((Ptr)entry, label->refs);
}

/* mark where label is defined */
void labelDefine(label, gen)
     Label_t *label;
     CodeGen gen;
{
    ASSERT(label->def < 0); /* don't set more than once */
    label->def = gen->len;
}

/* resolve all labels (and free labelDefs) */
void labelResolve(gen)
     CodeGen gen;
{
    List labelist = gen->labels;
    Label_t *lab;

    while (lab = (Label_t *)listPop(labelist)) {
	LabelDef_t *used;
	int where;
	List refs;

	ASSERT(lab->def >= 0); /* label undefined? */
	where = lab->def;
	refs = lab->refs;
	while (used = (LabelDef_t *)listPop(refs)) {
	    (used->ref)->num = where - used->pos;
	    memFreeBlock((Ptr)used, labeldefAlloc);
	}
	listFree(refs, labelDefFree);
	memFreeBlock((Ptr)lab, labelAlloc);
    }
}

/* set up function for calling special form compilers */
static void
compSpecial(sform, args, gen, tail, opt)
     Obj sform, args;
     CodeGen gen;
     Boolean tail;
     optInfo_t *opt;
{
#define SF_ARGC 32
    Obj ptr, *argv, *temp, targv[ SF_ARGC ];
    int argc, numargs;

    for (argc = 0, ptr = args; CLASS(ptr) == Pair; argc++) ptr = objCdr(ptr);
    if (ptr != NilSymb)
	errorPrint(BadSyntax, "in arglist of %O", sform);

    numargs = DATA(sform, numargs, procInst);
    if (checkCond(sform, OPTARG)) {
	if (numargs > argc)
	    errorPrint(BadArgs, "to %O (passing %d, expecting at least %d)",
		       sform, argc, numargs);
    }
    else {
	if (numargs != argc)
	    errorPrint(BadArgs, "to %O (passing %d, expecting %d)",
		       sform, argc, numargs);
    }

    temp = argv = (argc > SF_ARGC ? NEWLOCVEC(Obj, argc) : targv);

    for (ptr = args; ptr != NilSymb; ptr = objCdr(ptr))
	*(temp++) = objCar(ptr);

    /* invoke special form compiler */
    (*DATA(sform, compiler, specialInst))(argv, argc, gen, tail, opt);

    if (argc > SF_ARGC) FREELOCVEC(argv);
}

/* construct a list (exactly like primList) */
SF_EVAL(codeCollect)
{
    Obj args = NilSymb;

    argv += argc;
    while (--argc >= 0)
	args = newPair(gcNew, *--argv, args);
    return args;
}

static Boolean compTailRecursiveIdent(b1, b2)
     Obj b1, b2;
{
    if (CLASS(b1) == FBinding && CLASS(b2) == FBinding)
	return objCar(b1) == objCar(b2);
    return b1 == b2;
}
	
/* compile function call */
static void compFunction(op, args, gen, tail, opt)
     Obj op, args;
     CodeGen gen;
     Boolean tail;
     optInfo_t *opt;
{
    int nargs;

    if (objIsClass(op, Special)) {
	compSpecial(op, args, gen, tail, opt);
	return ;
    }

    for (nargs = 0; args != NilSymb; ++nargs) {
	if (CLASS(args) != Pair)
	    errorPrint(BadSyntax, "argument list to %O", objCar(op));
	compExpr(objCar(args), gen, FALSE, opt);
	args = objCdr(args);
    }
    if (tail && opt->id && compTailRecursiveIdent(op, opt->id)) {
	/* generate a goto instead of a tail recursive call */
	if (nargs == opt->numargs || (nargs > opt->numargs && opt->optarg)) {
	    if (opt->optarg) {
		int optargs = nargs - opt->numargs;

		if (optargs > 0) {
		    codeInst(code_cfunc, gen);
		    codeFcnOp(codeCollect, gen);
		    codeOp(optargs, gen);
		}
		else codePush(NilSymb, gen);
	    }
	    codeInst(code_replace, gen);
	    codeOp(opt->numargs + (opt->optarg != 0), gen);
	    codeBranch(code_goto, opt->top, gen);
	}
	else errorPrint(BadArgs, "to %O (passing %d, expecting %d)",
			objCar(op), nargs, opt->numargs);
    }
    else {
	compExpr(op, gen, FALSE, opt);
	codeInst((tail ? code_tail_call : code_call), gen);
	codeOp(nargs, gen);
    }
}

/* Output compiled expression to gen.  tail is TRUE if expr is in
 * the tail position of lambda body. */
void compExpr(expr, gen, tail, opt)
     Obj expr;
     CodeGen gen;
     Boolean tail;
     optInfo_t *opt;
{
    Class type = CLASS(expr);

    if (type == Pair)
	compFunction(objCar(expr), objCdr(expr), gen, tail, opt);
    else if (type == Binding) {
	codeInst(code_unbind, gen);
	codeGCOp(expr, gen);
    }
    else if (type == FBinding) {
	int links = DATA(expr, links, fbindInst);

	if (links == 0) {
	    codeInst(code_local, gen);
	    codeOp(DATA(expr, offset, fbindInst), gen);
	}
	else {
	    codeInst(code_access, gen);
	    codeOp(links, gen);
	    codeOp(DATA(expr, offset, fbindInst), gen);
	}
    }
    else codePush(expr, gen);
}

/* indent num positions using character ch */
static void indent(num, ch)
     int num;
     char ch;
{
    while (--num >= 0) putchar(ch);
}

/* Print code vector vec with indentation of depth. */
void codevecPrint(cvec, depth)
     Obj cvec;
     int depth;
{
    opcode op;
    bcode *vec, *pvec;
    int nops, i;

    ASSERT(objIsClass(cvec, CodeVec));
    pvec = vec = DATA(cvec, vec, codevecInst);
    indent(depth, '*');
    objPrintf("\n%O\n", DATA(cvec, expr, codevecInst));

    while ((op = codeNext(vec).op) != code_term) {
	nops = opcodeNops(op);
	indent(depth, ' ');
	(void)printf("%4d: %s(", (vec - pvec) - 1, opcodeName(op));

	switch (op) {
	
	case code_push:
	case code_unbind:
	    if (CLASS(vec->obj) == CodeVec) {
		indent(depth + 1, '*');
		puts(")");
		codevecPrint(vec->obj, depth + 1);
	    }
	    else objPrintf("%O)\n", vec->obj);
	    break ;

	case code_cfuncN:
	    i = vec[2].num;
	    (void)printf("cfunc=%#lx, n=%d, N=%d",
			 (long)vec[0].fcn, vec[1].num, vec[2].num);
	    for (i = 0; i < vec[2].num; ++i)
		(void)printf(", op%d=%d", i, vec[2+i].num);
	    puts(")");
	    vec += vec[2].num;
	    break ;

	case code_cfunc:
	    (void)printf("cfunc=0x%lx, n=%d)\n", (long)vec[0].fcn, vec[1].num);
	    break ;
	    
	default:
	    for (i = 0; i < nops; i++)
		(void)printf("%s%ld", (i == 0 ? "" : ", "), (long)vec[i].num);
	    puts(")");
	    break ;
	}
	vec += nops;
    }
    indent(depth, '*'); putchar('\n');
}

	/* code-vector type */

/* printer */

/* destructor */
static void codeDestroy(code)
     Obj code;
{
    Obj tmp;
    bcode *refs;

    refs = DATA(code, refs, codevecInst);
    while (tmp = (refs++)->obj) objUnlink(tmp);
    (void)free((char *)DATA(code, vec, codevecInst));
    if (tmp = DATA(code, locals, codevecInst)) objUnlink(tmp);
    if (tmp = DATA(code, expr, codevecInst)) objUnlink(tmp);
}

/* Convert gen into a codevector
 *  
 * exp is the lambda expression that was compiled
 * fp is a list of parameters and local variables in exp
 * gen is the code generator containing the instructions which are
 * to be placed in the codevector */
Obj newCodevec(alloc, gen, locals, expr)
     F_OBJ alloc;
     CodeGen gen;
     Obj locals, expr;
{
    Obj new;
    bcode *vec, *inst, *refs;
    List l;
    int len, rlen;

    len = gen->len;
    rlen = listLength(gen->refs);
    vec = NEWVEC(bcode, len + rlen + 2);

    new = (*alloc)(CodeVec);
    DATA(new, vec, codevecInst) = vec;
    DATA(new, refs, codevecInst) = refs = vec + len + 1;
    if (DATA(new, locals, codevecInst) = locals) objLink(locals);
    if (DATA(new, expr, codevecInst) = expr) objLink(expr);

    labelResolve(gen);

    l = gen->refs;
    while (--rlen >= 0)
	(refs++)->obj = (Obj)listPop(l);
    refs->obj = (Obj)NULL;
    
    l = gen->insts;
    vec += len;
    vec->op = code_term;
    while (--len >= 0) {
	inst = (bcode *)listPop(l);
	*(--vec) = *inst;
	bcodeFree(inst);
    }
    codeFree(gen);

    return new;
}
	
basicClass_t protoCode =
    DEFBASIC(Basic, codevecInst_t, (F_VOID)NULL, codeDestroy, "code-vector");
