;;; The "first pass" of this Scheme compiler reads the source files and
;;; performs the following operations:
;;;
;;;	- macro and special form expansion
;;;	- alpha-conversion
;;;     - lexical variable usage recording
;;;
;;; At the end of this pass, all bindings and control flows should be visible
;;; in the tree.
;;;

;*              Copyright 1989 Digital Equipment Corporation
;*                         All Rights Reserved
;*
;* Permission to use, copy, and modify this software and its documentation is
;* hereby granted only under the following terms and conditions.  Both the
;* above copyright notice and this permission notice must appear in all copies
;* of the software, derivative works or modified versions, and any portions
;* thereof, and both notices must appear in supporting documentation.
;*
;* Users of this software agree to the terms and conditions set forth herein,
;* and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
;* right and license under any changes, enhancements or extensions made to the
;* core functions of the software, including but not limited to those affording
;* compatibility with other hardware or software environments, but excluding
;* applications which incorporate this software.  Users further agree to use
;* their best efforts to return to Digital any such changes, enhancements or
;* extensions that they make and inform Digital of noteworthy uses of this
;* software.  Correspondence should be provided to Digital at:
;* 
;*                       Director of Licensing
;*                       Western Research Laboratory
;*                       Digital Equipment Corporation
;*                       100 Hamilton Avenue
;*                       Palo Alto, California  94301  
;* 
;* This software may be distributed (but not offered for sale or transferred
;* for compensation) to third parties, provided such third parties agree to
;* abide by the terms and conditions of this notice.  
;* 
;* THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
;* WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
;* MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
;* CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
;* DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;* PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
;* ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
;* SOFTWARE.

(module expform)

;;; Pick up external declarations.

(include "plist.sch")
(include "expform.sch")
(include "lambdaexp.sch")

;;; During compilation, variable binding information is kept in the following
;;; global variables.  Each is an a-list with entries of the form:
;;;
;;;	(external-name alpha-converted-name)
;;;
;;; LEXICAL-BOUND-VARS contains the variables which are bound at the current
;;; lexical level.  LEXICAL-FREE-VARS contains the variables which are
;;; lexically bound at higher lexical levels.  GLOBAL-FREE-VARS contains those
;;; variables which are bound at the "top-level".
 
(define GLOBAL-FREE-VARS '())

(define LEXICAL-FREE-VARS '())

(define LEXICAL-BOUND-VARS '())

;;; Information relating to the current lambda expression is kept in the
;;; following variables.  CURRENT-LAMBDA-ID is the identifier for the current
;;; lambda expression.

(define CURRENT-LAMBDA-ID 'top-level)

;;; Alpha-conversion requires the generation of unique names.  The sequence
;;; number which is used is kept in MAKE-ALPHA-SEQ.

(define MAKE-ALPHA-SEQ 0)

;;; Source processing starts with the following function which is entered with
;;; a generator function for the source.  It will return a list of forms which
;;; is the result of the first pass.

(define (EXPAND-FORMS)
    (let ((results '()))
	 (do ((exp (read-text) (read-text)))
	     ((eof-object? exp) (set! results (reverse results)))
	     (set! lexical-free-vars '())
	     (set! current-lambda-id 'top-level)
	     (set! exp (exp-form exp exp-form))
	     (if exp (set! results (cons exp results))))
	 (for-each
	     (lambda (var-alpha)
		     (let ((var   (car var-alpha))
			   (alpha (cadr var-alpha)))
			  (if (and (eq? (id-use alpha) 'global)
				   (not (id-module alpha)))
			      (begin (set! current-define-name
					   (id-undefref alpha))
				     (report-warning
					 "Variable assumed to be TOP-LEVEL:"
					 var)
				     (set-id-use! alpha 'top-level)
				     (set-id-module! alpha 'top-level)
				     (set-id-vname! alpha
					 (string-append
					     (hex28 "" (lchexname var))
					     "_v"))
				     (set! quote-constants
					   (cons (list var alpha)
						 quote-constants))))))
	     global-free-vars)
	 results))

;;; The expressions are recursively expanded by the following function which
;;; is called with the expression and the expansion function.  The expansion
;;; process is similar to macro expansion, but it does the alpha-conversion
;;; using the functions stored under the property EXPAND.

(define (EXP-FORM exp exp-func)
    (cond  ((symbol? exp)
	    (bound exp))
	   ((or (number? exp) (string? exp) (char? exp)
		(member exp '(#t #f)))
	    (exp-func (list 'quote exp) exp-func))
	   ((islist exp 1)
	    (let ((func (if (symbol? (car exp)) (get (car exp) 'expand) '())))
		 (apply (if func func call-exp)	(list exp exp-func))))
	   (else
	    (expand-error "" exp))))

;;; A similar function is used to expand a list of functions.

(define (EXP-FORM-LIST exp-list func)
    (if (islist exp-list 0)
	(map (lambda (exp) (func exp func)) exp-list)
	(expand-error 'expression-list exp-list)))

;;; During the alpha-conversion phase, all variables will be replaced with
;;; unique variables.  Information about each variable will be saved as
;;; properties of the alpha-converted variable.  The items saved are:
;;;
;;; PRINTNAME:	original program variable name.
;;; VNAME:	C name to access the item as a variable.
;;; CNAME:	C name to access the item as a procedure.
;;; MODULE:	module name containing the item.
;;; USE:	tag indicating what the variable signifies.  The possible
;;;		tags are:  LABEL, LAMBDA, LEXICAL, CONSTANT, GLOBAL, TOP-LEVEL,
;;;	        TEMPORARY, CLOSUREP, and MACRO.
;;; TYPE:	data type which is either false indicating a TSCP or the
;;;		appropriate C datatype.
;;; DISPLAY:	boolean that indicates that the variable is be allocated in a
;;;		display cell.
;;; BOUNDID:	id of the lambda expression where this variable is bound.
;;; LAMBDA:	id of the lambda expression which is this var's value.
;;; EXTERNAL:   indicates that variable is external to this compile and is
;;;		referenced.
;;; VALUE:      value for identifiers which are constants.
;;; SET!:	boolean indicating that the variable has been SET!.
;;; REFS:	counter for # of times a lambda bound variable is referenced.
;;; CALLS:      counter for # of times a lambda bound variable is called as a
;;;		function.
;;; ALIAS:      label alias (see emit-lap).
;;; GOTOS:      counter for # of branches to a label.
;;; UNDEFREF    current-define-name for first use when undefined.

(define (ID-PRINTNAME id) (id-printname id))

(define (SET-ID-PRINTNAME! id name)  (set-id-printname! id name))

(define (ID-VNAME id)  (id-vname id))

(define (SET-ID-VNAME! id name) (set-id-vname! id name))

(define (ID-CNAME id) (id-cname id))

(define (SET-ID-CNAME! id name)  (set-id-cname! id name))

(define (ID-MODULE id)  (id-module id))

(define (SET-ID-MODULE! id name)  (set-id-module! id name))

(define (ID-USE id) (id-use id))

(define (SET-ID-USE! id tag) (set-id-use! id tag))

(define (ID-TYPE id) (id-type id))

(define (SET-ID-TYPE! id tag) (set-id-type! id tag))

(define (ID-DISPLAY id) (id-display id))

(define (SET-ID-DISPLAY! id flag) (set-id-display! id flag))

(define (ID-BOUNDID id) (id-boundid id))

(define (SET-ID-BOUNDID id value)(set-id-boundid id value))

(define (ID-LAMBDA id) (id-lambda id))

(define (SET-ID-LAMBDA! id lambda-id) (set-id-lambda! id lambda-id))

(define (ID-EXTERNAL id) (id-external id))

(define (SET-ID-EXTERNAL! id flag) (set-id-external! id flag))

(define (ID-VALUE id) (id-value id))

(define (SET-ID-VALUE! id x) (set-id-value! id x))

(define (ID-SET! id) (id-set! id))

(define (SET-ID-SET!! id flag) (set-id-set!! id flag))

(define (ID-REFS id) (id-refs id))

(define (SET-ID-REFS! id cnt) (set-id-refs! id cnt))

(define (ID-CALLS id) (id-calls id))

(define (SET-ID-CALLS! id cnt) (set-id-calls! id cnt))

(define (ID-ALIAS id) (id-alias id))
    
(define (SET-ID-ALIAS! id label) (set-id-alias! id label))

(define (ID-GOTOS id) (id-gotos id))

(define (SET-ID-GOTOS! id cnt) (set-id-gotos! id cnt))

(define (ID-UNDEFREF id) (id-undefref id))

(define (SET-ID-UNDEFREF! id var) (set-id-undefref! id var))

;;; Variables which represent globally defined items will have their property
;;; GLOBAL set to their alphatized variable.  This allows rapid global lookup.

(define (ID-GLOBAL id) (id-global id))

(define (SET-ID-GLOBAL! id alpha) (set-id-global! id alpha))

;;; Names are generated for externally visible variables by the following
;;; function.

(define (ASSIGN-KNOWN-NAME var)
    (let* ((use    (id-use var))
	   (module (id-module var))
	   (name   (lchexname (id-printname var))))
	  (cond ((memq use '(lexical closurep))
		 (let ((lcvar (lchexname var)))
		      (cond ((id-lambda var)
			     (set-id-cname! var
				 (string-append module-name "_" lcvar))
			     (set-id-vname! var (string-append lcvar "_v")))
			    (else
			     (set-id-vname! var lcvar)))))
		((and (eq? use 'global) (id-type var)))
		(else
		 (set-id-vname! var (string-append (hex28 module name) "_v"))
		 (set-id-cname! var (string-append (hex28 module name)))))))

;;; This function is called to establish the linkage between a variable and a
;;; lambda expression.

(define (NAME-A-LAMBDA name exp)
    (set! exp ($lambda-id exp))
    (if exp
	(begin (set-id-lambda! name exp)
	       (set-lambda-name! exp name))))

;;; Often one wants the VNAME or CNAME of an arbitrary expression.  These
;;; functions  will produce it.

(define (VNAME exp)
    (if (symbol? exp)
	(begin (if (and (eq? (id-use exp) 'lambda) (lambda-name exp))
		   (set! exp (lambda-name exp)))
	       (id-vname exp))
	exp))

(define (CNAME exp)
    (if (symbol? exp)
        (begin (if (and (eq? (id-use exp) 'lambda) (lambda-name exp))
		   (set! exp (lambda-name exp)))
               (id-cname exp))
        exp))

;;; This function is called to convert a name into its "lower case hex" format.

(define (LCHEXNAME name)
    (if (symbol? name) (set! name (symbol->string name)))
    (do ((c '())
	 (i 0 (+ 1 i))
	 (new (list 1)))
	((= i (string-length name)) (list->string (cdr new)))
	(set! c (string-ref name i))
	(cond ((char=? c #\_)
	       (set-cdr! (last-pair new) (list #\_ #\_)))
	      ((and (char>=? c #\A) (char<=? c #\Z))
	       (set-cdr! (last-pair new)
		   (list (integer->char (+ (char->integer c) 32)))))
	      ((or (and (char>=? c #\a) (char<=? c #\z))
		   (and (char>=? c #\0) (char<=? c #\9) (> i 0)))
	       (set-cdr! (last-pair new) (list c)))
	      (else
		(set-cdr! (last-pair new) (cons #\_ (char->dl c 16 2)))))))

;;; This function is one of those that you hope you never have to write, but
;;; inevitably you must.  It exists because vcc will only recognize the first
;;; 31 characters of a variable name.  In order to force the first 31
;;; characters of a generated name to be unique, it is necessary that the
;;; lchexnames of the module and variable be less than or equal to 28
;;; characters.  If it doesn't fit, then a name is generated consisting of
;;; the last 9 characters of the module name, the last 10 characters of the
;;; name, and the hex crc-32 of the module and name.

(define (HEX28 module name)
    (if (<= (+ (string-length module) (string-length name)) 28)
	(if (equal? module "") name (string-append module "_" name))
	(let ((crc 0))
	     (for-each
		 (lambda (c)
			 (do ((bit (char->integer c) (bit-rsh bit 1))
			      (cnt 8 (- cnt 1)))
			     ((= cnt 0))
			     (set! crc (crc32 bit crc))))
		 (string->list (string-append module name)))
	     (set! crc (number->string crc 16))
	     (set! crc (substring crc 2 (string-length crc)))
	     (format '() "~a_~a_~a"
		     (substring module (max 0 (- (string-length module) 9))
			 (string-length module))
		     (substring name (max 0 (- (string-length name) 10))
			 (string-length name))
		     crc))))

(define (CRC32 bit crc)
    (let* ((bitxor31 (bit-and (bit-xor bit crc) 1))
	   (mask (if (eq? bitxor31 0) 0 #b11011011011100010000011001000000)))
	  (bit-xor (bit-lsh bit 31) (bit-rsh (bit-xor mask crc) 1))))

;;; This function converts the character "c" into numeric string of length
;;; "len" in base "base".

(define (CHAR->DL c base len)
    (set! c (char->integer c))
    (do ((dl '()))
	((zero? len) dl)
	(set! dl (cons (string-ref "0123456789abcdef" (remainder c base)) dl))
	(set! c (quotient c base))
	(set! len (- len 1))))

;;; Variables are initially bound and their alpha-converted value is returned
;;; by the following function.  It takes the variable name and an optional
;;; list of properties and values.  It returns the alphabetized name.

(define (NEWV var . pl)
    (let* ((oldalpha (id-global var))
	   (use      (cadr (memq 'use pl)))
	   (alpha    '()))
	  (if (and oldalpha (memq use '(global macro lexical)))
	      (begin (if (and (id-module oldalpha)
			      (or (eq? (id-use oldalpha) 'macro)
				  (eq? use 'global)))
			     (report-warning
				 "Duplicately defined symbol:" var))
		     (if (eq? use 'global)
			 (begin (set! alpha oldalpha)
				(set-id-lambda! alpha '())
				(set-id-module! alpha '())
				(set-id-vname! alpha '())
				(set-id-cname! alpha '()))
			 (set! alpha (make-alpha var))))
	      (set! alpha (make-alpha var)))
	  (set-id-printname! alpha var)
	  (do ((pl pl (cddr pl)))
	      ((null? pl)
	       (case (id-use alpha)
		     ((global macro top-level)
		      (set-id-global! var alpha)
		      (if (not (eq? alpha oldalpha))
		          (set! global-free-vars
			        (cons (list var alpha) global-free-vars))))
		     ((lexical)
		      (set! lexical-bound-vars
			    (cons (list var alpha) lexical-bound-vars)))
		     ((label constant lambda temporary closurep)
		      (let ((dsa (downshift alpha)))
			   (set-id-printname! alpha alpha)
			   (if (eq? (id-use alpha) 'lambda)
			       (set-id-cname! alpha (hex28 module-name dsa))
			       (set-id-cname! alpha (hex28 "" dsa)))
			   (set-id-vname! alpha (hex28 "" dsa)))))
	       alpha)
	      (put alpha (car pl) (cadr pl)))))

;;; All variable names will be alpha-converted by taking the first character
;;; of their name and following it with an id number.

(define (MAKE-ALPHA var)
    (let* ((c (string-ref (symbol->string var) 0))
	   (alpha (string->symbol (format '() "~A~A" c make-alpha-seq))))
	  (set! make-alpha-seq (+ make-alpha-seq 1))
	  (if (id-printname alpha)
	      (make-alpha var)
	      alpha)))

;;; The following function looks up a variable in the current bindings.  If it
;;; is not found, then it will be added to GLOBAL-FREE-VARS.  TOP-LEVEL
;;; variables which are referenced will have a symbol pointer added to the
;;; constant list so that their value can be looked up.

(define (BOUND var)
    (let* ((varalist (assq var lexical-bound-vars))
	   (varlex   (or varalist (assq var lexical-free-vars)))
	   (varglob  (or varlex (id-global var))))
	  (cond (varalist
		 (cadr varalist))
		(varlex
		 (set! varlex (cadr varlex))
		 varlex)
		(varglob
		 (if (and (eq? (id-use varglob) 'top-level)
			  (not (assoc var quote-constants)))
		     (set! quote-constants
			   (cons (list var varglob) quote-constants)))
		 varglob)
		(else
		 (newv var 'use 'global 'undefref current-define-name)))))

;;; Syntax errors are reported by the following function which will return
;;; (begin #t) as its value.

(define (EXPAND-ERROR form exp)
    (report-error "Illegal" form "syntax:" exp)
    '(begin #t))
