/************************************************************************/
/*                                                                      */
/* Description: Ascii to PostScript printer program.                    */
/* File: imag:/users/local/a2ps/a2ps.c                                  */
/* Created: Mon Nov 28 15:22:15 1988 by miguel@imag (Miguel Santana)    */
/* Version: 2.0                                                         */
/*                                                                      */
/* Edit history:                                                        */
/* 1) Derived of shell program written by evan@csli (Evan Kirshenbaum). */
/*    Written in C for improve speed execution and portability. Many    */
/*    improvements have been added.                                     */
/* Fixes by Oscar Nierstrasz @ cui.uucp:                                */
/* 2) Fixed incorrect handling of stdin (removed error if no file names)*/
/* 3) Added start_page variable to eliminate blank pages printed for    */
/*      files that are exactly multiples of 132 lines (e.g., man pages) */
/* Modified by miguel@imag:                                             */
/* 4) Added new options at installation : sheet format (height/width in */
/*    inches), page format (number of columns per line and of lines per */
/*    page).                                                            */
/* Modified by miguel@imag:                                             */
/* 5) Added new option to print n copies of a same document.            */
/* 6) Cut long filenames if don't fit in the page header.               */
/* Modified by Tim Clark (T.Clark@warwick.ac.uk):                       */
/* 7) Two additional modes of printing (portrait and wide format modes) */
/* 8) Fixed to cope with filenames which contain a character which must */
/*    be escaped in a PostScript string.                                */
/* Modified by miguel@imag.fr to                                        */
/* 9) Add new option to suppress heading printing.                      */
/* 10) Add new option to suppress page surrounding border printing.     */
/* 11) Add new option to change font size. Number of lines and columns  */
/*     are now automatically adjusted, depending on font size and       */
/*     printing mode used.                                              */
/* 12) Minor changes (best layout, usage message, etc).                 */
/* Modified by mai@wolfen.cc.uow.edu.au  May/90                         */
/* 13) Add option -l# where # is the number of lines per page. This is  */
/*     useful if the user wish two specify the number of lines manually */
/*     when he/she prints preformatted files by some other text         */
/*     processors. The font size is scaled up automatically by          */
/*     linesperpage/real_lines. If real_lines < MIN_LINES then scaling  */
/*     may result in too large font -> scaling is suppressed by setting */
/*     real_fontsize = TRUE.                                            */
/* 14) Minor change to option -ffont_size in error message. This also   */
/*     set the flag real_fontsize to TRUE to suppress scaling if -l# is */
/*     used to set the number of lines per page.                        */
/* 15) Change to command line parsing to allow options to be placed any */
/*     where in the command line. Subsequently, a flag no_files is set  */
/*     to TRUE if no files found. This for later printing of files.     */
/* 16) Backspacing did not work before. It is now working properly.     */
/*     for nroff underlining. Bolding sequence of nroff was not         */
/*     supported, now it is supported by default (knows UNIX manual)    */
/*     Option -m is added to automatically set 66 lines per page,       */
/*     suppress numbering and understand Nroff UNIX manual output. Also */
/*     option -nm is added to ignore Nroff BOLD & BACKSPACE sequence.   */
/*                                                                      */
/************************************************************************/

/*
 * Copyright (c) 1988, Miguel Santana, miguel@imag.imag.fr
 *
 * Permission is granted to copy and distribute this file in modified
 * or unmodified form, for noncommercial use, provided (a) this copyright
 * notice is preserved, (b) no attempt is made to restrict redistribution
 * of this file, and (c) this file is not distributed as part of any
 * collection whose redistribution is restricted by a compilation copyright.
*/

#include <stdio.h>
#include <ctype.h>
#ifdef  TURBOC
#include <alloc.h>      /* turbo C needs this */
#endif
#ifdef ANSIC
#include <types.h>
#include <time.h>
#else
#ifdef BSD
#include <sys/time.h>
#else
#ifdef SYSV
#include <sys/types.h>
#include <sys/timeb.h>
#include <time.h>
#else
error !
#endif
#endif
#endif

#ifndef HEADER_PS
#ifdef  ANSIC
#define HEADER_PS       ".\\header.ps"
#else
#define HEADER_PS       "/usr/local/lib/a2ps/header.ps"
#endif
#endif

#ifndef WIDTH
#define WIDTH   8.27
#endif

#ifndef HEIGHT
#define HEIGHT  11.64
#endif

#ifndef MARGIN
#define MARGIN  1.2
#endif

#ifndef DIR_SEP
#define DIR_SEP '\\'
#endif

#define PORTRAIT_HEADER         0.29
#define LANDSCAPE_HEADER        0.22
#define PIXELS_INCH             72
#define MAXFILENAME             20
#define DEFAULT_LINES           66          /* default no. of lines */
#define MAX_LINES               160         /* max. lines per page  */
#define MIN_LINES               40          /* min. lines per page  */
#define FALSE                   0
#define TRUE                    1

#define IS_BOLD         1                   /* nroff bold flag      */
#define IS_BACKSPACE    2                   /* nroff underline flag */

int fold_line();
void print_file();
int printchar();
void skip_page();
int mygetc();                   /* the normal routine for getc()    */
int pggetc();                   /* special for book printing        */
int (* getch)() = mygetc;       /* function pointer to one of them  */

int start_page;
char *program;                  /* this program name */
int no_files = TRUE;            /* any file on command line ? */
int column = 0;                 /* Column number (in current line) */
int line = 0;                   /* Line number (in current page) */
int line_number = 0;            /* Source line number */
int first_page;                 /* First page for a file */
int nonprinting_chars, chars;   /* Number of nonprinting and total chars */
int prefix_width;               /* Width in characters for line prefix */
int PageNo = 0;                 /* Absolute page number for -bk */
int EndPage = FALSE;            /* end of page signal for book printing */
int pages = 0;                  /* Number of logical pages printed */
int sheets = 0;                 /* Number of physical pages printed */
int real_fontsize = FALSE;      /* flag for user-specified font size */
int real_lines = 0;             /* real number of lines per page */
int linesperpage;               /* Lines per page */
int columnsperline;             /* Characters per output line */
int fd;                         /* buffer file descriptor */
char path[128];                 /* buffer filename */

double font_size = 0.0;         /* Size of a char for body font */
int know_manuals = TRUE;        /* understand UNIX manual pages */
int makebook = FALSE;           /* print in a double-sided book style */
int numbering = TRUE;           /* Line numbering option */
int folding = TRUE;             /* Line folding option */
int restart = TRUE;             /* Restart page number at each file option */
int only_printable = FALSE;     /* Replace non printable char by space option */
int interpret = TRUE;           /* Interpret TAB, FF and BS chars option */
int print_binaries = FALSE;     /* Force printing for binary files */
int copies_number = 1;          /* Number of copies to print */
int landscape = TRUE;           /* Otherwise portrait format sheets */
int wide_pages = FALSE;         /* TRUE implies landscape, not twinpage */
int twinpage = TRUE;            /* 2 pages per sheet if true, 1 otherwise */
int no_border = FALSE;          /* TRUE if user don't want the border */
int no_header = FALSE;          /* TRUE if user don't want the header */
int column_width = 8;           /* default column tab width (8) */

main(argc, argv)
int argc;
char *argv[];
{
   register int narg;
   register char *arg;
   double char_width, header_size;
   double page_height, page_width;
   int i;
   extern double atof();

   program = argv[0];
   /* Option processing */
   no_files = TRUE;
   arg = argv[narg = 1];
   while (narg < argc)
   {
      if(arg[0] != '-')                         /* print file */
      {  no_files = FALSE;                      /* skip to next argument */
	 arg = argv[++narg];
	 continue;
      }
      switch (arg[1])
      {
      case '?':                                 /* help */
	 goto usage;
      case '#':                                 /* n copies */
	 copies_number = 0;
	 arg += 2;
	 while (*arg != NULL && isdigit(*arg))
	    copies_number = copies_number*10 + (*arg++ - '0');
	 if (*arg != NULL || copies_number <= 0)
	    goto usage;
	 break;
      case 'b':                                 /* print binary files */
	 if (arg[2] != NULL)
	 {  if(arg[2] != 'k')
		goto usage;
	    else                                /* -bk for book style */
		makebook = TRUE;
	 }
	 else
	    print_binaries = TRUE;
	 break;
      case 'f':                                 /* change font size */
	 if (arg[2] == NULL) {
	    folding = TRUE;
	    break;
	 }
	 if ((font_size = atof(&arg[2])) == 0.0) {
	    fprintf(stderr, "Wrong value `%s' for option -ffont_size\n", &arg[2]);
	    exit(1);
	 }
	 real_fontsize = TRUE;
	 break;
      case 'h':                                 /* help */
	 goto usage;
      case 'i':                                 /* interpret control chars */
	 if (arg[2] != NULL)
	    goto usage;
	 interpret = TRUE;
	 break;
      case 'l':
	 if(arg[2] != NULL)
	 {  if(!isdigit(arg[2]))
		goto usage;
	    real_lines = atoi(&arg[2]);
	    if(real_lines > MAX_LINES)
	    {   fprintf(stderr, "Only %d lines per page allowed\n");
		exit(-1);
	    }
	    else if(real_lines < MIN_LINES)
	    {   fprintf(stderr,"Only %d lines per page, no font scaling!\n",real_lines);
		real_fontsize = TRUE;  /* set flag for no scaling */
	    }
	 }
	 break;
      case 'm':
	 if (arg[2] != NULL)
	    goto usage;
	 real_lines = DEFAULT_LINES;
	 know_manuals = TRUE;
	 numbering = FALSE;
	 break;
      case 'n':                                 /* number file lines */
	 if (arg[2] == NULL)
	 {
	    numbering = TRUE;
	    break;
	 }
	 if (arg[3] != NULL)
	    goto usage;
	 switch (arg[2])
	 {
	 case 'b':                              /* don't print binaries */
	    print_binaries = FALSE;
	    break;
	 case 'f':                              /* cut lines too long */
	    folding = FALSE;
	    break;
	 case 'h':                              /* don't print header */
	    no_header = TRUE;
	    break;
	 case 'i':                              /* don't interpret ctrl chars */
	    interpret = FALSE;
	    break;
	 case 'm':                              /* don't know UNIX manuals */
	    know_manuals = FALSE;
	    break;
	 case 'n':                              /* don't number lines */
	    numbering = FALSE;
	    break;
	 case 'p':                              /* landscape format */
	    landscape = TRUE;
	    break;
	 case 'r':                              /* don't restart sheet number */
	    restart = FALSE;
	    break;
	 case 's':                              /* no surrounding border */
	    no_border = TRUE;
	    break;
	 case 'v':                              /* only printable chars */
	    only_printable = TRUE;
	    break;
	 case 'w':                              /* twin pages */
	    wide_pages = FALSE;
	    break;
	 default:
	    goto usage;
	 }
	 break;
      case 'p':                                 /* portrait format */
	 if (arg[2] != NULL)
	    goto usage;
	 if (wide_pages) {
	    fprintf(stderr, "a2ps: options -p and -w are incompatible\n");
	    exit(1);
	 }
	 landscape = FALSE;
	 break;
      case 'r':                                 /* restart sheet number */
	 if (arg[2] != NULL)
	    goto usage;
	 restart = TRUE;
	 break;
      case 't':                                 /* set tab size */
	 if (arg[2] == NULL || (column_width = atoi(arg+2)) <= 0)
	    goto usage;
	 break;
      case 'v':                                 /* print control chars */
	 if (arg[2] != NULL)
	    goto usage;
	 only_printable = FALSE;
	 break;
      case 'w':                                 /* wide format */
	 if (arg[2] != NULL)
	    goto usage;
	 if (!landscape) {
	    fprintf(stderr, "a2ps: options -p and -w are incompatible\n");
	    exit(1);
	 }
	 wide_pages = TRUE;
	 break;
      default:
      usage:
	 fprintf(stderr,"Usage: %s [options] [f1 f2 ... fn]\n", argv[0]);
	 fprintf(stderr,"options = -#num\t(number of copies to print)\n");
	 fprintf(stderr,"          -?\t(print this information)\n");
	 fprintf(stderr,"          -b\t(print binary files)\n");
	 fprintf(stderr,"          -bk\t(print files in book format)\n");
	 fprintf(stderr,"          -f\t(fold lines too large)\n");
	 fprintf(stderr,"          -fnum\t(font size, num is a float number)\n");
	 fprintf(stderr,"          -h\t(print this information)\n");
	 fprintf(stderr,"          -i\t(interpret tab, bs and ff chars)\n");
	 fprintf(stderr,"          -m\t(know about UNIX manuals)\n");
	 fprintf(stderr,"          -n\t(number line files)\n");
	 fprintf(stderr,"          -lnum\t(lines per page, num is an integer in [40..160])\n");
	 fprintf(stderr,"          -p\t(print in portrait mode)\n");
	 fprintf(stderr,"          -r\t(restart page number after each file)\n");
	 fprintf(stderr,"          -tn\t(set tab size to n)\n");
	 fprintf(stderr,"          -v\t(show non-printing chars in a clear form)\n");
	 fprintf(stderr,"          -w\t(print in wide format)\n");
	 fprintf(stderr,"          -nb\t(don't force printing binary files)\n");
	 fprintf(stderr,"          -nf\t(don't fold lines)\n");
	 fprintf(stderr,"          -nh\t(don't print the header)\n");
	 fprintf(stderr,"          -ni\t(don't interpret special chars)\n");
	 fprintf(stderr,"          -nm\t(don't know about UNIX manual output)\n");
	 fprintf(stderr,"          -nn\t(don't number output lines)\n");
	 fprintf(stderr,"          -np\t(don't print in portrait format)\n");
	 fprintf(stderr,"          -nr\t(don't restart page number)\n");
	 fprintf(stderr,"          -ns\t(don't print surrounding borders)\n");
	 fprintf(stderr,"          -nv\t(replace non-printing chars by space)\n");
	 fprintf(stderr,"          -nw\t(don't print in wide format)\n");
	 exit(1);
      }
      arg = argv[++narg];
   }
   if (arg != NULL && strcmp(arg, "?") == 0)
      goto usage;

   if(makebook)
   {  landscape  = TRUE;
      wide_pages = FALSE;
      folding    = FALSE;
      getch      = pggetc;
   }

   twinpage = landscape && !wide_pages;
   if (font_size == 0.0)
      font_size = landscape ? 6.8 : 9.0;
   page_height = (HEIGHT - MARGIN) * PIXELS_INCH;
   page_width  = (WIDTH - MARGIN) * PIXELS_INCH;

   if(makebook)                 /* reduce 10% for better binding    */
	page_height *=  0.9;
   char_width = 0.6 * font_size;
   if (landscape) {
      header_size = no_header ? 0.0 : LANDSCAPE_HEADER * PIXELS_INCH;
      linesperpage = ((page_width - header_size) / font_size) - 1;
      if(real_lines)
      {  if(real_fontsize == FALSE)
	    font_size *= (float)linesperpage/(float)real_lines;
	 char_width = 0.6 * font_size;
      }
      if (wide_pages)
	 columnsperline = (page_height / char_width) - 1;
      else
	 columnsperline = ((page_height / 2) / char_width) - 1;
   }
   else {
      header_size = no_header ? 0.0 : PORTRAIT_HEADER * PIXELS_INCH;
      linesperpage = ((page_height - header_size) / font_size) - 1;
      if(real_lines)
      {  if(real_fontsize == FALSE)
	    font_size *= (float)linesperpage/(float)real_lines;
	 char_width = 0.6*font_size;
      }
      columnsperline = (page_width / char_width) - 1;
   }
   if(real_lines)
      linesperpage = real_lines;

   if (linesperpage <= 0 || columnsperline <= 0) {
      fprintf(stderr, "Font %g too big !!\n", font_size);
      exit(1);
   }

   /* Header printing (postcript prolog) */
   print_header();

   /* Print files designated or standard input */
   prefix_width = numbering ? 6 : 1;
   if (no_files == TRUE)
   {  if(makebook == FALSE)
	print_file("stdin");
      else
	print_book("stdin");
   }
   else
   {  narg = 0;
      arg  = argv[++narg];
      while (narg < argc)
      {
	 if(arg[0] == '-')
	 {   arg = argv[++narg];
	     continue;
	 }
	 if(arg[0] == '\\' && (arg[1] == '-' || arg[1] == '\\'))
	     arg = ++arg;
	 if (freopen(arg, "r", stdin) == NULL)
	 {
	    fprintf(stderr, "Error opening %s\n", arg);
	    printf("\n%%%%Trailer\ncleanup\ndocsave restore end\n");
	    exit(1);
	 }
	 if(makebook == FALSE)
	    print_file(arg);
	 else
	    print_book(arg);
	 arg = argv[++narg];
      }
   }

   printf("\n%%%%Trailer\ncleanup\ndocsave restore end\n");
}

/*  Print the file in book format */
print_book(name)
char *name;
{
   fd = BuildIndex();       /* build page index table   */
   print_file(name);        /* print the file           */
   close(fd);               /* close and unlink buffer  */
   unlink(path);            /* file                     */
}

/*  Print a file    */
void print_file(name)
char *name;
{
   register int c;
   int  start_line, continue_exit;
   int  char_width;
   int  status;
   char new_name[MAXFILENAME+1];
   char *p;

   /*
    * Boolean to indicates that previous char is \n (or interpreted \f)
    * and a new page would be started, if more text follows
    */
   start_page = FALSE;

   /*
    * Printing binary files is not very useful. We stop printing
    * if we detect one of these files. Our heuristic to detect them:
    * if 50% characters of first page are non-printing characters,
    * the file is a binary file.
    * Option -b force binary files impression.
    */
   first_page = TRUE;
   nonprinting_chars = chars = 0;

   /*
    * Preprocessing (before printing):
    * - TABs expansion (see interpret option)
    * - FF and BS interpretation
    * - replace non printable characters by a space or a char sequence
    *   like:
    *     ^X for ascii codes < 0x20 (X = [@, A, B, ...])
    *     ^? for del char
    *     M-c for ascii codes > 0x3f
    * - prefix parents and backslash ['(', ')', '\'] by backslash
    *   (escape character in postcript)
    */
   column = 0;
   line = line_number = 0;
   start_line = TRUE;

   if (strlen(name) > MAXFILENAME) {
      cut_filename(name, new_name);
      name = new_name;
   }
   putchar('(');
   for (p = name; *p != NULL;)
      printchar(*p++);
   printf(") newfile\n");

   if (restart)
      printf("/sheet 1 def\n");

   pages = 0;
   skip_page();

   c = getch(&status);
   while (c != EOF)
   {
      /* Form feed */
      if (c == '\f' && interpret)
      {
	 /* Close current line */
	 if (!start_line)
	 {
	    printf(") s\n");
	    start_line = TRUE;
	 }
	 /* start a new page ? */
	 if (start_page)
	    skip_page();
	 /* Close current page and begin another */
	 printf("endpage\n") ;
	 start_page = TRUE;
	 /* Verification for binary files */
	 if (first_page && is_binaryfile(name))
	    return;
	 line = 0;
	 if ((c = getch(&status)) == EOF)
	    break;
      }

      /* Start a new line? */
      if (start_line)
      {
	 if (start_page)
	 {       /* only if there is something to print! */
	    skip_page();
	    start_page = FALSE ;
	 }

/**/     printf("ml\n");            /* push currentpoint onto stack */
	 if (numbering)
	    printf("(%-5d ", ++line_number);
	 else
	    printf("( ");
	 start_line = FALSE;
      }

/**/  if(know_manuals == TRUE)
      {                               /* want to understand UNIX manual */
	if(status == IS_BACKSPACE)    /* ie: bolding and underlining    */
	{   printf(") show\n");
	    printf("ml\n");           /* push current point onto stack  */
	    if(c != '(' && c != ')' && c != '\\')
		printf("(%c) show\n", c);
	    else
		printf("(\\%c) show\n", c);
	    printf("moveto\n");       /* pop x,y from stack  to current */
	    putchar('(');             /* point. Then get new character. */
	    c = getch(&status);
	}
	else if(status == IS_BOLD)
	{
	    printf(") show\n");
	    printf("boldfont setfont\n");
	    while(status == IS_BOLD)
	    {   if(c != '(' && c != ')' && c != '\\')
		   printf("(%c) show\n", c);
		else
		   printf("(\\%c) show\n", c);
		c = getch(&status);  /* get next character then try again*/
	    }
	    printf("bodyfont setfont\n");
	    printf("(");
	    continue;
	}
/**/  }

      /* Interpret each character */
      switch (c)
      {
      case '\b':
	 if (!interpret)
	    goto print;
	 if (column)
	    column--;
	 putchar(c);
	 break;
      case '\n':
	 column = 0;
	 start_line = TRUE;
	 printf(") s\n");

/**/     printf("nl\n");        /* pop current position from stack */
/**/                            /* move down by one line           */

	 if (++line >= linesperpage || EndPage == TRUE)
	 {
	    printf("endpage\n");
	    start_page = TRUE ;
	    if (first_page && is_binaryfile(name))
	       return;
	    line = 0;
	    EndPage = FALSE;
	 }
	 break;
      case '\t':
	 if (interpret)
	 {
	    continue_exit = FALSE;
	    do
	    {
	       if (++column + prefix_width > columnsperline)
		  if (folding)
		  {
		     if (fold_line(name) == FALSE)
			return;
		  }
		  else
		  {
		     c = cut_line();
		     continue_exit = TRUE;
		     break;
		  }
	       putchar(' ');
	    } while (column % column_width);
	    if (continue_exit)
	       continue;
	    break;
	}
      default:
      print:
	 if (only_printable)
	    char_width = 1;
	 else
	 {
	    char_width = c > 0177 ? 2 : 0;
	    char_width += c < ' ' || c == 0177 ? 2 : 1;
	 }
	 if (prefix_width + (column += char_width) > columnsperline)
	    if (folding)
	    {
	       if (fold_line(name) == FALSE)
		  return;
	    }
	    else
	    {
	       c = cut_line();
	       continue;
	    }
	 nonprinting_chars += printchar(c);
	 chars++;
	 break;
      }
      c = getch(&status);
   }

   if (!start_line)
      printf(") s\n");
   if (!start_page)
      printf("endpage\n");
}

/*
 * Cut long filenames.
 */
int cut_filename(old_name, new_name)
char *old_name, *new_name;
{
   register char *p;
   register int i;

   p = old_name + (strlen(old_name)-1);
   while (p >= old_name && *p != DIR_SEP) p--;

   for (i = 0, p++; *p != NULL && i < MAXFILENAME; i++)
      *new_name++ = *p++;
   *new_name = NULL;
}

/*
 * Fold a line too long.
 */
int fold_line(name)
char *name;
{
   column = 0;
   printf(") s\n");
   printf("nl\n");              /* move down to next line   */
   printf("ml\n");              /* mark this line           */
   if (++line >= linesperpage)
   {
      printf("endpage\n");
      skip_page();
      if (first_page && is_binaryfile(name))
	 return FALSE;
      line = 0;
   }
   if (numbering)
      printf("(      ");
   else
      printf("( ");

   return TRUE;
}

/*
 * Cut a textline too long to the size of a page line.
 */
cut_line()
{
   int  c;
   int  i;

   while ((c = getch(&i)) != EOF && c != '\n' && c != '\f')
	;
   return c;
}

/*
 * Print a char in a form accept by postscript printers.
 */
int printchar(c)
unsigned char c;
{
   if (c >= ' ' && c < 0177)
   {
      if (c == '(' || c == ')' || c == '\\')
	 putchar('\\');
      putchar(c);
      return 0;
   }

   if (only_printable)
   {
      putchar(' ');
      return 1;
   }

   if (c > 0177)
   {
      printf("M-");
      c &= 0177;
   }
   if (c < ' ')
   {
      putchar('^');
      if ((c = c + '@') == '(' || c == ')' || c == '\\')
	 putchar('\\');
      putchar(c);
   }
   else if (c == 0177)
      printf("^?");
   else
      putchar(c);

   return 1;
}

/*
 * Begins a new physical page.
 */
void skip_page()
{
   if(makebook && PageNo)   /* update the absolute page number  */
   {  printf("gsave\n");
      printf("empty abspnum copy pop\n");
      printf("%d abspnum cvs\n", PageNo);
      printf("grestore\n");
   }
   pages++;
   if (twinpage == FALSE || (pages & 0x1))
   {
      sheets++;
      printf("%%%%Page: %d %d\n", sheets, sheets);
   }
   printf("startpage\n");
}

/*
 * Test if we have a binary file.
 */
is_binaryfile(name)
char *name;
{
   first_page = FALSE;
   if (!print_binaries && (nonprinting_chars*100 / chars) >= 75)
   {
      fprintf(stderr, "%s is a binary file: printing aborted\n", name);
      return TRUE;
   }
   return FALSE;
}

print_header()
{
   register int c;
   FILE *f;
   char *string;
#ifdef ANSIC
   time_t date;
#else
#ifdef BSD
   struct timeval date;
   struct tm *p;
#else
#ifdef SYSV
    struct timeb date;
#endif
#endif
#endif

   if ((f = fopen(HEADER_PS, "r")) == NULL)
   {
      fprintf(stderr, "Poscript header missing\n");
      exit(1);
   }

   /* Initialize some postcript variables */
   printf("%%! a2ps 3.0\n\n");
   printf("/$a2psdict 100 dict def\n");
   printf("$a2psdict begin\n");
   printf("%% Initialize page description variables.\n");
   printf("/inch {72 mul} bind def\n");
   printf("/landscape %s def\n", landscape ? "true" : "false");
   printf("/twinpage %s def\n", twinpage ? "true" : "false");
   printf("/sheetheight %g inch def\n", HEIGHT);
   printf("/sheetwidth %g inch def\n", WIDTH);
   printf("/margin %g inch def\n", MARGIN);
   printf("/makebook %s def\n", makebook ? "true" : "false");
   printf("/noborder %s def\n", no_border ? "true" : "false");
   if (no_header) {
      printf("/noheader true def\n");
      printf("/headersize 0.0 def\n");
   }
   else {
      printf("/noheader false def\n");
      printf("/headersize %g inch def\n",
	     landscape ? LANDSCAPE_HEADER : PORTRAIT_HEADER);
   }
   printf("/bodyfontsize %g def\n", font_size);
   printf("/boldfontsize %g def\n", font_size);
   printf("/lines %d def\n", linesperpage);
   printf("/columns %d def\n", columnsperline);

   /* Retrieve date and hour */
#ifdef ANSIC
   if (time(&date) == -1)
   {
      fprintf(stderr, "Error calculing time\n");
      exit(1);
   }
   string = ctime(&date);

   /* and print them */
   printf("/date (%.6s %.4s %.8s) def\n", string+4, string+20, string+11);
#else
#ifdef BSD
   (void) gettimeofday(&date, (struct timezone *)0);
   p = localtime(&date.tv_sec);
   string = asctime(p);

   /* and print them */
   printf("/date (%.6s %.4s %.8s) def\n", string+4, string+20, string+11);
#else
#ifdef SYSV
   (void)ftime(&date);
   string = ctime(&date.time);
   printf("/date (%.6s %.4s %.8s) def\n", string+4, string+20, string+11);
#endif
#endif
#endif

   /* Header file printing */
   while ((c = getc(f)) != EOF)
      putchar(c);
   /* Close prolog */
   printf("%%%%EndProlog\n\n");

   /* Ask for printing n copies */
   if (copies_number > 1)
      printf("/#copies %d def\n", copies_number);

   /* Go on */
   printf("/docsave save def\n");
   printf("startdoc\n");
}

/*----------------------------------------------------------------------*\
    This a new module added by mai@wolfen.cc.uow.edu.au

    The module is an extension for a2ps version 3.0.
    The extension is for the book style printing. In this style the
    printing is not double-sided but it is set to twinpage, landscape
    mode. The two logical pages are shifted to the right to allow for
    cutting at the middle and punching holes to fit into a folder.
    The order of the pages is as follows :
    Let n is the number of pages
	N = (n / 2)*2   and if n is odd, an odd page is the remainder

	Left        Right
	N/2         1
	N/2 + 1     2
	N/2 + 2     3
	N/2 + 3     4
	...         ...
    In this sense one simply cut the pile at the middle and stack the
    right part on top of the left part and the whole pile is in order
    for binding.

    The two routines mygetc() and pggetc() are written to replace
    getchar(). They all return one character at a time and the status
    if a sequence is encountered ie: nroff Bold or Underline. The pointer
    getch is assigned to one of the two functions before print_file()
    is called.
\*----------------------------------------------------------------------*/
#include<fcntl.h>

#ifdef  ANSIC
#define BUFFER_SIZE     5000
#define MAX_PAGES       1000
#else
#define BUFFER_SIZE    10000
#define MAX_PAGES       4000
#endif

typedef struct                  /* Index node       */
    {   short   page;           /* page number      */
	long    where;          /* file position    */
	short   bytes;          /* bytes per page   */
    }   PageT;

PageT   **PageTable;            /* Indices to pages */
int     start_page;

/*  --------------------------------------------------------------------
    This routine buffer a line of input, release one character at a time
    or a whole sequence of character with some meaning like BOLD or
    BACKSPACE used for underlining by NROFF.
	BOLD sequence is        <c><\b><c><\b><c><\b><c>
	Underline sequence is   <_><\b><c>
    --------------------------------------------------------------------*/
mygetc(status)
int   *status;
{
static  int   i = 0;
static  int   nchars = 0;
static  unsigned char buffer[1024];
int     c;

    *status = 0;
    if(i >= nchars)
    {   if(NULL == gets(buffer))
	    return  EOF;
	nchars = strlen(buffer);
	buffer[nchars]   = '\n';
	buffer[++nchars] = '\0';
	i = 0;
    }
    if(buffer[i] == '\r')
	return buffer[i++];
    if(buffer[i+1] != '\b')     /* this is not a special sequence   */
	return  buffer[i++];
    if(know_manuals == FALSE)   /* know nroff UNIX manuals ?        */
	return  buffer[i++];

    c = buffer[i];              /* check if it is BOLD or BACKSPACE */
    if((buffer[i]   == buffer[i+2]) &&
       (buffer[i]   == buffer[i+4]) &&
       (buffer[i]   == buffer[i+6]) &&
       (buffer[i+1] == buffer[i+1]) &&
       (buffer[i+1] == buffer[i+3]) &&
       (buffer[i+1] == buffer[i+5]) )
    {   *status = IS_BOLD;
	i += 7;
    }
    else
    {   *status = IS_BACKSPACE;
	i += 2;
    }
    return  c;
}

/*----------------------------------------------------------------------*\
    Buffer one pages at a time into memory and release a character at a
    time to the calling routine.
\*----------------------------------------------------------------------*/
pggetc(status)
int   *status;
{
static  int   i = 0;
static  int   n = 0;
static  int   nchars = 0;
static  unsigned char  *buffer = NULL;
int     c;

    *status = 0;
    if(i >= nchars)
    {
	if(PageTable[n] == NULL)
	{   i = 0;
	    n = 0;
	    nchars = 0;
	    return  EOF;
	}
	else
	{   lseek(fd, PageTable[n]->where, 0L);
	    if(PageTable[n]->bytes > BUFFER_SIZE)
	    {   fprintf(stderr, "%s: page is too large\n", program);
		unlink(path);
		exit(-1);
	    }
	    if(buffer == NULL)      /* allocate memory for the first time */
	    {   if((buffer = (unsigned char *)malloc(BUFFER_SIZE)) == NULL)
		{   fprintf(stderr, "%s: sorry, out of memory ?\n", program);
		    Exit(-1);
		}
	    }
				    /* buffer a whole page into memory    */
	    nchars = read(fd, buffer, PageTable[n]->bytes);
	    if(nchars != PageTable[n]->bytes)
	    {   fprintf(stderr, "%s: sorry, i/o error!\n", program);
		Exit(-1);
	    }
	    PageNo = PageTable[n]->page;
	    ++n;
	}
	i = 0;
	line_number = (PageNo-1)*linesperpage;
    }
    if(i+1 == nchars)
	EndPage = TRUE;
    if(buffer[i+1] != '\b')     /* this is not a special sequence   */
	return  buffer[i++];
    if(know_manuals == FALSE)   /* know nroff UNIX manuals ?        */
	return  buffer[i++];

    c = buffer[i];              /* check if it is BOLD or BACKSPACE */
    if((buffer[i]   == buffer[i+2]) &&
       (buffer[i]   == buffer[i+4]) &&
       (buffer[i]   == buffer[i+6]) &&
       (buffer[i+1] == buffer[i+1]) &&
       (buffer[i+1] == buffer[i+3]) &&
       (buffer[i+1] == buffer[i+5]) )
    {   *status = IS_BOLD;
	i += 7;
    }
    else
    {   *status = IS_BACKSPACE;
	i += 2;
    }
    return  c;
}

/*  Allocate an index record to a page in the being printed file    */
PageT
*GetPindex(page, where, bytes)
long    where;
{
PageT   *p;

    if((p = (PageT *)malloc(sizeof(PageT))) == NULL)
    {   fprintf(stderr, "Sorry, out of memory!\n");
	Exit(-1);
    }
    p->page  = page;
    p->where = where;
    p->bytes = bytes;
    return  p;
}


/*  Build a temporary file for the input and a table of indices to every
    page in the file. The memory is allocated from the heap by malloc() */

BuildIndex()
{
int     c, i, j, n, page1, page2, bytes, lines, pages;
long    where, start;
FILE    *outfp;
PageT   **temp_table;
char    *flag = "w";

#ifdef TURBOC
/**/sprintf(path, "a2ps.tmp\0"); /* creating a temporary file */
    flag = "wb";                 /* "wb" is for Turbo C for   */
#else                            /* binary file               */
    sprintf(path, "/tmp/a2ps.%d\0", getuid());
#endif
    EndPage = FALSE;
    if((outfp = fopen(path, flag)) == NULL)
    {    fprintf(stderr, "%s: cannot open temporary file!\n", program);
	exit(-1);
    }

    if(PageTable == NULL)        /* allocate the page table  */
    {   if((PageTable = (PageT **)malloc(MAX_PAGES*sizeof(PageT *))) == NULL)
	{   fprintf(stderr, "%s: sorry, out of memory ?\n", program);
	    Exit(-1);
	}
	memset(PageTable, '\0', MAX_PAGES*sizeof(PageT *));
    }

    i     = 0;
    where = 0;
    lines = 0;
    pages = 0;
    bytes = 0;
    start = 0;
    while((c=getc(stdin)) != EOF)
    {
#ifdef ANSIC
	if(c == '\r')           /* the case of DOS CR-NL end-of-line   */
	{   if((c=getchar()) != '\n')
	    {   ungetc(c, stdin);
		c = '\r';
	    }
	}
#endif
	if(!(c == '\f' && interpret))
	{   putc(c, outfp);         /* put away the character if it */
	    ++where;                /* is a normal character        */
	    ++bytes;
	}

	if(c == '\n')
	    ++lines;
	if(lines == linesperpage || (interpret && c == '\f'))
	{   if(PageTable[pages] != NULL)
		free(PageTable[pages]);
	    PageTable[pages] = GetPindex(pages+1,start, bytes);
	    lines = 0;
	    if(bytes > BUFFER_SIZE)
	    {   fprintf(stderr, "%s: page is too large %d bytes\n", program, bytes);
		Exit(-1);
	    }
	    bytes = 0;
	    start = where;
	    if(++pages == MAX_PAGES)
	    {   fprintf(stderr, "Only %d pages allowed!\n", MAX_PAGES);
		Exit(-1);
	    }

	   if(c == '\f' && interpret)
		continue;
	}
    }
    if(lines)
    {   if(PageTable[pages] != NULL)
	    free(PageTable[pages]);
	PageTable[pages] = GetPindex(pages+1, start, bytes);
	++pages;
    }
    fclose(outfp);

    if((temp_table = (PageT **)malloc(pages*sizeof(PageT *))) == NULL)
    {   fprintf(stderr, "%s: sorry, out of memory ?\n", program);
	Exit(-1);
    }

/**/n = (pages/2)*2;            /* lines page 1 to n/2 on left side */
    j = 0;                      /* lines page n/2 to n on right side*/
    for(i=1; i<=n/2; ++i)
    {   page1 = n/2+i;
	page2 = i;
	temp_table[j++] = PageTable[page1-1];
	temp_table[j++] = PageTable[page2-1];
    }
    if(n < pages)
	temp_table[j] = PageTable[pages-1];
    memcpy(PageTable, temp_table, pages*sizeof(PageT *));
    PageTable[pages] = NULL;
    free(temp_table);

/**/if((fd = open(path, O_RDONLY)) == -1)
    {   fprintf(stderr, "%s: sorry! i/o error\n", program);
	Exit(-1);
    }
    printf("gsave\n");
    printf("empty abspnum copy pop\n");
    printf("%d abspnum cvs\n", (n/2)+1);
    printf("grestore\n");

    return  fd;
}

#ifndef SYSV
memset(s, c, n)
unsigned  char  *s,  c;
{
int     i;
    for(i=0; i < n; ++i)
	s[i] = c;
}
memcpy(s1, s2, n)
unsigned  char  *s1, *s2;
{
int     i;
    for(i=0; i < n; ++i)
	s1[i] = s2[i];
}
#endif

Exit(status)
{
    unlink(path);
    exit(status);
}

/*  End of a2ps.c   */
