#include "OwnedFacility.h"
#include "FifoScheduler.h"
#include "SimulationMultiplexor.h"
#include "Thread.h"
#include "assert.h"
#include <math.h>

OwnedFacility::OwnedFacility(int xservers, ThreadContainer *xscheduler)
    : (xservers, xscheduler)
{
    if (servers == 1) {
	beingServiced.single = NULL;
    } else {
	beingServiced.many = new OwnedFacilityKey[servers];
	for (register int i = 0; i < servers; i++) {
	    beingServiced.many[i] = NULL;
	}
    }
}

//
//	Facility will check that OwnedFacility is de-allocated
//
OwnedFacility::~OwnedFacility()
{
}

void
OwnedFacility::commonReserve( double delayTime, OwnedFacilityKey xkey)
{
    dataLock.reserve();

    pTotalReserves++;
    pTotalDelay += delayTime;

    if (servers == 1) {
	if (whenServiced.single == NullTime) {
	    whenServiced.single = CurrentSimulatedTime;
	    beingServiced.single = xkey;
	    dataLock.release();
	    return;
	}
    } else {
	for (int i = 0; i < servers; i++) {
	    if (whenServiced.many[i] == NullTime) {
		beingServiced.many[i] = xkey;
		whenServiced.many[i] = CurrentSimulatedTime;
		dataLock.release();
		return;
	    }
	}
    }
    
    dataLock.release();	// need to release so reportErrorState works

    reportErrorState(cerr);
    assert2(FALSE, "[OwnedFacility] state error with facility semaphore");
}

void OwnedFacility::reserve()
{
    double startedReserve = CurrentSimulatedTime;
    Semaphore::reserve();
    commonReserve(CurrentSimulatedTime - startedReserve,
		  ThisCpu -> CurrentThread() );
}

bool OwnedFacility::reserveNoBlock()
{
    if (Semaphore::reserveNoBlock()) {
	commonReserve(0.0, ThisCpu -> CurrentThread());
	return(TRUE);
    } else {
	return(FALSE);
    }
}

void OwnedFacility::release()
{
    releaseKeyed(ThisCpu -> CurrentThread());
}

void OwnedFacility::reserveKeyed(OwnedFacilityKey xkey)
{
    double start = CurrentSimulatedTime;
    Semaphore::reserve();
    commonReserve( CurrentSimulatedTime - start, xkey);
}
    
    
void OwnedFacility::releaseKeyed(OwnedFacilityKey xkey)
{
    dataLock.reserve();

    double now = CurrentSimulatedTime;
    bool error;
    
    if (servers == 1) {
	if (whenServiced.single != NullTime) {
	    totalServiceTime += (now - whenServiced.single);
	    whenServiced.single = NullTime;
	    error = (beingServiced.single != xkey);
	} else {
	    error = 1;
	}
    } else {
	for (int i = 0; i < servers; i++) {
	    if (whenServiced.many[i] != NullTime
		&& beingServiced.many[i] == xkey) {
		totalServiceTime += (now - whenServiced.many[i]);
		whenServiced.many[i] = NullTime;
		break;
	    }
	}
	error = (i == servers);
    }
    
    dataLock.release();

    if (error) {
	cerr << " Attempted to release an un-reserved facility with key ";
	cerr << hex(long(xkey)) << "\n";
	reportErrorState(cerr);
	exit(1);
    }

    Semaphore::release();
}

#ifdef UNDEF
void OwnedFacility::classPrintOn(ostream &out)
{
    out << "OwnedFacility with " << activeServers() << " active servers and "
	<< queueLength() << " queued requests";
}
#endif
